<?php
// If wizard not completed or a specific wizard step is requested, show wizard and return
$wizard_completed = get_option('hunnt_ai_wizard_completed', false);
if (!$wizard_completed || isset($_GET['step'])) {
    include_once plugin_dir_path(__FILE__) . 'wizard/wizard-setup.php';
    return;
}
?>

<div class="wrap hunnt-ai-admin-wrap">
    <h1><?php echo esc_html__('Hunnt AI Settings', 'hunnt-ai'); ?></h1>

    <?php if (isset($_GET['hunnt_ai_status'])): ?>
        <?php $notice_class = $_GET['hunnt_ai_status'] === 'success' ? 'notice-success' : 'notice-error'; ?>
        <div class="notice <?php echo esc_attr($notice_class); ?> is-dismissible">
            <p><?php echo esc_html($_GET['hunnt_ai_message'] ?? ''); ?></p>
        </div>
    <?php endif; ?>

    <h2 class="nav-tab-wrapper">
        <a href="#overview" class="nav-tab nav-tab-active"><?php esc_html_e('Overview', 'hunnt-ai'); ?></a>
        <a href="#technical-seo" class="nav-tab"><?php esc_html_e('Technical SEO', 'hunnt-ai'); ?></a>
        <a href="#post-types" class="nav-tab"><?php esc_html_e('Post Types', 'hunnt-ai'); ?></a>
        <a href="#google-settings" class="nav-tab"><?php esc_html_e('Google Settings', 'hunnt-ai'); ?></a>
    </h2>

    <div id="overview" class="tab-content" style="display:block;">
        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
            <input type="hidden" name="action" value="hunnt_ai_save_settings">
            <?php wp_nonce_field('hunnt_ai_save_settings_nonce'); ?>

            <table class="form-table">
                <tr valign="top">
                    <th scope="row">
                        <label for="hunnt_ai_api_key"><?php echo esc_html__('License Key', 'hunnt-ai'); ?></label>
                    </th>
                    <td>
                        <input type="password" id="hunnt_ai_api_key" name="hunnt_ai_api_key" value="<?php echo esc_attr(get_option('hunnt_ai_api_key', '')); ?>" class="regular-text" required>
                        <p class="description"><?php echo esc_html__('Enter your Hunnt AI License key here.', 'hunnt-ai'); ?></p>
                    </td>
                </tr>
            </table>
            <?php submit_button(__('Save License Key', 'hunnt-ai')); ?>
        </form>
    </div>

    <div id="google-settings" class="tab-content" style="display:none;">
        <?php
        // Include Google OAuth settings content
        if (function_exists('hunnt_ai_google_oauth_settings_page')) {
            hunnt_ai_google_oauth_settings_page(['layout' => 'tab']);
        }
        ?>
    </div>

    <div id="technical-seo" class="tab-content" style="display:none;">
        <h2><?php esc_html_e('Technical SEO Status', 'hunnt-ai'); ?></h2>
        <p><?php esc_html_e('We check and apply performance optimizations: Cache, Minify (CSS/JS/HTML), and Gzip.', 'hunnt-ai'); ?></p>
        
        <?php
        $tech_seo_enabled = function_exists('hunnt_ai_is_technical_seo_enabled')
            ? hunnt_ai_is_technical_seo_enabled()
            : true;

        $tech_seo_status = function_exists('hunnt_ai_get_technical_seo_status')
            ? hunnt_ai_get_technical_seo_status()
            : [];

        $tech_seo_diagnostics = function_exists('hunnt_ai_get_technical_seo_diagnostics')
            ? hunnt_ai_get_technical_seo_diagnostics()
            : [];

        $tech_seo_summary = isset($tech_seo_diagnostics['summary']) && is_array($tech_seo_diagnostics['summary'])
            ? $tech_seo_diagnostics['summary']
            : [];

        $tech_seo_last_debug = isset($tech_seo_diagnostics['last_debug']) && is_array($tech_seo_diagnostics['last_debug'])
            ? $tech_seo_diagnostics['last_debug']
            : [];

        $tech_seo_warmup_state = isset($tech_seo_diagnostics['warmup']) && is_array($tech_seo_diagnostics['warmup'])
            ? $tech_seo_diagnostics['warmup']
            : [];

        $tech_seo_log_snapshot = isset($tech_seo_diagnostics['logs']['minify']) && is_array($tech_seo_diagnostics['logs']['minify'])
            ? $tech_seo_diagnostics['logs']['minify']
            : null;

        $static_cache_summary = function_exists('hunnt_ai_get_static_cache_summary')
            ? hunnt_ai_get_static_cache_summary()
            : [];

        $static_cache_enabled = !empty($static_cache_summary['enabled']);
        $static_cache_dependency_met = !empty($static_cache_summary['technical_seo_enabled']);
        $static_cache_has_write_access = isset($static_cache_summary['has_write_access']) ? (bool) $static_cache_summary['has_write_access'] : true;
        $static_cache_last_error = isset($static_cache_summary['last_error']) && is_array($static_cache_summary['last_error'])
            ? $static_cache_summary['last_error']
            : null;
        $static_cache_directories = isset($static_cache_summary['directories']) && is_array($static_cache_summary['directories'])
            ? $static_cache_summary['directories']
            : [];
        $static_cache_actions_enabled = $static_cache_dependency_met && $static_cache_has_write_access;
        $static_cache_inventory = isset($static_cache_summary['inventory']) && is_array($static_cache_summary['inventory'])
            ? $static_cache_summary['inventory']
            : ['count' => 0, 'bytes' => 0];
        $static_cache_last_generated = !empty($static_cache_summary['last_generated'])
            ? (int) $static_cache_summary['last_generated']
            : null;
        $static_cache_ttl = isset($static_cache_summary['ttl']) ? (int) $static_cache_summary['ttl'] : 0;
        $static_cache_ttl_human = $static_cache_ttl > 0
            ? human_time_diff(current_time('timestamp'), current_time('timestamp') + $static_cache_ttl)
            : __('n/a', 'hunnt-ai');

        $static_cache_last_generated_text = $static_cache_last_generated
            ? date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $static_cache_last_generated)
            : __('Never generated', 'hunnt-ai');
        $static_cache_count = isset($static_cache_inventory['count']) ? (int) $static_cache_inventory['count'] : 0;
        $static_cache_bytes = isset($static_cache_inventory['bytes']) ? (float) $static_cache_inventory['bytes'] : 0;

        $tech_seo_last_run_text = !empty($tech_seo_last_debug['timestamp'])
            ? mysql2date(get_option('date_format') . ' ' . get_option('time_format'), $tech_seo_last_debug['timestamp'])
            : __('Never', 'hunnt-ai');

        $tech_seo_last_result = isset($tech_seo_last_debug['success'])
            ? ($tech_seo_last_debug['success'] ? __('Success', 'hunnt-ai') : __('Failed', 'hunnt-ai'))
            : __('Unknown', 'hunnt-ai');

        $tech_seo_failure_reason = $tech_seo_last_debug['failure_reason'] ?? '';
        $tech_seo_total_saved = isset($tech_seo_last_debug['total_bytes_saved']) ? (float) $tech_seo_last_debug['total_bytes_saved'] : 0;
        $tech_seo_total_ratio = isset($tech_seo_last_debug['total_reduction_ratio']) ? (float) $tech_seo_last_debug['total_reduction_ratio'] : 0;

        $tech_seo_next_warmup = !empty($tech_seo_warmup_state['next_timestamp'])
            ? date_i18n(get_option('date_format') . ' ' . get_option('time_format'), (int) $tech_seo_warmup_state['next_timestamp'])
            : __('Not scheduled', 'hunnt-ai');
        $tech_seo_last_warmup = !empty($tech_seo_warmup_state['last_timestamp'])
            ? date_i18n(get_option('date_format') . ' ' . get_option('time_format'), (int) $tech_seo_warmup_state['last_timestamp'])
            : __('Never', 'hunnt-ai');
        $tech_seo_last_warmup_result = !empty($tech_seo_warmup_state['last_result'])
            ? ($tech_seo_warmup_state['last_result'] === 'success' ? __('Success', 'hunnt-ai') : __('Failed', 'hunnt-ai'))
            : __('Unknown', 'hunnt-ai');

        $tech_seo_log_display = '';
        if (!empty($tech_seo_log_snapshot)) {
            $tech_seo_log_display = wp_json_encode($tech_seo_log_snapshot, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        }

        $static_cache_warm_placeholder = implode("\n", array_unique(array_filter([
            home_url('/'),
            home_url('/sample-page/'),
        ])));
        ?>
        
        <!-- Technical SEO Global Status -->
        <div class="notice <?php echo $tech_seo_enabled ? 'notice-success' : 'notice-warning'; ?>" id="tech-seo-global-status">
            <p>
                <strong><?php esc_html_e('Global Status:', 'hunnt-ai'); ?></strong>
                <span id="tech-seo-status-text">
                    <?php if ($tech_seo_enabled): ?>
                        <span class="dashicons dashicons-yes" style="color:#46b450"></span>
                        <?php esc_html_e('Technical SEO is currently ENABLED', 'hunnt-ai'); ?>
                    <?php else: ?>
                        <span class="dashicons dashicons-warning" style="color:#d63638"></span>
                        <?php esc_html_e('Technical SEO is currently DISABLED', 'hunnt-ai'); ?>
                    <?php endif; ?>
                </span>
            </p>
        </div>
        
    <?php
    $missing_modules = function_exists('hunnt_ai_check_modules') ? hunnt_ai_check_modules() : [];
        $all_ok = empty($missing_modules);
        ?>
        <div class="modules-status-summary" style="margin:8px 0;">
            <div class="notice <?php echo $all_ok ? 'notice-success' : 'notice-warning'; ?>">
                <p>
                    <?php if ($all_ok): ?>
                        <span class="dashicons dashicons-yes" style="color:#46b450"></span>
                        <?php esc_html_e('All required PHP extensions are enabled.', 'hunnt-ai'); ?>
                    <?php else: ?>
                        <span class="dashicons dashicons-warning" style="color:#d63638"></span>
                        <?php esc_html_e('Missing PHP extensions:', 'hunnt-ai'); ?>
                        <strong><?php echo esc_html(implode(', ', $missing_modules)); ?></strong>
                    <?php endif; ?>
                </p>
            </div>
        </div>
        <?php
        $features = [
            'cache' => ['name' => __('Cache', 'hunnt-ai'), 'desc' => __('Send cache headers', 'hunnt-ai')],
            'minify_css' => ['name' => __('Minify CSS', 'hunnt-ai'), 'desc' => __('Combine and minify CSS', 'hunnt-ai')],
            'minify_js' => ['name' => __('Minify JS', 'hunnt-ai'), 'desc' => __('Combine and minify JavaScript', 'hunnt-ai')],
            'minify_html' => ['name' => __('Minify HTML', 'hunnt-ai'), 'desc' => __('Compress HTML output', 'hunnt-ai')],
            'gzip' => ['name' => __('Gzip', 'hunnt-ai'), 'desc' => __('Enable gzip compression', 'hunnt-ai')],
        ];
        ?>
        <div id="settings-tech-seo-status" class="modules-grid" style="display:flex;flex-wrap:wrap;gap:12px;">
            <?php foreach ($features as $key => $meta): $ok = !empty($tech_seo_status[$key]); ?>
                <div class="module-item <?php echo $ok ? 'module-ok' : 'module-missing'; ?>" data-feature="<?php echo esc_attr($key); ?>" style="border:1px solid #ddd;padding:12px;width:280px;background:#fff;">
                    <div class="module-header" style="display:flex;align-items:center;gap:8px;">
                        <span class="module-icon"><?php echo $ok ? '<i class="dashicons dashicons-yes" style="color:#46b450"></i>' : '<i class="dashicons dashicons-no" style="color:#d63638"></i>'; ?></span>
                        <h4 style="margin:0;"><?php echo esc_html($meta['name']); ?></h4>
                    </div>
                    <p class="module-description" style="margin:6px 0 8px; color:#555;"><?php echo esc_html($meta['desc']); ?></p>
                    <span class="module-status <?php echo $ok ? 'status-ok' : 'status-missing'; ?>"><?php echo $ok ? esc_html__('Enabled', 'hunnt-ai') : esc_html__('Not Enabled', 'hunnt-ai'); ?></span>
                </div>
            <?php endforeach; ?>
        </div>

        <div class="hunnt-tech-seo-secondary" style="margin-top:15px;display:flex;flex-wrap:wrap;gap:12px;">
            <div class="card" style="border:1px solid #ddd;padding:12px;flex:2 1 340px;background:#fff;min-width:320px;">
                <h4 style="margin:0 0 8px;"><?php esc_html_e('Optimization Summary', 'hunnt-ai'); ?></h4>
                <ul style="list-style:none;margin:0;padding:0;display:grid;grid-template-columns:repeat(auto-fit,minmax(200px,1fr));gap:6px;">
                    <li><strong><?php esc_html_e('Last Result', 'hunnt-ai'); ?>:</strong> <span id="tech-seo-last-result"><?php echo esc_html($tech_seo_last_result); ?></span></li>
                    <li><strong><?php esc_html_e('Last Run', 'hunnt-ai'); ?>:</strong> <span id="tech-seo-last-run"><?php echo esc_html($tech_seo_last_run_text); ?></span></li>
                    <li><strong><?php esc_html_e('Bytes Saved', 'hunnt-ai'); ?>:</strong> <span id="tech-seo-total-saved"><?php echo esc_html(size_format($tech_seo_total_saved)); ?></span></li>
                    <li><strong><?php esc_html_e('Reduction Ratio', 'hunnt-ai'); ?>:</strong> <span id="tech-seo-total-ratio"><?php echo esc_html(number_format_i18n($tech_seo_total_ratio * 100, 2)); ?>%</span></li>
                    <li><strong><?php esc_html_e('Warmup Status', 'hunnt-ai'); ?>:</strong> <span id="tech-seo-last-warmup"><?php echo esc_html($tech_seo_last_warmup_result); ?></span></li>
                    <li><strong><?php esc_html_e('Last Warmup', 'hunnt-ai'); ?>:</strong> <span id="tech-seo-last-warmup-at"><?php echo esc_html($tech_seo_last_warmup); ?></span></li>
                    <li><strong><?php esc_html_e('Next Warmup', 'hunnt-ai'); ?>:</strong> <span id="tech-seo-next-warmup"><?php echo esc_html($tech_seo_next_warmup); ?></span></li>
                    <li id="tech-seo-last-failure" style="grid-column:1 / -1;color:#d63638;<?php echo empty($tech_seo_failure_reason) ? 'display:none;' : ''; ?>">
                        <strong><?php esc_html_e('Last Failure Reason', 'hunnt-ai'); ?>:</strong>
                        <span><?php echo esc_html($tech_seo_failure_reason); ?></span>
                    </li>
                </ul>
                <div style="margin-top:10px;display:flex;flex-wrap:wrap;gap:6px;">
                    <button type="button" class="button button-primary" id="tech-seo-rebuild-assets" <?php disabled(!$tech_seo_enabled); ?>><?php esc_html_e('Rebuild Optimized Assets', 'hunnt-ai'); ?></button>
                    <button type="button" class="button" id="tech-seo-warmup-now" <?php disabled(!$tech_seo_enabled); ?>><?php esc_html_e('Warm Minified Assets', 'hunnt-ai'); ?></button>
                </div>
            </div>

            <div class="card" style="border:1px solid #ddd;padding:12px;flex:1 1 280px;background:#fff;min-width:280px;max-width:340px;">
                <h4 style="margin:0 0 8px;"><?php esc_html_e('Static Cache', 'hunnt-ai'); ?></h4>
                <ul style="list-style:none;margin:0;padding:0;display:flex;flex-direction:column;gap:4px;">
                    <li><strong><?php esc_html_e('Status', 'hunnt-ai'); ?>:</strong>
                        <span id="hunnt-static-cache-status"
                              data-enabled="<?php echo $static_cache_enabled ? '1' : '0'; ?>"
                              data-tech-enabled="<?php echo $static_cache_dependency_met ? '1' : '0'; ?>"
                              data-write-access="<?php echo $static_cache_has_write_access ? '1' : '0'; ?>"
                              class="hunnt-status-label <?php echo $static_cache_dependency_met ? ($static_cache_enabled ? 'hunnt-status-success' : 'hunnt-status-neutral') : 'hunnt-status-warning'; ?>">
                            <?php
                            if (!$static_cache_dependency_met) {
                                esc_html_e('Enable Technical SEO to use static cache.', 'hunnt-ai');
                            } elseif (!$static_cache_has_write_access) {
                                esc_html_e('Filesystem permissions prevent static cache from running.', 'hunnt-ai');
                            } elseif ($static_cache_enabled) {
                                esc_html_e('Enabled', 'hunnt-ai');
                            } else {
                                esc_html_e('Disabled', 'hunnt-ai');
                            }
                            ?>
                        </span>
                    </li>
                    <li><strong><?php esc_html_e('WP_CACHE', 'hunnt-ai'); ?>:</strong> <span id="hunnt-static-cache-wp"><?php echo !empty($static_cache_summary['wp_cache']) ? esc_html__('On', 'hunnt-ai') : esc_html__('Off', 'hunnt-ai'); ?></span></li>
                    <li><strong><?php esc_html_e('Drop-in Active', 'hunnt-ai'); ?>:</strong> <span id="hunnt-static-cache-dropin"><?php echo !empty($static_cache_summary['dropin_active']) ? esc_html__('Yes', 'hunnt-ai') : esc_html__('No', 'hunnt-ai'); ?></span></li>
                    <li><strong><?php esc_html_e('Cached Files', 'hunnt-ai'); ?>:</strong> <span id="hunnt-static-cache-count"><?php echo esc_html(number_format_i18n($static_cache_count)); ?></span></li>
                    <li><strong><?php esc_html_e('Cache Size', 'hunnt-ai'); ?>:</strong> <span id="hunnt-static-cache-bytes"><?php echo esc_html(size_format($static_cache_bytes)); ?></span></li>
                    <li><strong><?php esc_html_e('TTL', 'hunnt-ai'); ?>:</strong> <span id="hunnt-static-cache-ttl"><?php echo esc_html($static_cache_ttl_human); ?></span></li>
                    <li><strong><?php esc_html_e('Last Generated', 'hunnt-ai'); ?>:</strong> <span id="hunnt-static-cache-generated"><?php echo esc_html($static_cache_last_generated_text); ?></span></li>
                </ul>
                <label for="hunnt-static-cache-warm-urls" style="display:block;margin-top:10px;">
                    <strong><?php esc_html_e('Warm specific URLs (optional)', 'hunnt-ai'); ?></strong>
                    <textarea id="hunnt-static-cache-warm-urls" rows="3" style="width:100%;margin-top:4px;" placeholder="<?php echo esc_attr($static_cache_warm_placeholder); ?>"></textarea>
                </label>
                <div style="margin-top:8px;display:flex;flex-direction:column;gap:6px;">
                    <button type="button" class="button button-secondary" id="hunnt-static-cache-warm" <?php disabled(!$static_cache_actions_enabled); ?>><?php esc_html_e('Warm Cache', 'hunnt-ai'); ?></button>
                    <button type="button" class="button" id="hunnt-static-cache-purge" <?php disabled(!$static_cache_actions_enabled); ?>><?php esc_html_e('Purge Cache', 'hunnt-ai'); ?></button>
                    <button type="button" class="button button-primary" id="hunnt-static-cache-toggle" data-enabled="<?php echo $static_cache_enabled ? '1' : '0'; ?>" data-tech-enabled="<?php echo $static_cache_dependency_met ? '1' : '0'; ?>" data-write-access="<?php echo $static_cache_has_write_access ? '1' : '0'; ?>" <?php disabled(!$static_cache_dependency_met); ?>>
                        <?php
                        if (!$static_cache_dependency_met) {
                            esc_html_e('Enable Technical SEO to use Static Cache', 'hunnt-ai');
                        } elseif (!$static_cache_has_write_access) {
                            esc_html_e('Fix permissions to enable Static Cache', 'hunnt-ai');
                        } else {
                            echo $static_cache_enabled
                                ? esc_html__('Disable Static Cache', 'hunnt-ai')
                                : esc_html__('Enable Static Cache', 'hunnt-ai');
                        }
                        ?>
                    </button>
                </div>
                <div id="hunnt-static-cache-permission" class="notice notice-error" style="<?php echo $static_cache_has_write_access ? 'display:none;' : 'margin-top:8px;'; ?>">
                    <p id="hunnt-static-cache-permission-message" style="margin:8px 0 4px;">
                        <?php
                        if ($static_cache_last_error && !empty($static_cache_last_error['message'])) {
                            echo esc_html($static_cache_last_error['message']);
                        } else {
                            esc_html_e('Hunnt AI cannot create its static cache folders. Please update the filesystem permissions for the paths below.', 'hunnt-ai');
                        }
                        ?>
                    </p>
                    <ul id="hunnt-static-cache-permission-paths" style="margin:0;padding-left:18px;">
                        <?php foreach ($static_cache_directories as $key => $dir_meta):
                            $is_writable = !empty($dir_meta['writable']);
                            $status_text = $is_writable ? esc_html__('Writable', 'hunnt-ai') : esc_html__('Not writable', 'hunnt-ai');
                            ?>
                            <li data-key="<?php echo esc_attr($key); ?>">
                                <code><?php echo esc_html($dir_meta['path']); ?></code>
                                <span class="hunnt-static-cache-permission-state <?php echo $is_writable ? 'state-ok' : 'state-bad'; ?>">— <?php echo $status_text; ?></span>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            </div>
        </div>

        <div class="notice notice-info" id="tech-seo-diagnostics-feedback" style="display:none;margin-top:12px;">
            <p></p>
        </div>

        <div class="tech-seo-log-container" style="margin-top:12px;">
            <button type="button" class="button" id="tech-seo-log-toggle" data-open="0"><?php esc_html_e('Show Optimization Log', 'hunnt-ai'); ?></button>
            <pre id="tech-seo-log-viewer" style="display:none;background:#1e1e1e;color:#fff;padding:12px;margin-top:8px;max-height:320px;overflow:auto;white-space:pre-wrap;"><?php echo $tech_seo_log_display ? esc_html($tech_seo_log_display) : esc_html__('No optimization logs recorded yet.', 'hunnt-ai'); ?></pre>
        </div>

        <p style="margin-top:10px;">
            <button type="button" class="button button-primary" id="settings-apply-tech-seo" <?php echo !$tech_seo_enabled ? 'style="display:none;"' : ''; ?>>
                <?php esc_html_e('Fix Now', 'hunnt-ai'); ?>
            </button>
            <button type="button" class="button" id="settings-refresh-tech-seo">
                <?php esc_html_e('Refresh', 'hunnt-ai'); ?>
            </button>
            <button type="button" class="button button-warning" id="settings-disable-tech-seo" <?php echo !$tech_seo_enabled ? 'style="display:none;"' : ''; ?>>
                <span class="dashicons dashicons-dismiss" style="margin-top:3px;"></span>
                <?php esc_html_e('Disable All', 'hunnt-ai'); ?>
            </button>
            <button type="button" class="button button-primary" id="settings-enable-tech-seo" <?php echo $tech_seo_enabled ? 'style="display:none;"' : ''; ?>>
                <span class="dashicons dashicons-yes" style="margin-top:3px;"></span>
                <?php esc_html_e('Enable Technical SEO', 'hunnt-ai'); ?>
            </button>
        </p>
    </div>

    <div id="post-types" class="tab-content" style="display:none;">
        <h2><?php esc_html_e('Post Types - Create New', 'hunnt-ai'); ?></h2>
        <p><?php esc_html_e('Choose which post types to auto-create new AI articles for.', 'hunnt-ai'); ?></p>
        <?php
        $saved_plan = get_option('hunnt_ai_new_posts_plan', []);
        $exclude = ['attachment','revision','nav_menu_item','wp_template','wp_template_part'];
        $post_types = get_post_types(['public' => true], 'objects');
        foreach ($exclude as $ex) { unset($post_types[$ex]); }
        ?>
        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" id="hunnt-plan-form">
            <input type="hidden" name="action" value="hunnt_ai_save_post_types_plan">
            <?php wp_nonce_field('hunnt_ai_save_post_types_plan', 'hunnt_ai_plan_nonce'); ?>
            <div style="display:flex;flex-wrap:wrap;gap:12px;">
                <?php foreach ($post_types as $type_key => $obj):
                    $row = isset($saved_plan[$type_key]) ? $saved_plan[$type_key] : [];
                    $enabled = !empty($row['enabled']);
                    $count = isset($row['count']) ? intval($row['count']) : 0;
                ?>
                <div class="card" style="border:1px solid #ddd;padding:12px;width:320px;">
                    <h4 style="margin:0 0 8px;"><?php echo esc_html($obj->labels->name); ?></h4>
                    <label>
                        <input type="checkbox" name="plan_enabled[<?php echo esc_attr($type_key); ?>]" class="plan-enable" data-type="<?php echo esc_attr($type_key); ?>" <?php checked($enabled); ?>>
                        <?php esc_html_e('Create new posts', 'hunnt-ai'); ?>
                    </label>
                </div>
                <?php endforeach; ?>
            </div>
            <p style="margin-top:10px;">
                <button type="submit" class="button button-primary" id="plan-save-btn"><?php esc_html_e('Save Plan', 'hunnt-ai'); ?></button>
                <button type="button" class="button button-warning" id="plan-get-btn" style="display:none;"><span class="dashicons dashicons-star-filled"></span> <?php esc_html_e('Get a Plan', 'hunnt-ai'); ?></button>
            </p>
        </form>
        <p><small><?php esc_html_e('Note: To improve existing articles, use the Improve Posts tab.', 'hunnt-ai'); ?></small></p>
    </div>

    <div id="improve-posts" class="tab-content" style="display:none;">
        <h2><?php esc_html_e('Improve Posts', 'hunnt-ai'); ?></h2>
        <p><?php esc_html_e('Select articles to improve. Use filters to narrow results.', 'hunnt-ai'); ?></p>
        <?php
        $selected = get_option('hunnt_ai_selected_articles', []);
        $all_types = get_post_types(['public'=>true], 'objects');
        unset($all_types['attachment']);
        $filter_type = isset($_GET['hunnt_type']) ? sanitize_text_field($_GET['hunnt_type']) : 'post';
        $filter_s = isset($_GET['hunnt_s']) ? sanitize_text_field($_GET['hunnt_s']) : '';
        $filter_status = isset($_GET['hunnt_status']) ? sanitize_text_field($_GET['hunnt_status']) : '';
        $paged = isset($_GET['hunnt_paged']) ? max(1, intval($_GET['hunnt_paged'])) : 1;
        $per_page = 10;
        $args = [
            'post_type' => array_key_exists($filter_type, $all_types) ? $filter_type : 'post',
            's' => $filter_s,
            'posts_per_page' => $per_page,
            'paged' => $paged,
            'orderby' => 'date',
            'order' => 'DESC',
        ];
        if (in_array($filter_status, ['publish','draft','private','pending'], true)) {
            $args['post_status'] = $filter_status;
        } else {
            $args['post_status'] = ['publish','draft','private'];
        }
        $q = new WP_Query($args);
        ?>
        <form method="get" action="" style="margin-bottom:8px;">
            <input type="hidden" name="page" value="hunnt-ai">
            <label><?php esc_html_e('Type', 'hunnt-ai'); ?>
                <select name="hunnt_type">
                    <?php foreach ($all_types as $key=>$obj): ?>
                        <option value="<?php echo esc_attr($key); ?>" <?php selected($filter_type, $key); ?>><?php echo esc_html($obj->labels->name); ?></option>
                    <?php endforeach; ?>
                </select>
            </label>
            <label style="margin-left:8px;">
                <?php esc_html_e('Status', 'hunnt-ai'); ?>
                <select name="hunnt_status">
                    <option value=""><?php esc_html_e('Any', 'hunnt-ai'); ?></option>
                    <?php foreach (['publish','draft','private','pending'] as $st): ?>
                        <option value="<?php echo esc_attr($st); ?>" <?php selected($filter_status, $st); ?>><?php echo esc_html(ucfirst($st)); ?></option>
                    <?php endforeach; ?>
                </select>
            </label>
            <label style="margin-left:8px;">
                <?php esc_html_e('Search', 'hunnt-ai'); ?>
                <input type="search" name="hunnt_s" value="<?php echo esc_attr($filter_s); ?>" />
            </label>
            <button class="button" style="margin-left:8px;"><?php esc_html_e('Filter', 'hunnt-ai'); ?></button>
        </form>

        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" id="hunnt-improve-form">
            <input type="hidden" name="action" value="hunnt_ai_save_improve_selection">
            <?php wp_nonce_field('hunnt_ai_save_improve_selection', 'hunnt_ai_improve_nonce'); ?>
            <table class="widefat fixed striped">
                <thead>
                    <tr>
                        <th><?php esc_html_e('Select', 'hunnt-ai'); ?></th>
                        <th><?php esc_html_e('Title', 'hunnt-ai'); ?></th>
                        <th><?php esc_html_e('Status', 'hunnt-ai'); ?></th>
                        <th><?php esc_html_e('Date', 'hunnt-ai'); ?></th>
                    </tr>
                </thead>
                <tbody>
                <?php if ($q->have_posts()): while ($q->have_posts()): $q->the_post(); $p = get_post(); ?>
                    <tr>
                        <td><input type="checkbox" name="selected_posts[]" value="<?php echo esc_attr($p->ID); ?>" <?php checked(in_array($p->ID, $selected, true)); ?>></td>
                        <td><?php echo esc_html(get_the_title($p) ?: __('(No title)', 'hunnt-ai')); ?></td>
                        <td><?php echo esc_html(ucfirst($p->post_status)); ?></td>
                        <td><?php echo esc_html(get_the_date('', $p)); ?></td>
                    </tr>
                <?php endwhile; wp_reset_postdata(); else: ?>
                    <tr><td colspan="4"><?php esc_html_e('No posts found.', 'hunnt-ai'); ?></td></tr>
                <?php endif; ?>
                </tbody>
            </table>
            <?php if ($q->max_num_pages > 1): ?>
                <div class="tablenav">
                    <div class="tablenav-pages">
                        <?php
                        $base_url = remove_query_arg('hunnt_paged');
                        $prev = $paged > 1 ? add_query_arg('hunnt_paged', $paged-1, $base_url) : '';
                        $next = $paged < $q->max_num_pages ? add_query_arg('hunnt_paged', $paged+1, $base_url) : '';
                        ?>
                        <span class="displaying-num"><?php echo intval($q->found_posts); ?> <?php esc_html_e('items', 'hunnt-ai'); ?></span>
                        <span class="pagination-links">
                            <a class="prev-page button<?php echo $prev ? '' : ' disabled'; ?>" href="<?php echo esc_url($prev ?: '#'); ?>">&lsaquo;</a>
                            <span class="paging-input"><?php echo $paged; ?> / <span class="total-pages"><?php echo intval($q->max_num_pages); ?></span></span>
                            <a class="next-page button<?php echo $next ? '' : ' disabled'; ?>" href="<?php echo esc_url($next ?: '#'); ?>">&rsaquo;</a>
                        </span>
                    </div>
                </div>
            <?php endif; ?>
            <p style="margin-top:10px;">
                <button type="submit" class="button button-primary" id="improve-save-btn"><?php esc_html_e('Save Selection', 'hunnt-ai'); ?></button>
                <button type="button" class="button button-warning" id="improve-get-btn" style="display:none;"><span class="dashicons dashicons-star-filled"></span> <?php esc_html_e('Get a Plan', 'hunnt-ai'); ?></button>
            </p>
        </form>
    </div>
</div>

<script>
(function(){
    // Tabs
    const tabs = document.querySelectorAll('.nav-tab-wrapper .nav-tab');
    const panes = document.querySelectorAll('.tab-content');
    tabs.forEach(tab => tab.addEventListener('click', (e) => {
        e.preventDefault();
        tabs.forEach(t => t.classList.remove('nav-tab-active'));
        panes.forEach(p => p.style.display = 'none');
        tab.classList.add('nav-tab-active');
        const target = document.querySelector(tab.getAttribute('href'));
        if (target) target.style.display = 'block';
    }));
})();

// Technical SEO actions
jQuery(function($){
    var nonce = '<?php echo wp_create_nonce('hunnt_wizard_nonce'); ?>';
    var pricingUrl = '<?php echo esc_url(hunnt_ai_get_pricing_url()); ?>';
    var subscriptionStatusCache = null;
    var subscriptionFetchPromise = null;
    
    /**
     * Update global status indicator
     */
    function updateGlobalStatus(enabled) {
        var $statusBox = $('#tech-seo-global-status');
        var $statusText = $('#tech-seo-status-text');
        
        if (enabled) {
            $statusBox.removeClass('notice-warning').addClass('notice-success');
            $statusText.html('<span class="dashicons dashicons-yes" style="color:#46b450"></span> <?php echo esc_js(__('Technical SEO is currently ENABLED', 'hunnt-ai')); ?>');
            $('#settings-apply-tech-seo, #settings-disable-tech-seo').show();
            $('#settings-enable-tech-seo').hide();
            $('#tech-seo-rebuild-assets, #tech-seo-warmup-now').prop('disabled', false);
        } else {
            $statusBox.removeClass('notice-success').addClass('notice-warning');
            $statusText.html('<span class="dashicons dashicons-warning" style="color:#d63638"></span> <?php echo esc_js(__('Technical SEO is currently DISABLED', 'hunnt-ai')); ?>');
            $('#settings-apply-tech-seo, #settings-disable-tech-seo').hide();
            $('#settings-enable-tech-seo').show();
            $('#tech-seo-rebuild-assets, #tech-seo-warmup-now').prop('disabled', true);
        }
    }
    
    /**
     * Render technical SEO status for each feature
     */
    function renderStatus(status){
        $('#settings-tech-seo-status .module-item').each(function(){
            var key = $(this).data('feature');
            var ok = !!status[key];
            var $statusLabel = $(this).find('.module-status');

            $(this).toggleClass('module-ok', ok).toggleClass('module-missing', !ok);
            $(this).find('.module-icon').html(ok ? '<span class="dashicons dashicons-yes" style="color:#46b450"></span>' : '<span class="dashicons dashicons-no" style="color:#d63638"></span>');

            $statusLabel.toggleClass('status-ok', ok).toggleClass('status-missing', !ok).text(ok ? '<?php echo esc_js(__('Enabled', 'hunnt-ai')); ?>' : '<?php echo esc_js(__('Not Enabled', 'hunnt-ai')); ?>');
        });
    }
    
    /**
     * Fetch current technical SEO status
     */
    function fetchStatus(cb){
        $.post(ajaxurl, { action: 'hunnt_ai_get_technical_seo_status', nonce: nonce }, function(resp){
            if (resp && resp.success && resp.data) { 
                renderStatus(resp.data.status||{}); 
                if(cb) cb(resp.data.status||{}); 
            }
        });
    }
    
    /**
     * Apply technical SEO fixes
     */
    function applyFixes(missing){
        $('#settings-apply-tech-seo').prop('disabled', true).text('<?php echo esc_js(__('Applying...', 'hunnt-ai')); ?>');
        $.post(ajaxurl, { action: 'hunnt_ai_apply_technical_seo', nonce: nonce, features: missing }, function(resp){
            if (resp && resp.success){ 
                renderStatus(resp.data.status||{}); 
                $('#settings-apply-tech-seo').text('<?php echo esc_js(__('Applied', 'hunnt-ai')); ?>'); 
                setTimeout(function(){ 
                    $('#settings-apply-tech-seo').prop('disabled', false).text('<?php echo esc_js(__('Fix Now', 'hunnt-ai')); ?>'); 
                }, 1200);
            } else { 
                alert((resp&&resp.data)||'<?php echo esc_js(__('Could not apply fixes. Ensure your subscription is active.', 'hunnt-ai')); ?>'); 
                $('#settings-apply-tech-seo').prop('disabled', false).text('<?php echo esc_js(__('Fix Now', 'hunnt-ai')); ?>'); 
            }
        }).fail(function(){ 
            alert('<?php echo esc_js(__('Network error while applying fixes', 'hunnt-ai')); ?>'); 
            $('#settings-apply-tech-seo').prop('disabled', false).text('<?php echo esc_js(__('Fix Now', 'hunnt-ai')); ?>'); 
        });
    }
    
    /**
     * Disable all technical SEO features
     */
    function disableTechnicalSEO(){
        if (!confirm('<?php echo esc_js(__('Are you sure you want to disable all Technical SEO optimizations? This will turn off Cache, Minification, and Gzip compression.', 'hunnt-ai')); ?>')) {
            return;
        }
        
        $('#settings-disable-tech-seo').prop('disabled', true).html('<span class="dashicons dashicons-update spin"></span> <?php echo esc_js(__('Disabling...', 'hunnt-ai')); ?>');
        
        $.post(ajaxurl, { action: 'hunnt_ai_disable_technical_seo', nonce: nonce }, function(resp){
            if (resp && resp.success) {
                renderStatus(resp.data.status || {});
                updateGlobalStatus(false);
                if (resp.data && resp.data.static_cache_summary && window.hunntAiUpdateStaticCache) {
                    window.hunntAiUpdateStaticCache(resp.data.static_cache_summary);
                }
                alert(resp.data.message || '<?php echo esc_js(__('Technical SEO disabled successfully', 'hunnt-ai')); ?>');
            } else {
                alert((resp && resp.data) || '<?php echo esc_js(__('Failed to disable Technical SEO', 'hunnt-ai')); ?>');
            }
            $('#settings-disable-tech-seo').prop('disabled', false).html('<span class="dashicons dashicons-dismiss" style="margin-top:3px;"></span> <?php echo esc_js(__('Disable All', 'hunnt-ai')); ?>');
        }).fail(function(){
            alert('<?php echo esc_js(__('Network error while disabling Technical SEO', 'hunnt-ai')); ?>');
            $('#settings-disable-tech-seo').prop('disabled', false).html('<span class="dashicons dashicons-dismiss" style="margin-top:3px;"></span> <?php echo esc_js(__('Disable All', 'hunnt-ai')); ?>');
        });
    }
    
    /**
     * Enable technical SEO globally
     */
    function enableTechnicalSEO(){
        $('#settings-enable-tech-seo').prop('disabled', true).html('<span class="dashicons dashicons-update spin"></span> <?php echo esc_js(__('Enabling...', 'hunnt-ai')); ?>');
        
        $.post(ajaxurl, { action: 'hunnt_ai_enable_technical_seo', nonce: nonce }, function(resp){
            if (resp && resp.success) {
                updateGlobalStatus(true);
                renderStatus(resp.data.status || {});
                if (resp.data && resp.data.static_cache_summary && window.hunntAiUpdateStaticCache) {
                    window.hunntAiUpdateStaticCache(resp.data.static_cache_summary);
                }
                alert(resp.data.message || '<?php echo esc_js(__('Technical SEO enabled successfully', 'hunnt-ai')); ?>');
            } else {
                alert((resp && resp.data) || '<?php echo esc_js(__('Failed to enable Technical SEO', 'hunnt-ai')); ?>');
            }
            $('#settings-enable-tech-seo').prop('disabled', false).html('<span class="dashicons dashicons-yes" style="margin-top:3px;"></span> <?php echo esc_js(__('Enable Technical SEO', 'hunnt-ai')); ?>');
        }).fail(function(){
            alert('<?php echo esc_js(__('Network error while enabling Technical SEO', 'hunnt-ai')); ?>');
            $('#settings-enable-tech-seo').prop('disabled', false).html('<span class="dashicons dashicons-yes" style="margin-top:3px;"></span> <?php echo esc_js(__('Enable Technical SEO', 'hunnt-ai')); ?>');
        });
    }
    
    /**
     * Check subscription status
     */
    function fetchSubscriptionStatus() {
        return new Promise(function(resolve) {
            $.post(ajaxurl, { action: 'hunnt_ai_check_subscription_status', nonce: nonce }, function(resp){
                resolve(!!(resp && resp.success && resp.data && resp.data.active));
            }).fail(function(){
                resolve(false);
            });
        });
    }

    function getSubscriptionStatus(forceRefresh) {
        if (!forceRefresh && subscriptionStatusCache !== null) {
            return Promise.resolve(subscriptionStatusCache);
        }

        if (!forceRefresh && subscriptionFetchPromise) {
            return subscriptionFetchPromise;
        }

        var request = fetchSubscriptionStatus().then(function(active){
            subscriptionStatusCache = active;
            return active;
        });

        request.then(function(){
            subscriptionFetchPromise = null;
        });

        if (!forceRefresh) {
            subscriptionFetchPromise = request;
        }

        return request;
    }

    function checkSub(cb, opts){
        if (typeof cb !== 'function') {
            return;
        }

        var options = opts || {};

        getSubscriptionStatus(!!options.force).then(function(active){
            cb(!!active);
        });
    }
    
    // Event Handlers
    $('#settings-refresh-tech-seo').on('click', function(e){ 
        e.preventDefault(); 
        fetchStatus(); 
    });
    
    $('#settings-apply-tech-seo').on('click', function(e){ 
        e.preventDefault(); 
        checkSub(function(active){ 
            if(!active){ 
                window.open(pricingUrl,'_blank'); 
                return; 
            } 
            fetchStatus(function(st){ 
                var missing=[]; 
                ['cache','minify_css','minify_js','minify_html','gzip'].forEach(function(k){ 
                    if(!st[k]) missing.push(k);
                }); 
                if(missing.length){ 
                    applyFixes(missing);
                } 
            }); 
        }, { force: true }); 
    });
    
    $('#settings-disable-tech-seo').on('click', function(e){
        e.preventDefault();
        disableTechnicalSEO();
    });
    
    $('#settings-enable-tech-seo').on('click', function(e){
        e.preventDefault();
        enableTechnicalSEO();
    });

    // Initial gating for Technical SEO button label
    checkSub(function(active){
        if (!active) {
            $('#settings-apply-tech-seo').removeClass('button-primary').addClass('button-warning').html('<span class="dashicons dashicons-star-filled"></span> <?php echo esc_js(__('Fix Now (Premium)', 'hunnt-ai')); ?>');
        } else {
            $('#settings-apply-tech-seo').removeClass('button-warning').addClass('button-primary').text('<?php echo esc_js(__('Fix Now', 'hunnt-ai')); ?>');
        }
    });

    // Post Types gating
    function gatePlan(active){
        $('#hunnt-plan-form .plan-enable, #hunnt-plan-form .plan-count, #plan-save-btn').prop('disabled', !active);
        if (!active) { $('#plan-get-btn').show().off('click').on('click', function(){ window.open(pricingUrl, '_blank'); }); }
        else { $('#plan-get-btn').hide(); }
    }
    $(document).on('change', '.plan-enable', function(){ var type=$(this).data('type'); var on=$(this).is(':checked'); $('input[name="plan_count['+type+']"]').prop('disabled', !on); });
    checkSub(gatePlan);

    // Improve gating
    function gateImprove(active){
        $('#hunnt-improve-form input, #hunnt-improve-form #improve-save-btn').prop('disabled', !active);
        if (!active) { $('#improve-get-btn').show().off('click').on('click', function(){ window.open(pricingUrl, '_blank'); }); }
        else { $('#improve-get-btn').hide(); }
    }
    checkSub(gateImprove);
});
</script>
