<?php

/**
 * Hunnt AI Wizard AJAX Handler
 * 
 * @package HunntAI
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Handle License key validation via AJAX
 */
function hunnt_validate_api_key()
{
    // Security & capability checks
    $nonce = $_POST['_wpnonce'] ?? '';
    if (empty($nonce) || !wp_verify_nonce($nonce, 'hunnt_ai_save_settings_nonce')) {
        wp_send_json_error(__('Security check failed', 'hunnt-ai'));
    }
    if (!current_user_can('manage_options')) {
        wp_send_json_error(__('Unauthorized', 'hunnt-ai'));
    }

    // Normalize field name (support api_key from JS)
    if (isset($_POST['api_key']) && !isset($_POST['hunnt_ai_api_key'])) {
        $_POST['hunnt_ai_api_key'] = $_POST['api_key'];
    }
    $api_key = isset($_POST['hunnt_ai_api_key']) ? sanitize_text_field($_POST['hunnt_ai_api_key']) : '';
    if (empty($api_key)) {
        wp_send_json_error(__('License key is required', 'hunnt-ai'));
    }

    // Ensure correct nonce name for handler check_admin_referer
    if (!isset($_POST['_wpnonce'])) {
        $_POST['_wpnonce'] = $nonce;
    }

    // Intercept the redirect produced by handler and return JSON instead
    add_filter('wp_redirect', function ($location, $status = 302) {
        // Parse redirect query parameters
        $parts = wp_parse_url($location);
        $query = [];
        if (!empty($parts['query'])) {
            parse_str($parts['query'], $query);
        }
        $status_key = isset($query['hunnt_ai_status']) ? sanitize_text_field($query['hunnt_ai_status']) : 'error';
        $message = isset($query['hunnt_ai_message']) ? sanitize_text_field($query['hunnt_ai_message']) : '';

        if ($status_key === 'success') {
            // Get user type from saved option (already set by hunnt_ai_save_settings_handler)
            $user_type = get_option('hunnt_ai_user_type', '');
            $is_agency = ($user_type === 'agency');

            // Check cached Google token status (already set by save-api-hunnt-ai.php)
            $google_verified = $is_agency ? (bool) get_transient('hunnt_ai_google_token_verified') : false;

            wp_send_json_success([
                'message' => __('License key validated successfully!', 'hunnt-ai'),
                'google_verified' => $google_verified,
                'is_agency' => $is_agency
            ]);
        }
        wp_send_json_error(__('Invalid or rejected License key', 'hunnt-ai'));
    }, 1, 2); // Priority 1 so we run before anything else outputs headers

    // Call existing handler (this will reach wp_redirect, our filter will output JSON & exit)
    hunnt_ai_save_settings_handler();
}
add_action('wp_ajax_hunnt_validate_api_key', 'hunnt_validate_api_key');

/**
 * Handle wizard settings save via AJAX
 */
function hunnt_save_wizard_settings()
{
    // Verify nonce
    if (!wp_verify_nonce($_POST['hunnt_wizard_nonce'], 'hunnt_wizard_nonce')) {
        wp_send_json_error(__('Security check failed', 'hunnt-ai'));
    }

    // Save settings
    $settings = array(
        'site_name' => sanitize_text_field($_POST['site_name']),
        'admin_email' => sanitize_email($_POST['admin_email']),
        'ai_language' => sanitize_text_field($_POST['ai_language']),
        'content_type' => sanitize_text_field($_POST['content_type']),
        'enable_seo' => isset($_POST['enable_seo']) ? 1 : 0,
        'enable_content' => isset($_POST['enable_content']) ? 1 : 0,
        'enable_analytics' => isset($_POST['enable_analytics']) ? 1 : 0,
    );

    update_option('hunnt_ai_settings', $settings);
    update_option('hunnt_ai_wizard_completed', true);

    wp_send_json_success(__('Settings saved successfully!', 'hunnt-ai'));
}
add_action('wp_ajax_hunnt_save_wizard_settings', 'hunnt_save_wizard_settings');

/**
 * Get technical SEO status
 */
function hunnt_ai_ajax_get_technical_seo_status()
{
    if (!wp_verify_nonce($_POST['nonce'] ?? '', 'hunnt_wizard_nonce')) {
        wp_send_json_error(__('Security check failed', 'hunnt-ai'));
    }
    if (!current_user_can('manage_options')) {
        wp_send_json_error(__('Unauthorized', 'hunnt-ai'));
    }
    if (!function_exists('hunnt_ai_get_technical_seo_status')) {
        wp_send_json_error(__('Function missing', 'hunnt-ai'));
    }
    $status = hunnt_ai_get_technical_seo_status();
    wp_send_json_success(['status' => $status]);
}
add_action('wp_ajax_hunnt_ai_get_technical_seo_status', 'hunnt_ai_ajax_get_technical_seo_status');

function hunnt_ai_ajax_get_technical_seo_diagnostics()
{
    if (!wp_verify_nonce($_POST['nonce'] ?? '', 'hunnt_wizard_nonce')) {
        wp_send_json_error(__('Security check failed', 'hunnt-ai'));
    }

    if (!current_user_can('manage_options')) {
        wp_send_json_error(__('Unauthorized', 'hunnt-ai'));
    }

    if (!function_exists('hunnt_ai_get_technical_seo_diagnostics')) {
        wp_send_json_error(__('Diagnostics function missing', 'hunnt-ai'));
    }

    $diagnostics = hunnt_ai_get_technical_seo_diagnostics();

    wp_send_json_success([
        'diagnostics' => $diagnostics,
    ]);
}
add_action('wp_ajax_hunnt_ai_get_technical_seo_diagnostics', 'hunnt_ai_ajax_get_technical_seo_diagnostics');

function hunnt_ai_ajax_trigger_technical_seo_warmup()
{
    if (!wp_verify_nonce($_POST['nonce'] ?? '', 'hunnt_wizard_nonce')) {
        wp_send_json_error(__('Security check failed', 'hunnt-ai'));
    }

    if (!current_user_can('manage_options')) {
        wp_send_json_error(__('Unauthorized', 'hunnt-ai'));
    }

    if (!function_exists('hunnt_ai_is_technical_seo_enabled') || !hunnt_ai_is_technical_seo_enabled()) {
        wp_send_json_error(__('Technical SEO is disabled', 'hunnt-ai'));
    }

    if (!get_option('hunnt_ai_assets_minify_enable')) {
        wp_send_json_error(__('Asset optimization is disabled', 'hunnt-ai'));
    }

    $result = false;

    if (function_exists('hunnt_ai_runtime_minify_warm_cache')) {
        $result = (bool) hunnt_ai_runtime_minify_warm_cache();
    }

    update_option('hunnt_ai_technical_seo_last_warmup', current_time('timestamp'));
    update_option('hunnt_ai_technical_seo_last_warmup_result', $result ? 'success' : 'failed');

    $response = [
        'result' => $result ? 'success' : 'failed',
    ];

    if (function_exists('hunnt_ai_get_technical_seo_diagnostics')) {
        $response['diagnostics'] = hunnt_ai_get_technical_seo_diagnostics();
    }

    if (function_exists('hunnt_ai_get_static_cache_summary')) {
        $response['summary'] = hunnt_ai_get_static_cache_summary();
    }

    wp_send_json_success($response);
}
add_action('wp_ajax_hunnt_ai_trigger_technical_seo_warmup', 'hunnt_ai_ajax_trigger_technical_seo_warmup');

function hunnt_ai_ajax_rebuild_technical_seo_assets()
{
    if (!wp_verify_nonce($_POST['nonce'] ?? '', 'hunnt_wizard_nonce')) {
        wp_send_json_error(__('Security check failed', 'hunnt-ai'));
    }

    if (!current_user_can('manage_options')) {
        wp_send_json_error(__('Unauthorized', 'hunnt-ai'));
    }

    if (!function_exists('hunnt_ai_is_technical_seo_enabled') || !hunnt_ai_is_technical_seo_enabled()) {
        wp_send_json_error(__('Technical SEO is disabled', 'hunnt-ai'));
    }

    if (!function_exists('hunnt_ai_apply_minify_assets')) {
        wp_send_json_error(__('Optimization function missing', 'hunnt-ai'));
    }

    $result = (bool) hunnt_ai_apply_minify_assets();

    $response = [
        'result' => $result ? 'success' : 'failed',
    ];

    if (function_exists('hunnt_ai_get_technical_seo_diagnostics')) {
        $response['diagnostics'] = hunnt_ai_get_technical_seo_diagnostics();
    }

    wp_send_json_success($response);
}
add_action('wp_ajax_hunnt_ai_rebuild_technical_seo_assets', 'hunnt_ai_ajax_rebuild_technical_seo_assets');

function hunnt_ai_ajax_toggle_static_cache()
{
    if (!wp_verify_nonce($_POST['nonce'] ?? '', 'hunnt_wizard_nonce')) {
        wp_send_json_error(__('Security check failed', 'hunnt-ai'));
    }

    if (!current_user_can('manage_options')) {
        wp_send_json_error(__('Unauthorized', 'hunnt-ai'));
    }

    $enable = isset($_POST['enable']) ? (bool) $_POST['enable'] : false;

    if ($enable) {
        if (function_exists('hunnt_ai_is_technical_seo_enabled') && !hunnt_ai_is_technical_seo_enabled()) {
            wp_send_json_error(__('Enable Technical SEO to use static cache.', 'hunnt-ai'));
        }

        if (function_exists('hunnt_ai_check_subscription_active')) {
            $subscription = hunnt_ai_check_subscription_active();
            if (empty($subscription['active'])) {
                wp_send_json_error(__('Technical SEO subscription required to enable static cache.', 'hunnt-ai'));
            }
        }
    }

    $result = $enable ? hunnt_ai_enable_static_cache() : hunnt_ai_disable_static_cache();
    $summary = hunnt_ai_get_static_cache_summary();

    if ($enable) {
        if (!$result || empty($summary['has_write_access'])) {
            $message = __('Unable to update static cache setting.', 'hunnt-ai');
            if (!empty($summary['last_error']['message'])) {
                $message = $summary['last_error']['message'];
            }
            wp_send_json_error($message);
        }
    } elseif (!$result) {
        wp_send_json_error(__('Unable to update static cache setting.', 'hunnt-ai'));
    }

    wp_send_json_success([
        'summary' => $summary,
    ]);
}
add_action('wp_ajax_hunnt_ai_toggle_static_cache', 'hunnt_ai_ajax_toggle_static_cache');

function hunnt_ai_ajax_static_cache_purge()
{
    if (!wp_verify_nonce($_POST['nonce'] ?? '', 'hunnt_wizard_nonce')) {
        wp_send_json_error(__('Security check failed', 'hunnt-ai'));
    }

    if (!current_user_can('manage_options')) {
        wp_send_json_error(__('Unauthorized', 'hunnt-ai'));
    }

    $summary_before = hunnt_ai_get_static_cache_summary();

    if (empty($summary_before['has_write_access'])) {
        $message = __('Failed to purge static cache.', 'hunnt-ai');
        if (!empty($summary_before['last_error']['message'])) {
            $message = $summary_before['last_error']['message'];
        }
        wp_send_json_error($message);
    }

    $success = hunnt_ai_static_cache_purge_all();

    if (!$success) {
        wp_send_json_error(__('Failed to purge static cache.', 'hunnt-ai'));
    }

    wp_send_json_success([
        'summary' => hunnt_ai_get_static_cache_summary(),
    ]);
}
add_action('wp_ajax_hunnt_ai_static_cache_purge', 'hunnt_ai_ajax_static_cache_purge');

function hunnt_ai_ajax_static_cache_warm()
{
    if (!wp_verify_nonce($_POST['nonce'] ?? '', 'hunnt_wizard_nonce')) {
        wp_send_json_error(__('Security check failed', 'hunnt-ai'));
    }

    if (!current_user_can('manage_options')) {
        wp_send_json_error(__('Unauthorized', 'hunnt-ai'));
    }

    if (function_exists('hunnt_ai_is_technical_seo_enabled') && !hunnt_ai_is_technical_seo_enabled()) {
        wp_send_json_error(__('Enable Technical SEO to warm the static cache.', 'hunnt-ai'));
    }

    $summary_before = hunnt_ai_get_static_cache_summary();

    if (empty($summary_before['has_write_access'])) {
        $message = __('Static cache directories are not writable.', 'hunnt-ai');
        if (!empty($summary_before['last_error']['message'])) {
            $message = $summary_before['last_error']['message'];
        }
        wp_send_json_error($message);
    }

    $urls = isset($_POST['urls']) ? (array) $_POST['urls'] : [];
    $urls = array_filter(array_map('esc_url_raw', $urls));
    if (empty($urls)) {
        $urls = [home_url('/')];
    }

    $results = hunnt_ai_static_cache_warm_urls($urls);

    wp_send_json_success([
        'results' => $results,
        'summary' => hunnt_ai_get_static_cache_summary(),
    ]);
}
add_action('wp_ajax_hunnt_ai_static_cache_warm', 'hunnt_ai_ajax_static_cache_warm');

function hunnt_ai_ajax_static_cache_summary()
{
    if (!wp_verify_nonce($_POST['nonce'] ?? '', 'hunnt_wizard_nonce')) {
        wp_send_json_error(__('Security check failed', 'hunnt-ai'));
    }

    if (!current_user_can('manage_options')) {
        wp_send_json_error(__('Unauthorized', 'hunnt-ai'));
    }

    wp_send_json_success([
        'summary' => hunnt_ai_get_static_cache_summary(),
    ]);
}
add_action('wp_ajax_hunnt_ai_static_cache_summary', 'hunnt_ai_ajax_static_cache_summary');

/**
 * Apply technical SEO fixes (requires active subscription)
 */
function hunnt_ai_ajax_apply_technical_seo()
{
    if (!wp_verify_nonce($_POST['nonce'] ?? '', 'hunnt_wizard_nonce')) {
        wp_send_json_error(__('Security check failed', 'hunnt-ai'));
    }
    if (!current_user_can('manage_options')) {
        wp_send_json_error(__('Unauthorized', 'hunnt-ai'));
    }
    $features = isset($_POST['features']) && is_array($_POST['features']) ? array_map('sanitize_text_field', $_POST['features']) : [];

    if (!function_exists('hunnt_ai_check_subscription_active')) {
        wp_send_json_error(__('Missing subscription checker', 'hunnt-ai'));
    }
    $subscription = hunnt_ai_check_subscription_active();
    if (empty($subscription['active'])) {
        wp_send_json_error(__('No active subscription', 'hunnt-ai'));
    }

    $applied = [];
    $errors = [];

    if (in_array('cache', $features, true)) {
        hunnt_ai_enable_basic_cache_headers();
        $applied[] = 'cache';
    }
    if (in_array('minify_css', $features, true) || in_array('minify_js', $features, true)) {
        if (hunnt_ai_apply_minify_assets()) {
            if (in_array('minify_css', $features, true))
                $applied[] = 'minify_css';
            if (in_array('minify_js', $features, true))
                $applied[] = 'minify_js';
        } else {
            $errors[] = 'minify_assets_failed';
        }
    }
    if (in_array('minify_html', $features, true)) {
        hunnt_ai_enable_html_minify();
        $applied[] = 'minify_html';
    }
    if (in_array('gzip', $features, true)) {
        if (hunnt_ai_apply_gzip()) {
            $applied[] = 'gzip';
        } else {
            $errors[] = 'gzip_failed';
        }
    }

    $status = hunnt_ai_get_technical_seo_status();
    if (function_exists('hunnt_ai_notify_technical_seo_status')) {
        hunnt_ai_notify_technical_seo_status($status);
    }

    $log = [];
    if (function_exists('hunnt_ai_get_minify_summary')) {
        $log['minify'] = hunnt_ai_get_minify_summary();
    }

    if (!empty($log) && function_exists('hunnt_ai_store_minify_log')) {
        hunnt_ai_store_minify_log($log);
    }

    wp_send_json_success([
        'applied' => $applied,
        'errors' => $errors,
        'status' => $status,
    ]);
}
add_action('wp_ajax_hunnt_ai_apply_technical_seo', 'hunnt_ai_ajax_apply_technical_seo');

/**
 * Disable technical SEO features via AJAX
 * Requires admin capabilities
 */
function hunnt_ai_ajax_disable_technical_seo()
{
    // Verify nonce
    if (!wp_verify_nonce($_POST['nonce'] ?? '', 'hunnt_wizard_nonce')) {
        wp_send_json_error(__('Security check failed', 'hunnt-ai'));
    }

    // Check user permissions
    if (!current_user_can('manage_options')) {
        wp_send_json_error(__('Unauthorized', 'hunnt-ai'));
    }

    // Check if disable function exists
    if (!function_exists('hunnt_ai_disable_technical_seo')) {
        wp_send_json_error(__('Disable function missing', 'hunnt-ai'));
    }

    // Disable all technical SEO features
    $disabled = hunnt_ai_disable_technical_seo();

    if (!$disabled) {
        wp_send_json_error(__('Failed to disable technical SEO', 'hunnt-ai'));
    }

    // Get updated status
    $status = function_exists('hunnt_ai_get_technical_seo_status')
        ? hunnt_ai_get_technical_seo_status()
        : [];
    $static_cache_summary = function_exists('hunnt_ai_get_static_cache_summary')
        ? hunnt_ai_get_static_cache_summary()
        : [];
    $static_cache_summary = function_exists('hunnt_ai_get_static_cache_summary')
        ? hunnt_ai_get_static_cache_summary()
        : [];

    wp_send_json_success([
        'message' => __('Technical SEO has been disabled successfully', 'hunnt-ai'),
        'status' => $status,
        'enabled' => hunnt_ai_is_technical_seo_enabled(),
        'static_cache_summary' => $static_cache_summary,
    ]);
}
add_action('wp_ajax_hunnt_ai_disable_technical_seo', 'hunnt_ai_ajax_disable_technical_seo');

/**
 * Enable technical SEO features via AJAX
 * Requires admin capabilities
 * Immediately applies all optimizations that were previously configured
 */
function hunnt_ai_ajax_enable_technical_seo()
{
    // Verify nonce
    if (!wp_verify_nonce($_POST['nonce'] ?? '', 'hunnt_wizard_nonce')) {
        wp_send_json_error(__('Security check failed', 'hunnt-ai'));
    }

    // Check user permissions
    if (!current_user_can('manage_options')) {
        wp_send_json_error(__('Unauthorized', 'hunnt-ai'));
    }

    // Check subscription
    if (!function_exists('hunnt_ai_check_subscription_active')) {
        wp_send_json_error(__('Subscription checker missing', 'hunnt-ai'));
    }

    $subscription = hunnt_ai_check_subscription_active();
    if (empty($subscription['active'])) {
        wp_send_json_error(__('No active subscription. Technical SEO requires an active subscription.', 'hunnt-ai'));
    }

    // Enable technical SEO globally first
    if (!function_exists('hunnt_ai_enable_technical_seo')) {
        wp_send_json_error(__('Enable function missing', 'hunnt-ai'));
    }

    $enabled = hunnt_ai_enable_technical_seo();
    if (!$enabled) {
        wp_send_json_error(__('Failed to enable technical SEO', 'hunnt-ai'));
    }

    // Now immediately apply all the optimizations
    $applied = [];
    $errors = [];

    // Apply cache headers
    if (function_exists('hunnt_ai_enable_basic_cache_headers')) {
        hunnt_ai_enable_basic_cache_headers();
        $applied[] = 'cache';
    }

    // Apply HTML minification
    if (function_exists('hunnt_ai_enable_html_minify')) {
        hunnt_ai_enable_html_minify();
        $applied[] = 'minify_html';
    }

    // Apply asset minification (this generates the files)
    if (function_exists('hunnt_ai_apply_minify_assets')) {
        $asset_result = hunnt_ai_apply_minify_assets();
        if ($asset_result) {
            $applied[] = 'minify_css';
            $applied[] = 'minify_js';
        } else {
            $errors[] = 'minify_assets_failed';
        }
    }

    // Apply Gzip
    if (function_exists('hunnt_ai_apply_gzip')) {
        $gzip_result = hunnt_ai_apply_gzip();
        if ($gzip_result) {
            $applied[] = 'gzip';
        } else {
            $errors[] = 'gzip_failed';
        }
    }

    // Get updated status after applying everything
    $status = function_exists('hunnt_ai_get_technical_seo_status')
        ? hunnt_ai_get_technical_seo_status()
        : [];

    // Notify backend
    if (function_exists('hunnt_ai_notify_technical_seo_status')) {
        hunnt_ai_notify_technical_seo_status($status);
    }

    $static_cache_summary = function_exists('hunnt_ai_get_static_cache_summary')
        ? hunnt_ai_get_static_cache_summary()
        : [];

    wp_send_json_success([
        'message' => __('Technical SEO has been enabled and all optimizations applied successfully.', 'hunnt-ai'),
        'status' => $status,
        'enabled' => hunnt_ai_is_technical_seo_enabled(),
        'applied' => $applied,
        'errors' => $errors,
        'static_cache_summary' => $static_cache_summary,
    ]);
}
add_action('wp_ajax_hunnt_ai_enable_technical_seo', 'hunnt_ai_ajax_enable_technical_seo');

/**
 * Check subscription status via AJAX
 */
function hunnt_ai_ajax_check_subscription_status()
{
    if (!wp_verify_nonce($_POST['nonce'] ?? '', 'hunnt_wizard_nonce')) {
        wp_send_json_error(__('Security check failed', 'hunnt-ai'));
    }
    if (!current_user_can('manage_options')) {
        wp_send_json_error(__('Unauthorized', 'hunnt-ai'));
    }
    if (!function_exists('hunnt_ai_check_subscription_active')) {
        wp_send_json_error(__('Missing subscription checker', 'hunnt-ai'));
    }
    $result = hunnt_ai_check_subscription_active();
    wp_send_json_success([
        'active' => (bool) ($result['active'] ?? false),
        'message' => $result['message'] ?? '',
        'raw' => $result['raw'] ?? null,
    ]);
}
add_action('wp_ajax_hunnt_ai_check_subscription_status', 'hunnt_ai_ajax_check_subscription_status');
