<?php
/**
 * Static cache manager.
 *
 * @package Hunnt_AI
 */

if (!defined('ABSPATH')) {
    exit;
}

class Hunnt_AI_Static_Cache_Manager {
    private static bool $buffer_started = false;
    private static ?string $current_cache_key = null;
    private static ?array $current_metadata = null;

    /**
     * Register bootstrap hooks.
     */
    public static function bootstrap(): void {
        add_action('plugins_loaded', [__CLASS__, 'init'], 25);
        add_action('admin_init', [__CLASS__, 'maybe_install_dropin']);
    }

    /**
     * Set up runtime behaviour when the feature is enabled.
     */
    public static function init(): void {
        if (!self::is_enabled()) {
            return;
        }

        if (!Hunnt_AI_Static_Cache_Helper::ensure_directories()) {
            return;
        }
        self::write_runtime_config();

        add_action('template_redirect', [__CLASS__, 'attempt_runtime_serve'], 0);
        add_action('template_redirect', [__CLASS__, 'start_buffer'], PHP_INT_MAX);

        // Purge on content changes.
        add_action('save_post', [__CLASS__, 'purge_related_to_post'], 99, 3);
        add_action('deleted_post', [__CLASS__, 'purge_related_to_post_id']);
        add_action('trashed_post', [__CLASS__, 'purge_related_to_post_id']);
        add_action('wp_update_comment', [__CLASS__, 'purge_from_comment'], 99, 2);
        add_action('comment_post', [__CLASS__, 'purge_from_comment'], 99, 2);
        add_action('wp_set_comment_status', [__CLASS__, 'purge_comment_status'], 99, 2);
        add_action('switch_theme', [__CLASS__, 'purge_all']);
    }

    /**
     * Determine if static cache is enabled.
     */
    public static function is_enabled(): bool {
        $enabled = (bool) get_option('hunnt_ai_static_cache_enabled', false);
        if (!$enabled) {
            return false;
        }

        if (!self::is_technical_seo_active()) {
            return false;
        }

        return true;
    }

    /**
     * Enable or disable the feature programmatically.
     */
    public static function update_enabled(bool $enabled): void {
        $tech_active = self::is_technical_seo_active();

        if ($enabled && !$tech_active) {
            $enabled = false;
        }

        if ($enabled) {
            if (!Hunnt_AI_Static_Cache_Helper::ensure_directories()) {
                update_option('hunnt_ai_static_cache_enabled', 0);
                return;
            }

            update_option('hunnt_ai_static_cache_enabled', 1);
            self::write_runtime_config();
            self::maybe_install_dropin();
        } else {
            update_option('hunnt_ai_static_cache_enabled', 0);
            self::purge_all();
            self::remove_dropin_if_owned();
            self::delete_runtime_config();
        }
    }

    private static function is_technical_seo_active(): bool {
        if (function_exists('hunnt_ai_is_technical_seo_enabled')) {
            return (bool) hunnt_ai_is_technical_seo_enabled();
        }

        return true;
    }

    /**
     * Attempt to serve a cached page during the WordPress lifecycle.
     */
    public static function attempt_runtime_serve(): void {
        if (!self::should_cache_current_request()) {
            return;
        }

        $key = self::build_current_cache_key();
        if (!$key) {
            return;
        }

        $meta = self::read_metadata($key);
        if (!$meta || self::is_meta_expired($meta)) {
            return;
        }

        $html_path = Hunnt_AI_Static_Cache_Helper::get_page_path($key);
        $gzip_path = Hunnt_AI_Static_Cache_Helper::get_gzip_path($key);

        if (!is_readable($html_path)) {
            return;
        }

        self::send_cached_response($meta, $html_path, $gzip_path);
    }

    /**
     * Begin output buffering to capture fresh HTML for caching.
     */
    public static function start_buffer(): void {
        if (self::$buffer_started) {
            return;
        }

        if (!self::should_cache_current_request()) {
            return;
        }

        $key = self::build_current_cache_key();
        if (!$key) {
            return;
        }

        self::$buffer_started   = true;
        self::$current_cache_key = $key;
        self::$current_metadata = [
            'generated' => time(),
            'host'      => hunnt_ai_static_cache_normalize_host($_SERVER['HTTP_HOST'] ?? ''),
            'uri'       => hunnt_ai_static_cache_normalize_uri($_SERVER['REQUEST_URI'] ?? '/'),
            'variant'   => Hunnt_AI_Static_Cache_Helper::detect_variant(),
        ];

        ob_start([__CLASS__, 'buffer_callback']);
    }

    /**
     * Output buffer callback used to persist the page contents.
     */
    public static function buffer_callback(string $html): string {
        if (!self::$buffer_started || !self::$current_cache_key) {
            return $html;
        }

        try {
            self::store_page(self::$current_cache_key, $html);
    } catch (\Throwable $exception) {
            error_log('Hunnt AI static cache store failed: ' . $exception->getMessage());
        }

        self::$buffer_started = false;
        self::$current_cache_key = null;
        self::$current_metadata = null;

        return $html;
    }

    /**
     * Store the generated HTML and related metadata to disk.
     */
    private static function store_page(string $key, string $html): void {
        if (trim($html) === '') {
            return;
        }

        if (is_404() || is_search()) {
            return;
        }

        $status_code = http_response_code();
        if (!in_array($status_code, [200, 203], true)) {
            return;
        }

        $normalized_html = self::normalize_html_payload($html);

        $meta = self::$current_metadata ?: [];
        $meta['status']  = $status_code;
        $meta['headers'] = self::collect_response_headers();
        $meta['expires'] = time() + self::get_cache_ttl();

        if (!Hunnt_AI_Static_Cache_Helper::ensure_directories()) {
            return;
        }

        $html_path = Hunnt_AI_Static_Cache_Helper::get_page_path($key);
        $gzip_path = Hunnt_AI_Static_Cache_Helper::get_gzip_path($key);
        $meta_path = Hunnt_AI_Static_Cache_Helper::get_meta_path($key);

        self::write_file_atomic($html_path, $normalized_html);

        if (self::client_supports_gzip() && function_exists('gzencode')) {
            $gzip = gzencode($normalized_html, 6);
            if ($gzip !== false) {
                self::write_file_atomic($gzip_path, $gzip);
            }
        }

        $meta_export = var_export($meta, true);
        self::write_file_atomic($meta_path, "<?php\nreturn " . $meta_export . ";\n");
    }

    /**
     * Ensure HTML payload is not stored in compressed form.
     */
    private static function normalize_html_payload(string $html): string {
        if ($html === '') {
            return $html;
        }

        $magic = substr($html, 0, 2);
        if ($magic === "\x1f\x8b" && function_exists('gzdecode')) {
            $decoded = @gzdecode($html);
            if ($decoded !== false && $decoded !== '') {
                return $decoded;
            }
        }

        return $html;
    }

    /**
     * Collect response headers that should be replayed for cached pages.
     */
    private static function collect_response_headers(): array {
        if (!function_exists('headers_list')) {
            return [];
        }

        $raw_headers = headers_list();
        $persist = apply_filters('hunnt_ai_static_cache_persist_headers', [
            'content-type',
            'cache-control',
            'link',
            'expires',
            'vary',
        ]);

        $persist = array_map('strtolower', (array) $persist);
        $allowed_all = in_array('*', $persist, true);

        $headers = [];
        foreach ($raw_headers as $line) {
            if (stripos($line, 'set-cookie:') === 0) {
                continue;
            }

            $parts = explode(':', $line, 2);
            if (count($parts) !== 2) {
                continue;
            }

            [$name, $value] = $parts;
            $name  = trim($name);
            $value = trim($value);
            if ($name === '') {
                continue;
            }

            $key = strtolower($name);
            if (!$allowed_all && !in_array($key, $persist, true)) {
                continue;
            }

            $headers[$name] = $value;
        }

        return $headers;
    }

    /**
     * Serve cached response and exit.
     */
    private static function send_cached_response(array $meta, string $html_path, string $gzip_path): void {
        $headers_already_sent = headers_sent();
        $supports_gzip = self::client_supports_gzip();
        $html_readable = is_readable($html_path);
        $gzip_readable = is_readable($gzip_path);

        // Only serve gzip when headers can be modified so clients know how to decode it.
        $use_gzip = !$headers_already_sent && $supports_gzip && ($html_readable || $gzip_readable);

        $gzip_payload = null;
        if ($use_gzip) {
            if ($html_readable) {
                $source = file_get_contents($html_path);
                if ($source !== false && function_exists('gzencode')) {
                    $gzip_payload = gzencode($source, 6);
                }
            }

            if ($gzip_payload === null && $gzip_readable) {
                $existing = file_get_contents($gzip_path);
                if ($existing !== false) {
                    $gzip_payload = $existing;
                }
            }

            if ($gzip_payload === null) {
                $use_gzip = false;
            }
        }
        if (!$headers_already_sent) {
            header_remove('Last-Modified');
            $status = (int) ($meta['status'] ?? 200);
            status_header($status);

            $headers = isset($meta['headers']) && is_array($meta['headers']) ? $meta['headers'] : [];
            foreach ($headers as $name => $value) {
                $normalized_name = strtolower($name);
                if ($normalized_name === 'content-encoding' || $normalized_name === 'content-length') {
                    continue;
                }
                header($name . ': ' . $value, true);
            }

            $sent_headers = array_change_key_case($headers, CASE_LOWER);
            if (!isset($sent_headers['content-type'])) {
                header('Content-Type: text/html; charset=UTF-8');
            }

            header('X-Cache-Hit: HunntAI');
            if ($use_gzip) {
                header('Content-Encoding: gzip', true);
                header('Vary: Accept-Encoding', false);
                if ($gzip_payload !== null) {
                    header('Content-Length: ' . strlen($gzip_payload));
                }
            } elseif ($html_readable) {
                $size = filesize($html_path);
                if ($size !== false) {
                    header('Content-Length: ' . $size);
                }
            }
        }

        if ($use_gzip && $gzip_payload !== null) {
            echo $gzip_payload;
        } else {
            $file = $html_path;
            if ($use_gzip && $gzip_readable) {
                $file = $gzip_path;
            }

            if (is_readable($file)) {
                readfile($file);
            }
        }
        exit;
    }

    /**
     * Determine whether the current client accepts gzip responses.
     */
    private static function client_supports_gzip(): bool {
        $encoding = $_SERVER['HTTP_ACCEPT_ENCODING'] ?? '';

        return stripos($encoding, 'gzip') !== false;
    }

    /**
     * Build cache key for the current HTTP request.
     */
    private static function build_current_cache_key(): ?string {
        $host = hunnt_ai_static_cache_normalize_host($_SERVER['HTTP_HOST'] ?? '');
        $uri  = hunnt_ai_static_cache_normalize_uri($_SERVER['REQUEST_URI'] ?? '/');
        $variant = Hunnt_AI_Static_Cache_Helper::detect_variant();

        return Hunnt_AI_Static_Cache_Helper::build_cache_key($host, $uri, $variant);
    }

    /**
     * Determine if current request is cacheable.
     */
    private static function should_cache_current_request(): bool {
        if (!self::is_enabled()) {
            return false;
        }

        if (Hunnt_AI_Static_Cache_Helper::should_bypass_current_request()) {
            return false;
        }

        if (is_feed() || is_trackback()) {
            return false;
        }

        if (is_preview()) {
            return false;
        }

        if (is_user_logged_in()) {
            return false;
        }

        if (apply_filters('hunnt_ai_static_cache_skip_request', false)) {
            return false;
        }

        return true;
    }

    /**
     * Retrieve metadata for a cache key.
     */
    private static function read_metadata(string $key): ?array {
        $meta_path = Hunnt_AI_Static_Cache_Helper::get_meta_path($key);
        if (!is_readable($meta_path)) {
            return null;
        }

        $data = include $meta_path;
        return is_array($data) ? $data : null;
    }

    /**
     * Determine if cache metadata has expired.
     */
    private static function is_meta_expired(array $meta): bool {
        $expires = isset($meta['expires']) ? (int) $meta['expires'] : 0;

        return $expires > 0 && time() > $expires;
    }

    /**
     * Write file atomically.
     */
    private static function write_file_atomic(string $path, string $contents): void {
        $dir = dirname($path);
        if (!is_dir($dir)) {
            wp_mkdir_p($dir);
        }

        $tmp = self::create_temp_file_for_path($dir, $path);
        if (!$tmp) {
            file_put_contents($path, $contents);
            return;
        }

        file_put_contents($tmp, $contents);
        @chmod($tmp, 0644);
        if (!@rename($tmp, $path)) {
            // Fallback: attempt to overwrite directly if atomic rename fails.
            file_put_contents($path, $contents);
            @unlink($tmp);
        }
    }

    /**
     * Create a temporary file for atomic writes, loading WordPress helpers when available.
     */
    private static function create_temp_file_for_path(string $dir, string $path): ?string {
        if (self::ensure_wp_tempnam_available()) {
            $tmp = wp_tempnam($path);
            if ($tmp) {
                return $tmp;
            }
        }

        return self::tempnam_fallback($dir, $path);
    }

    /**
     * Ensure the WordPress file helper that declares wp_tempnam() is loaded.
     */
    private static function ensure_wp_tempnam_available(): bool {
        if (function_exists('wp_tempnam')) {
            return true;
        }

        if (!defined('ABSPATH')) {
            return false;
        }

        $file_include = rtrim(ABSPATH, "/\\") . '/wp-admin/includes/file.php';
        if (is_readable($file_include)) {
            require_once $file_include;
        }

        return function_exists('wp_tempnam');
    }

    /**
     * Provide a filesystem-level fallback when wp_tempnam() is unavailable.
     */
    private static function tempnam_fallback(string $dir, string $path): ?string {
        if (!is_dir($dir)) {
            if (!wp_mkdir_p($dir)) {
                return null;
            }
        }

        $prefix = 'hunnt_' . substr(md5($path . microtime(true)), 0, 8);
        $tmp = tempnam($dir, $prefix);
        if ($tmp === false) {
            return null;
        }

        return $tmp;
    }

    /**
     * Return TTL for cached pages.
     */
    private static function get_cache_ttl(): int {
        $ttl = (int) get_option('hunnt_ai_static_cache_ttl', 0);
        if ($ttl <= 0) {
            $ttl = Hunnt_AI_Static_Cache_Helper::get_default_ttl();
        }

        return max(60, $ttl);
    }

    /**
     * Purge cache entries related to a post.
     */
    public static function purge_related_to_post(int $post_id, \WP_Post $post, bool $update): void {
        if ($post_id <= 0) {
            return;
        }

        self::purge_url(get_permalink($post_id));
        self::purge_url(home_url('/'));
    }

    /**
     * Purge cache for a post ID.
     */
    public static function purge_related_to_post_id(int $post_id): void {
        if ($post_id <= 0) {
            return;
        }

        self::purge_url(get_permalink($post_id));
    }

    /**
     * Purge cache entries related to comment updates.
     */
    public static function purge_from_comment($comment_id, $comment = null): void {
        $post_id = 0;
        if ($comment && is_object($comment) && isset($comment->comment_post_ID)) {
            $post_id = (int) $comment->comment_post_ID;
        } elseif (is_numeric($comment_id)) {
            $comment_obj = get_comment((int) $comment_id);
            if ($comment_obj) {
                $post_id = (int) $comment_obj->comment_post_ID;
            }
        }

        if ($post_id > 0) {
            self::purge_related_to_post_id($post_id);
        }
    }

    /**
     * Purge on comment status changes.
     */
    public static function purge_comment_status($comment_id, $status): void {
        self::purge_from_comment($comment_id);
    }

    /**
     * Purge cache for a given URL.
     */
    public static function purge_url(?string $url): void {
        if (!$url) {
            return;
        }

        $parts = wp_parse_url($url);
        if (!$parts || empty($parts['host'])) {
            return;
        }

        $host = hunnt_ai_static_cache_normalize_host($parts['host']);
        $uri  = hunnt_ai_static_cache_normalize_uri($parts['path'] ?? '/');
        if (!empty($parts['query'])) {
            $uri .= '?' . $parts['query'];
        }

        $variant = 'desktop';
        foreach (['desktop', 'mobile'] as $candidate) {
            $key = Hunnt_AI_Static_Cache_Helper::build_cache_key($host, $uri, $candidate);
            self::delete_cache_key($key);
        }
    }

    /**
     * Purge the entire cache directory.
     */
    public static function purge_all(): bool {
        $root = Hunnt_AI_Static_Cache_Helper::get_cache_root_dir();
        if (!is_dir($root)) {
            return true;
        }

        $iterator = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($root, \RecursiveDirectoryIterator::SKIP_DOTS),
            \RecursiveIteratorIterator::CHILD_FIRST
        );

        foreach ($iterator as $file) {
            if ($file->isDir()) {
                @rmdir($file->getPathname());
            } else {
                @unlink($file->getPathname());
            }
        }

        return true;
    }

    /**
     * Delete runtime cache config.
     */
    private static function delete_runtime_config(): void {
        $config = Hunnt_AI_Static_Cache_Helper::get_config_path();
        if (file_exists($config)) {
            @unlink($config);
        }
    }

    /**
     * Delete cached payload for key.
     */
    private static function delete_cache_key(string $key): void {
        $paths = [
            Hunnt_AI_Static_Cache_Helper::get_page_path($key),
            Hunnt_AI_Static_Cache_Helper::get_gzip_path($key),
            Hunnt_AI_Static_Cache_Helper::get_meta_path($key),
        ];

        foreach ($paths as $path) {
            if (file_exists($path)) {
                @unlink($path);
            }
        }
    }

    /**
     * Warm the cache for a set of URLs.
     */
    public static function warm_urls(array $urls): array {
        $results = [];

        foreach ($urls as $url) {
            if (!is_string($url) || $url === '') {
                continue;
            }

            $sanitized = esc_url_raw($url);
            if ($sanitized === '') {
                continue;
            }

            $response = wp_remote_get($sanitized, [
                'timeout' => (int) apply_filters('hunnt_ai_static_cache_warm_timeout', 8),
                'sslverify' => (bool) apply_filters('hunnt_ai_static_cache_warm_sslverify', true),
                'headers' => [
                    'Cache-Control' => 'no-cache',
                    'Pragma'        => 'no-cache',
                ],
            ]);

            $is_error = is_wp_error($response);
            $code = $is_error ? 0 : (int) wp_remote_retrieve_response_code($response);
            $success = !$is_error && $code >= 200 && $code < 400;

            $results[] = [
                'url'     => $sanitized,
                'success' => $success,
                'code'    => $code,
                'error'   => $is_error ? $response->get_error_message() : null,
            ];
        }

        return $results;
    }

    /**
     * Collect cache inventory metrics.
     */
    public static function get_cache_inventory(): array {
        $root = Hunnt_AI_Static_Cache_Helper::get_cache_root_dir();
        $pages_dir = trailingslashit($root) . Hunnt_AI_Static_Cache_Helper::PAGES_SUBDIR;

        $count = 0;
        $bytes = 0;

        if (is_dir($pages_dir)) {
            $iterator = new \DirectoryIterator($pages_dir);
            foreach ($iterator as $fileinfo) {
                if ($fileinfo->isDot() || !$fileinfo->isFile()) {
                    continue;
                }
                if (substr($fileinfo->getFilename(), -5) !== '.html') {
                    continue;
                }
                $count++;
                $bytes += $fileinfo->getSize();
            }
        }

        return [
            'count' => $count,
            'bytes' => $bytes,
        ];
    }

    /**
     * Install drop-in if conditions allow.
     */
    public static function maybe_install_dropin(): void {
        if (!self::is_enabled()) {
            return;
        }

        $dropin_path = WP_CONTENT_DIR . '/advanced-cache.php';
        $template    = HUNNT_AI_PATH . 'includes/static-cache/dropin-template.php';

        if (!defined('WP_CACHE') || !WP_CACHE) {
            return;
        }

        if (!is_readable($template)) {
            return;
        }

        if (file_exists($dropin_path)) {
            $contents = file_get_contents($dropin_path);
            if ($contents === false || stripos($contents, Hunnt_AI_Static_Cache_Helper::DROPIN_SIGNATURE) === false) {
                return; // Another plugin controls the drop-in.
            }
        }

        $template_contents = file_get_contents($template);
        if ($template_contents === false) {
            return;
        }

        $template_contents = str_replace(
            ['{{signature}}', '{{generated}}'],
            [Hunnt_AI_Static_Cache_Helper::DROPIN_SIGNATURE, gmdate('c')],
            $template_contents
        );

        file_put_contents($dropin_path, $template_contents);
        @chmod($dropin_path, 0644);
    }

    /**
     * Remove the drop-in if it belongs to us.
     */
    private static function remove_dropin_if_owned(): void {
        $dropin_path = WP_CONTENT_DIR . '/advanced-cache.php';
        if (!file_exists($dropin_path)) {
            return;
        }

        $contents = file_get_contents($dropin_path);
        if ($contents !== false && stripos($contents, Hunnt_AI_Static_Cache_Helper::DROPIN_SIGNATURE) !== false) {
            @unlink($dropin_path);
        }
    }

    /**
     * Write runtime config for the drop-in.
     */
    private static function write_runtime_config(): void {
        $config_path = Hunnt_AI_Static_Cache_Helper::get_config_path();

        $config = [
            'enabled'   => true,
            'ttl'       => self::get_cache_ttl(),
            'cache_dir' => Hunnt_AI_Static_Cache_Helper::get_cache_root_dir(),
            'generated' => time(),
            'query_whitelist' => self::get_query_whitelist_setting(),
            'cookie_bypass_patterns' => self::get_cookie_bypass_patterns(),
        ];

        $export = var_export($config, true);
        self::write_file_atomic($config_path, "<?php\nreturn " . $export . ";\n");
    }

    /**
     * Normalized whitelist of query string parameters.
     *
     * @return array|true
     */
    private static function get_query_whitelist_setting() {
        return Hunnt_AI_Static_Cache_Helper::get_query_whitelist();
    }

    /**
     * Return cookie bypass patterns.
     */
    private static function get_cookie_bypass_patterns(): array {
        return Hunnt_AI_Static_Cache_Helper::get_cookie_bypass_patterns();
    }

    /**
     * Activation hook.
     */
    public static function on_activation(): void {
        Hunnt_AI_Static_Cache_Helper::ensure_directories();
        self::write_runtime_config();
    }

    /**
     * Deactivation hook.
     */
    public static function on_deactivation(): void {
        self::remove_dropin_if_owned();
        self::delete_runtime_config();
    }
}
