<?php
/**
 * Static cache helper utilities.
 *
 * @package Hunnt_AI
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Collection of helper methods for the Hunnt AI static cache subsystem.
 */
class Hunnt_AI_Static_Cache_Helper {
    public const CACHE_ROOT_SUBDIR = 'cache/hunnt-ai';
    public const PAGES_SUBDIR      = 'pages';
    public const META_SUBDIR       = 'meta';
    public const DROPIN_SIGNATURE  = 'Hunnt AI Static Cache Drop-In';
    public const CONFIG_FILENAME   = 'config.php';
    private const OPTION_LAST_ERROR = 'hunnt_ai_static_cache_fs_error';

    /**
     * Return the absolute path to the cache root directory.
     */
    public static function get_cache_root_dir(): string {
        $base = trailingslashit(WP_CONTENT_DIR) . self::CACHE_ROOT_SUBDIR;

        return wp_normalize_path($base);
    }

    /**
     * Ensure the cache directory structure exists.
     */
    public static function ensure_directories(): bool {
        $root = self::get_cache_root_dir();
        $paths = [
            $root,
            trailingslashit($root) . self::PAGES_SUBDIR,
            trailingslashit($root) . self::META_SUBDIR,
        ];

        $permission = (int) apply_filters('hunnt_ai_static_cache_directory_permission', 0755);
        $all_ok = true;

        foreach ($paths as $dir) {
            if (!is_dir($dir)) {
                wp_mkdir_p($dir);
            }

            if (!is_dir($dir)) {
                $all_ok = false;
                self::record_fs_issue($dir, 'mkdir_failed');
                continue;
            }

            if (!self::is_dir_writable($dir)) {
                @chmod($dir, $permission);
                clearstatcache(false, $dir);

                if (!self::is_dir_writable($dir)) {
                    $all_ok = false;
                    self::record_fs_issue($dir, 'not_writable');
                }
            }
        }

        if ($all_ok) {
            self::clear_last_fs_issue();
        }

        return $all_ok;
    }

    /**
     * Describe each directory used by the cache layer.
     */
    public static function get_directories_status(): array {
        $root = self::get_cache_root_dir();
        $paths = [
            'root'  => $root,
            'pages' => trailingslashit($root) . self::PAGES_SUBDIR,
            'meta'  => trailingslashit($root) . self::META_SUBDIR,
        ];

        $status = [];
        foreach ($paths as $key => $dir) {
            $exists = is_dir($dir);
            $writable = $exists ? self::is_dir_writable($dir) : self::is_parent_writable($dir);

            $status[$key] = [
                'path'     => wp_normalize_path($dir),
                'exists'   => $exists,
                'writable' => $writable,
            ];
        }

        return $status;
    }

    /**
     * Retrieve the last filesystem issue encountered by the cache helper.
     */
    public static function get_last_fs_issue(): ?array {
        $data = get_option(self::OPTION_LAST_ERROR, []);

        if (!is_array($data) || empty($data['code']) || empty($data['path'])) {
            return null;
        }

        return $data;
    }

    /**
     * Determine if a directory is writable using WordPress-safe checks.
     */
    private static function is_dir_writable(string $dir): bool {
        if (!is_dir($dir)) {
            return false;
        }

        if (function_exists('wp_is_writable')) {
            return wp_is_writable($dir);
        }

        return is_writable($dir);
    }

    /**
     * Determine if the parent directory of the supplied path is writable.
     */
    private static function is_parent_writable(string $dir): bool {
        $parent = dirname($dir);
        if ($parent === $dir) {
            return false;
        }

        if (function_exists('wp_is_writable')) {
            return wp_is_writable($parent);
        }

        return is_writable($parent);
    }

    /**
     * Persist the most recent filesystem issue for admin visibility.
     */
    private static function record_fs_issue(string $dir, string $code): void {
        $message = sprintf(
            /* translators: %s: directory path */
            __('Hunnt AI cannot write to %s. Please adjust filesystem permissions so the web server can create files in this directory.', 'hunnt-ai'),
            wp_normalize_path($dir)
        );

        $payload = [
            'path'      => wp_normalize_path($dir),
            'code'      => $code,
            'message'   => $message,
            'timestamp' => time(),
        ];

        update_option(self::OPTION_LAST_ERROR, $payload, false);
    }

    /**
     * Clear any stored filesystem issue state.
     */
    private static function clear_last_fs_issue(): void {
        delete_option(self::OPTION_LAST_ERROR);
    }

    /**
     * Return the allowed query-string whitelist.
     *
     * @return array|true
     */
    public static function get_query_whitelist() {
        $allowed = apply_filters('hunnt_ai_static_cache_allow_query_strings', []);

        if ($allowed === true) {
            return true;
        }

        if (!is_array($allowed)) {
            return [];
        }

        $normalized = [];
        foreach ($allowed as $item) {
            if (!is_string($item)) {
                continue;
            }
            $item = trim($item);
            if ($item === '') {
                continue;
            }
            $normalized[] = sanitize_key($item);
        }

        return array_values(array_unique(array_filter($normalized)));
    }

    /**
     * Retrieve cookie prefixes that should bypass static caching.
     */
    public static function get_cookie_bypass_patterns(): array {
        $patterns = apply_filters('hunnt_ai_static_cache_bypass_cookies', [
            'wordpress_logged_in',
            'comment_author',
            'woocommerce_cart_hash',
            'woocommerce_items_in_cart',
            'wp_woocommerce_session',
        ]);

        if (!is_array($patterns)) {
            return [];
        }

        return array_values(array_filter(array_map(static function ($pattern) {
            return is_string($pattern) ? trim($pattern) : '';
        }, $patterns)));
    }

    /**
     * Location of the metadata file for a given cache key.
     */
    public static function get_meta_path(string $key): string {
        $dir = trailingslashit(self::get_cache_root_dir()) . self::META_SUBDIR;

        return wp_normalize_path($dir . '/' . $key . '.php');
    }

    /**
     * Location of the HTML cache payload for a given cache key.
     */
    public static function get_page_path(string $key): string {
        $dir = trailingslashit(self::get_cache_root_dir()) . self::PAGES_SUBDIR;

        return wp_normalize_path($dir . '/' . $key . '.html');
    }

    /**
     * Return the gzip path for a given cache key.
     */
    public static function get_gzip_path(string $key): string {
        return self::get_page_path($key) . '.gz';
    }

    /**
     * Return the absolute path to the runtime config file.
     */
    public static function get_config_path(): string {
        return wp_normalize_path(trailingslashit(self::get_cache_root_dir()) . self::CONFIG_FILENAME);
    }

    /**
     * Compute the canonical cache key for a URL + context.
     */
    public static function build_cache_key(string $host, string $uri, string $variant = 'default'): string {
    return hunnt_ai_static_cache_build_key_raw($host, $uri, $variant);
    }

    /**
     * Determine the default TTL (seconds) for cached pages.
     */
    public static function get_default_ttl(): int {
        $ttl = (int) apply_filters('hunnt_ai_static_cache_default_ttl', 3600);

        return $ttl > 0 ? $ttl : 3600;
    }

    /**
     * Detect whether the current visitor should bypass caching.
     */
    public static function should_bypass_current_request(): bool {
        if (is_admin()) {
            return true;
        }

        if (PHP_SAPI === 'cli') {
            return true;
        }

        if (!in_array($_SERVER['REQUEST_METHOD'] ?? 'GET', ['GET', 'HEAD'], true)) {
            return true;
        }

        if (!empty($_GET) && !self::allow_query_string()) {
            return true;
        }

        if (self::has_auth_cookies()) {
            return true;
        }

    if (defined('DONOTCACHEPAGE') && constant('DONOTCACHEPAGE')) {
            return true;
        }

        if (apply_filters('hunnt_ai_static_cache_bypass', false)) {
            return true;
        }

        return false;
    }

    /**
     * Hydrate request variant (e.g., mobile/desktop).
     */
    public static function detect_variant(): string {
    $agent = $_SERVER['HTTP_USER_AGENT'] ?? '';

    $variant = hunnt_ai_static_cache_detect_variant_from_agent($agent);

    return $variant !== '' ? $variant : 'desktop';
    }

    /**
     * Determine if we allow query-string based caching.
     */
    private static function allow_query_string(): bool {
        $allowed = self::get_query_whitelist();

        if ($allowed === true) {
            return true;
        }

        if (!is_array($allowed) || empty($allowed)) {
            return empty($_GET);
        }

        foreach (array_keys($_GET) as $param) {
            if (!in_array($param, $allowed, true)) {
                return false;
            }
        }

        return true;
    }

    /**
     * Check for cookies that should bypass caching.
     */
    private static function has_auth_cookies(): bool {
        $patterns = self::get_cookie_bypass_patterns();

        return hunnt_ai_static_cache_cookie_matches($_COOKIE ?? [], (array) $patterns);
    }
}
