<?php
/**
 * Runtime Minify - Native Implementation
 * 
 * Uses native PHP optimization without external dependencies
 * Backward compatibility layer for existing code
 * 
 * @package Hunnt_AI
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Boot runtime minify system
 * Hooks into WordPress to replace assets with optimized versions
 */
function hunnt_ai_runtime_minify_boot(): void
{
    if (!hunnt_ai_is_technical_seo_enabled()) {
        return;
    }

    if (!get_option('hunnt_ai_assets_minify_enable')) {
        return;
    }

    if (is_admin() && !wp_doing_ajax()) {
        return;
    }

    // Assets will be replaced via hunnt_ai_replace_with_optimized_assets()
    // which is hooked in hunnt_ai_bootstrap_technical_seo()
}

add_action('plugins_loaded', 'hunnt_ai_runtime_minify_boot', 30);

/**
 * Warm cache - generate optimized assets
 * 
 * @return bool Success status
 */
function hunnt_ai_runtime_minify_warm_cache(): bool
{
    try {
        hunnt_ai_runtime_minify_prepare_environment();
        return hunnt_ai_apply_minify_assets();
    } catch (Throwable $exception) {
        error_log('Hunnt AI: Failed to warm runtime minify cache: ' . $exception->getMessage());
        return false;
    }
}

/**
 * Prepare environment for asset generation
 * Sets up necessary WordPress globals and hooks
 */
function hunnt_ai_runtime_minify_prepare_environment(): void
{
    if (php_sapi_name() === 'cli') {
        $homeUrl = function_exists('home_url') ? home_url('/') : null;
        if ($homeUrl) {
            $parsed = wp_parse_url($homeUrl);

            if (!isset($_SERVER['HTTP_HOST']) && isset($parsed['host'])) {
                $_SERVER['HTTP_HOST'] = $parsed['host'];
            }

            if (!isset($_SERVER['SERVER_NAME']) && isset($parsed['host'])) {
                $_SERVER['SERVER_NAME'] = $parsed['host'];
            }

            if (!isset($_SERVER['REQUEST_SCHEME']) && isset($parsed['scheme'])) {
                $_SERVER['REQUEST_SCHEME'] = $parsed['scheme'];
            }

            if (!isset($_SERVER['HTTPS']) && isset($parsed['scheme'])) {
                $_SERVER['HTTPS'] = $parsed['scheme'] === 'https' ? 'on' : 'off';
            }

            if (!isset($_SERVER['SERVER_PORT'])) {
                if (isset($parsed['port'])) {
                    $_SERVER['SERVER_PORT'] = (string) $parsed['port'];
                } else {
                    $_SERVER['SERVER_PORT'] = ($_SERVER['HTTPS'] ?? 'off') === 'on' ? '443' : '80';
                }
            }
        }

        if (!isset($_SERVER['REQUEST_URI'])) {
            $_SERVER['REQUEST_URI'] = '/';
        }
    }

    if (function_exists('wp') && !did_action('wp')) {
        wp();
    }

    if (function_exists('wp_styles')) {
        wp_styles();
    }

    if (function_exists('wp_scripts')) {
        wp_scripts();
    }

    $level = ob_get_level();
    ob_start();
    try {
        do_action('wp_enqueue_scripts');
    } finally {
        while (ob_get_level() > $level) {
            ob_end_clean();
        }
    }

    if (has_action('elementor/frontend/after_register_scripts')) {
        do_action('elementor/frontend/after_register_scripts');
    }

    if (has_action('elementor/frontend/after_enqueue_styles')) {
        do_action('elementor/frontend/after_enqueue_styles');
    }
}

/**
 * Check minification dependencies status
 * Native implementation doesn't require external dependencies
 * 
 * @return array Status information
 */
function hunnt_ai_minify_dependencies_status(): array
{
    $status = [
        'native_php' => true,
        'terser'    => false,
        'cleancss'  => false,
        'php_libs'  => false,
        'warnings'  => [],
    ];

    $status['warnings'][] = __('Using native PHP minification (no external dependencies required)', 'hunnt-ai');

    return $status;
}

/**
 * Generate minified assets (backward compatibility)
 * 
 * @return bool
 */
function hunnt_ai_generate_minified_assets(): bool
{
    return hunnt_ai_runtime_minify_warm_cache();
}

/**
 * Get minify manifest (backward compatibility)
 * 
 * @return array
 */
function hunnt_ai_minify_get_manifest(): array
{
    return hunnt_ai_get_minify_manifest();
}

/**
 * Reset manifest (backward compatibility)
 */
function hunnt_ai_minify_reset_manifest(): void
{
    $optimizer = new Hunnt_AI_Asset_Optimizer();
    $optimizer->clear_cache();
}
