<?php
/**
 * JS Processor - JavaScript minification using matthiasmullie/minify
 *
 * Clean and simple minification using the proven matthiasmullie/minify library
 *
 * @package Hunnt_AI
 */

if (!defined('ABSPATH')) {
    exit;
}

class Hunnt_AI_JS_Processor {
    
    /**
     * @var \MatthiasMullie\Minify\JS|null
     */
    private $minifier = null;

    /**
     * Minimum reduction ratio required to keep a minified asset.
     * Defaults to 15% and can be filtered via hunnt_ai_js_min_reduction_ratio.
     *
     * @var float
     */
    private $min_reduction_ratio = 0.0;

    /**
     * Whether .min.js sources should be passed through unchanged by default.
     *
     * @var bool
     */
    private $skip_minified_sources = true;

    /**
     * Cached list of handles that should always be passed through unchanged.
     *
     * @var array|null
     */
    private $passthrough_handles = null;
    
    /**
     * Initialize minifier
     */
    public function __construct() {
        $this->load_minifier();

        /**
         * Allow site owners to tweak the minimum reduction ratio required
         * to keep a minified output on disk.
         */
        $base_ratio = function_exists('hunnt_ai_get_assets_min_ratio')
            ? hunnt_ai_get_assets_min_ratio()
            : 0.15;

        $this->min_reduction_ratio = (float) apply_filters(
            'hunnt_ai_js_min_reduction_ratio',
            $base_ratio,
            null
        );

        /**
         * Allow toggling of the default behaviour for already minified assets.
         */
        $this->skip_minified_sources = (bool) apply_filters(
            'hunnt_ai_js_skip_minified_sources',
            $this->skip_minified_sources
        );
    }
    
    /**
     * Load matthiasmullie minifier
     */
    private function load_minifier() {
        $autoload = dirname(__DIR__, 2) . '/vendor/autoload.php';
        if (file_exists($autoload)) {
            require_once $autoload;
        }
        
        if (class_exists('\\MatthiasMullie\\Minify\\JS')) {
            $this->minifier = new \MatthiasMullie\Minify\JS();
        }
    }

    /**
     * Retrieve a list of handles that should bypass minification entirely.
     *
     * @return array<int, string>
     */
    private function get_passthrough_handles() {
        if ($this->passthrough_handles !== null) {
            return $this->passthrough_handles;
        }

        $handles = get_option('hunnt_ai_js_passthrough_handles', []);

        if (!is_array($handles)) {
            $handles = [];
        }

        $handles = array_filter(
            array_map(
                static function ($value) {
                    if (!is_string($value)) {
                        return null;
                    }

                    $value = sanitize_key(trim($value));
                    return $value !== '' ? $value : null;
                },
                $handles
            )
        );

        $handles = array_values(array_unique($handles));

        /**
         * Filter: hunnt_ai_js_passthrough_handles
         *
         * Allows third-parties to modify the list of handles that should always
         * bypass minification and be written as-is (or left untouched).
         *
         * @param array<int,string> $handles
         */
        $this->passthrough_handles = array_values(
            array_unique(
                (array) apply_filters('hunnt_ai_js_passthrough_handles', $handles)
            )
        );

        return $this->passthrough_handles;
    }

    /**
     * Check whether a given handle is configured to bypass minification.
     *
     * @param string $handle
     * @return bool
     */
    private function is_handle_passthrough($handle) {
        if (!is_string($handle) || $handle === '') {
            return false;
        }

        return in_array($handle, $this->get_passthrough_handles(), true);
    }

    /**
     * Evaluate how the current asset should be handled (minify, passthrough, skip).
     *
     * @param array  $asset
     * @param string $js_content
     * @return array{mode:string,reason:?string}
     */
    private function evaluate_asset_directive(array $asset, $js_content) {
        $handle = isset($asset['handle']) ? (string) $asset['handle'] : '';
        $src    = isset($asset['src']) ? (string) $asset['src'] : '';

        $filtered_skip = apply_filters('hunnt_ai_js_skip_asset', false, $asset, $js_content);
        if ($filtered_skip) {
            return [
                'mode'   => 'skip',
                'reason' => 'filtered_skip',
            ];
        }

        if ($this->is_es6_module($js_content)) {
            return [
                'mode'   => 'skip',
                'reason' => 'es6_module',
            ];
        }

        if ($this->is_handle_passthrough($handle)) {
            return [
                'mode'   => 'passthrough',
                'reason' => 'handle_passthrough',
            ];
        }

        $filtered_passthrough = apply_filters('hunnt_ai_js_force_passthrough_asset', false, $asset, $js_content);
        if ($filtered_passthrough) {
            return [
                'mode'   => 'passthrough',
                'reason' => 'filtered_passthrough',
            ];
        }

        if ($this->skip_minified_sources && $this->is_probably_minified_source($src, $js_content)) {
            return [
                'mode'   => 'passthrough',
                'reason' => 'already_minified',
            ];
        }

        return [
            'mode'   => 'minify',
            'reason' => null,
        ];
    }

    /**
     * Resolve the minimum reduction ratio for the provided asset.
     *
     * @param array $asset
     * @return float
     */
    private function get_min_reduction_ratio(array $asset) {
        $ratio = $this->min_reduction_ratio;

        $filtered_ratio = apply_filters('hunnt_ai_js_min_reduction_ratio', $ratio, $asset);

        if (is_numeric($filtered_ratio)) {
            $ratio = max(0.0, min(1.0, (float) $filtered_ratio));
        }

        return $ratio;
    }

    /**
     * Detect whether the asset source is already minified.
     *
     * @param string $src
     * @param string $js_content
     * @return bool
     */
    private function is_probably_minified_source($src, $js_content) {
        if (is_string($src) && preg_match('/\.min\.js($|\?)/i', $src)) {
            return true;
        }

        if (!is_string($js_content) || $js_content === '') {
            return false;
        }

        $line_breaks = substr_count($js_content, "\n");
        $length      = strlen($js_content);

        if ($length > 0 && ($line_breaks / max($length, 1)) < 0.0015) {
            return true;
        }

        return false;
    }
    
    /**
     * Process and combine JS files
     *
     * @param array $js_assets Array of JS asset information
     * @return string|false Combined JS content or false on failure
     */
    public function combine_js($js_assets) {
        if (empty($js_assets)) {
            return false;
        }

        $combined_js    = '';
        $processed_count = 0;
        $failed_files   = [];
        $debug_info     = [];
        $success_files  = [];

        foreach ($js_assets as $asset) {
            $file_path = $this->url_to_path($asset['src']);

            if (!$file_path) {
                $failed_files[] = $asset['handle'];
                $debug_info[] = [
                    'handle'     => $asset['handle'],
                    'src'        => $asset['src'],
                    'error'      => 'Could not convert URL to file path',
                    'error_code' => 'URL_CONVERSION_FAILED',
                ];
                continue;
            }

            if (!file_exists($file_path)) {
                $failed_files[] = $asset['handle'];
                $debug_info[] = [
                    'handle'     => $asset['handle'],
                    'src'        => $asset['src'],
                    'path'       => $file_path,
                    'error'      => 'File does not exist',
                    'error_code' => 'FILE_NOT_FOUND',
                ];
                continue;
            }

            if (!is_readable($file_path)) {
                $failed_files[] = $asset['handle'];
                $debug_info[] = [
                    'handle'     => $asset['handle'],
                    'src'        => $asset['src'],
                    'path'       => $file_path,
                    'error'      => 'File is not readable (permission denied)',
                    'error_code' => 'FILE_NOT_READABLE',
                ];
                continue;
            }

            $js_content = @file_get_contents($file_path);
            if ($js_content === false) {
                $failed_files[] = $asset['handle'];
                $debug_info[] = [
                    'handle'     => $asset['handle'],
                    'src'        => $asset['src'],
                    'path'       => $file_path,
                    'error'      => 'Could not read file content',
                    'error_code' => 'FILE_READ_FAILED',
                ];
                continue;
            }

            if (strlen(trim($js_content)) === 0) {
                $failed_files[] = $asset['handle'];
                $debug_info[] = [
                    'handle'     => $asset['handle'],
                    'src'        => $asset['src'],
                    'path'       => $file_path,
                    'error'      => 'File is empty',
                    'error_code' => 'EMPTY_FILE',
                    'file_size'  => filesize($file_path),
                ];
                continue;
            }

            // Skip ES6 module scripts - they cannot be bundled with traditional scripts
            // ES6 modules use import/export statements and must be loaded with type="module"
            if ($this->is_es6_module($js_content)) {
                $failed_files[] = $asset['handle'];
                $debug_info[] = [
                    'handle'     => $asset['handle'],
                    'src'        => $asset['src'],
                    'path'       => $file_path,
                    'error'      => 'ES6 module detected - cannot bundle (uses import/export)',
                    'error_code' => 'ES6_MODULE_DETECTED',
                    'file_size'  => filesize($file_path),
                ];
                continue;
            }

            $combined_js .= $js_content;
            $combined_js .= "\n;\n"; // ensure statement separation

            $processed_count++;
            $success_files[] = [
                'handle'         => $asset['handle'],
                'src'            => $asset['src'],
                'path'           => $file_path,
                'size'           => filesize($file_path),
                'content_length' => strlen($js_content),
            ];
        }

        // Log errors
        $upload_dir = wp_upload_dir();
        $log_file   = $upload_dir['basedir'] . '/hunnt-ai-optimized/js-errors.json';
        if (!file_exists(dirname($log_file))) {
            wp_mkdir_p(dirname($log_file));
        }

        file_put_contents(
            $log_file,
            wp_json_encode(
                [
                    'timestamp'                => current_time('mysql'),
                    'total_assets'             => count($js_assets),
                    'processed_count'          => $processed_count,
                    'failed_count'             => count($failed_files),
                    'combined_content_length'  => strlen($combined_js),
                    'success_files'            => $success_files,
                    'failed_files'             => $debug_info,
                ],
                JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES
            )
        );

        if ($processed_count === 0) {
            return false;
        }

        // Minify combined JS
        $minified_js = $this->minify_js($combined_js);
        
        if (strlen($minified_js) === 0 && strlen($combined_js) > 0) {
            error_log('Hunnt AI JS: Minification returned empty result, using original.');
            return $combined_js;
        }

        if (strlen($minified_js) >= strlen($combined_js)) {
            return $combined_js;
        }

        return $minified_js;
    }

    /**
     * Optimize each JS asset individually
     *
     * @param array  $js_assets  Array of collected JS assets
     * @param string $output_dir Destination directory for optimized files
     * @param string $output_url Public URL for the destination directory
     * @return array
     */
    public function optimize_assets(array $js_assets, $output_dir, $output_url) {
        if (empty($js_assets)) {
            return [
                'optimized' => [],
                'skipped'   => [],
            ];
        }

        if (!file_exists($output_dir)) {
            wp_mkdir_p($output_dir);
        }

        $optimized = [];
        $skipped   = [];
        $log_items = [];

        foreach ($js_assets as $asset) {
            $result = $this->optimize_single_asset($asset, $output_dir, $output_url);

            if ($result['success']) {
                $handle = $asset['handle'];
                $optimized[$handle] = $result['data'];
                $log_items[] = [
                    'handle'          => $handle,
                    'src'             => $asset['src'],
                    'optimized_path'  => $result['data']['optimized_path'],
                    'optimized_url'   => $result['data']['optimized_url'],
                    'hash'            => $result['data']['hash'],
                    'original_bytes'  => $result['data']['original_bytes'],
                    'optimized_bytes' => $result['data']['optimized_bytes'],
                    'saved_bytes'     => $result['data']['saved_bytes'],
                    'reduction_ratio' => $result['data']['reduction_ratio'],
                    'status'          => $result['data']['status'],
                    'warnings'        => $result['data']['warnings'],
                    'threshold_ratio' => $result['data']['threshold_ratio'],
                ];
            } else {
                $skipped[] = array_merge(
                    [
                        'handle' => $asset['handle'],
                        'src'    => $asset['src'],
                        'type'   => 'js',
                    ],
                    $result['error']
                );
            }
        }

        $this->write_js_log($js_assets, $optimized, $skipped, $log_items);

        return [
            'optimized' => $optimized,
            'skipped'   => $skipped,
        ];
    }

    /**
     * Aggregate optimized JS assets into bundles
     *
     * @param array  $optimized_js Optimized assets keyed by handle
     * @param string $output_dir   Destination directory for bundle files
     * @param string $output_url   Public URL for the destination directory
     * @return array
     */
    public function aggregate_bundles(array $optimized_js, $output_dir, $output_url) {
        if (empty($optimized_js)) {
            return [
                'bundles' => [],
                'skipped' => [],
            ];
        }

        if (!file_exists($output_dir)) {
            wp_mkdir_p($output_dir);
        }

        $groups = [];

        foreach ($optimized_js as $handle => $data) {
            $in_footer = !empty($data['in_footer']) ? 1 : 0;
            $bucket = $in_footer ? 'footer' : 'header';

            if (!isset($groups[$bucket])) {
                $groups[$bucket] = [
                    'in_footer' => (bool) $in_footer,
                    'assets'    => [],
                ];
            }

            $groups[$bucket]['assets'][] = array_merge($data, [
                'handle' => $handle,
            ]);
        }

        $bundles = [];
        $skipped = [];

        foreach ($groups as $bucket => $group) {
            $bundle_handle = sprintf('hunnt-ai-js-bundle-%s', $bucket);
            $bundle_content = '';
            $original_bytes = 0;
            $intermediate_bytes = 0;
            $bundle_saved_bytes = 0;
            
            // Collect all unique dependencies from bundled scripts
            $bundle_deps = [];
            $bundle_handles = [];
            $bundle_warnings = [];
            $bundle_statuses = [];
            $translations = [];

            foreach ($group['assets'] as $asset) {
                $path = $asset['optimized_path'] ?? '';

                if (!is_string($path) || $path === '' || !file_exists($path)) {
                    $skipped[] = [
                        'handle' => $asset['handle'],
                        'type'   => 'js',
                        'reason' => 'bundle_source_missing',
                        'path'   => $path,
                        'bucket' => $bucket,
                    ];
                    continue;
                }

                $chunk = @file_get_contents($path);

                if ($chunk === false) {
                    $skipped[] = [
                        'handle' => $asset['handle'],
                        'type'   => 'js',
                        'reason' => 'bundle_source_unreadable',
                        'path'   => $path,
                        'bucket' => $bucket,
                    ];
                    continue;
                }
                
                // Collect dependencies from this asset
                if (!empty($asset['deps']) && is_array($asset['deps'])) {
                    foreach ($asset['deps'] as $dep) {
                        if (!is_string($dep) || $dep === '') {
                            continue;
                        }

                        $bundle_deps[$dep] = true;
                    }
                }

                // Add inline before
                if (!empty($asset['inline_before']) && is_array($asset['inline_before'])) {
                    foreach ($asset['inline_before'] as $inline_snippet) {
                        $inline_snippet = trim((string) $inline_snippet);
                        if ($inline_snippet !== '') {
                            $bundle_content .= $inline_snippet . ";\n";
                        }
                    }
                }

                // Add inline data
                if (!empty($asset['inline_data'])) {
                    $inline_data = trim((string) $asset['inline_data']);
                    if ($inline_data !== '') {
                        $bundle_content .= $inline_data . ";\n";
                    }
                }

                $bundle_content .= $chunk;
                $bundle_content .= ";\n";

                // Add inline after
                if (!empty($asset['inline_after']) && is_array($asset['inline_after'])) {
                    foreach ($asset['inline_after'] as $inline_snippet) {
                        $inline_snippet = trim((string) $inline_snippet);
                        if ($inline_snippet !== '') {
                            $bundle_content .= $inline_snippet . ";\n";
                        }
                    }
                }

                $original_bytes += (int) ($asset['original_bytes'] ?? strlen($chunk));
                $intermediate_bytes += strlen($chunk);
                $bundle_saved_bytes += (int) ($asset['saved_bytes'] ?? 0);

                $bundle_handles[] = $asset['handle'];

                if (!empty($asset['warnings'])) {
                    $bundle_warnings = array_merge($bundle_warnings, (array) $asset['warnings']);
                }

                if (!empty($asset['status'])) {
                    $bundle_statuses[] = (string) $asset['status'];
                }

                if (!empty($asset['textdomain'])) {
                    $translation_key = $asset['textdomain'] . '|' . ($asset['translations_path'] ?? '');
                    $translations[$translation_key] = [
                        'textdomain' => (string) $asset['textdomain'],
                        'path'       => isset($asset['translations_path']) && is_string($asset['translations_path'])
                            ? $asset['translations_path']
                            : '',
                    ];
                }
            }
            
            $bundle_handles = array_values(array_unique($bundle_handles));
            if (empty($bundle_handles)) {
                continue;
            }

            $bundle_handles_map = array_fill_keys($bundle_handles, true);

            $all_deps = [];
            if (!empty($bundle_deps)) {
                foreach (array_keys($bundle_deps) as $dep) {
                    if (isset($bundle_handles_map[$dep])) {
                        continue; // Dependency satisfied within the bundle
                    }

                    $all_deps[] = $dep;
                }
            }

            // Remove duplicates and ensure jQuery is first when required
            $all_deps = array_values(array_unique($all_deps));
            if (in_array('jquery', $all_deps, true)) {
                $all_deps = array_values(array_diff($all_deps, ['jquery']));
                array_unshift($all_deps, 'jquery');
            }

            $bundle_content = trim($bundle_content);

            // Minify bundle
            $pre_minify_length = strlen($bundle_content);
            $minified_bundle = $this->minify_js($bundle_content);
            if (is_string($minified_bundle) && $minified_bundle !== '' && strlen($minified_bundle) <= $pre_minify_length) {
                $bundle_content = $minified_bundle;
            }

            $optimized_bytes = strlen($bundle_content);

            $bundle_reduction_ratio = $original_bytes > 0
                ? max(0, 1 - ($optimized_bytes / $original_bytes))
                : 0;

            $bundle_threshold = apply_filters(
                'hunnt_ai_js_bundle_min_reduction_ratio',
                $this->min_reduction_ratio,
                $bundle_handle,
                $group,
                $bundle_reduction_ratio
            );

            if (!is_numeric($bundle_threshold)) {
                $bundle_threshold = $this->min_reduction_ratio;
            }

            $bundle_threshold = max(0.0, min(1.0, (float) $bundle_threshold));

            if ($bundle_reduction_ratio < $bundle_threshold) {
                $allow_bundle = apply_filters(
                    'hunnt_ai_js_bundle_allow_below_threshold',
                    false,
                    $bundle_handle,
                    $bundle_reduction_ratio,
                    $bundle_threshold,
                    $group
                );

                if ($allow_bundle) {
                    $bundle_warnings[] = 'bundle_below_threshold_allowed';
                } else {
                    $retain_bundle = apply_filters(
                        'hunnt_ai_js_bundle_retain_low_gain',
                        true,
                        $bundle_handle,
                        $bundle_reduction_ratio,
                        $bundle_threshold,
                        $group
                    );

                    if ($retain_bundle && $bundle_reduction_ratio > 0) {
                        $bundle_warnings[] = 'bundle_below_threshold_retained';
                        $bundle_statuses[] = 'bundle_low_gain';
                    } else {
                        $skipped[] = [
                            'handle'                    => $bundle_handle,
                            'type'                      => 'js',
                            'reason'                    => 'bundle_below_threshold',
                            'bucket'                    => $bucket,
                            'bundle_reduction_ratio'    => $bundle_reduction_ratio,
                            'bundle_threshold_ratio'    => $bundle_threshold,
                        ];
                        continue;
                    }
                }
            }

            if ($bundle_content === '') {
                continue;
            }

            $hash = $this->generate_hash($bundle_content . $bucket);
            $filename  = sprintf('%s-%s.js', $bundle_handle, $hash);
            $file_out  = $output_dir . '/' . $filename;

            if (@file_put_contents($file_out, $bundle_content) === false) {
                $skipped[] = [
                    'type'   => 'js',
                    'reason' => 'bundle_write_failed',
                    'bucket' => $bucket,
                ];
                continue;
            }

            $bundles[$bundle_handle] = [
                'handle'          => $bundle_handle,
                'optimized_path'  => $file_out,
                'optimized_url'   => $output_url . '/' . $filename,
                'hash'            => $hash,
                'in_footer'       => $group['in_footer'],
                'deps'            => array_values($all_deps),
                'original_handles' => $bundle_handles,
                'original_bytes'  => $original_bytes,
                'optimized_bytes'  => $optimized_bytes,
                'intermediate_bytes' => $intermediate_bytes,
                'saved_bytes'      => max(0, $original_bytes - $optimized_bytes),
                'bundle_saved_bytes' => $bundle_saved_bytes,
                'reduction_ratio'  => $bundle_reduction_ratio,
                'threshold_ratio'  => $bundle_threshold,
                'warnings'        => array_values(array_unique($bundle_warnings)),
                'statuses'        => array_values(array_unique($bundle_statuses)),
                'translations'    => array_values($translations),
            ];
        }

        return [
            'bundles' => $bundles,
            'skipped' => $skipped,
        ];
    }

    /**
     * Optimize a single JS asset
     *
     * @param array  $asset
     * @param string $output_dir
     * @param string $output_url
     * @return array
     */
    private function optimize_single_asset(array $asset, $output_dir, $output_url) {
        $file_path = $this->url_to_path($asset['src']);
        
        if (!$file_path || !file_exists($file_path) || !is_readable($file_path)) {
            return [
                'success' => false,
                'error'   => [
                    'error_code' => 'FILE_NOT_ACCESSIBLE',
                    'message'    => 'File not accessible',
                ],
            ];
        }

        $js_content = @file_get_contents($file_path);
        if ($js_content === false || strlen(trim($js_content)) === 0) {
            return [
                'success' => false,
                'error'   => [
                    'error_code' => 'EMPTY_FILE',
                    'message'    => 'File is empty',
                ],
            ];
        }

        $directive = $this->evaluate_asset_directive($asset, $js_content);

        if ($directive['mode'] === 'skip') {
            $reason = isset($directive['reason']) ? $directive['reason'] : 'skip';

            return [
                'success' => false,
                'error'   => [
                    'error_code' => strtoupper($reason),
                    'message'    => sprintf('Asset skipped (%s)', $reason),
                ],
            ];
        }

        $original_length = strlen($js_content);
        $optimized_content = $js_content;
        $status = 'passthrough';
        $warnings = [];
        $threshold = $this->get_min_reduction_ratio($asset);

        if ($directive['mode'] === 'minify') {
            $status = 'minified';
            $optimized_content = $this->minify_js($js_content);

            if ($optimized_content === '' && $original_length > 0) {
                $warnings[] = 'minify_empty_result';
                $optimized_content = $js_content;
                $status = 'fallback_original';
            }
        } else {
            if (!empty($directive['reason'])) {
                $warnings[] = 'passthrough_' . $directive['reason'];
            }
        }

        $optimized_length = strlen($optimized_content);
        $reduction_ratio = $original_length > 0
            ? max(0, 1 - ($optimized_length / $original_length))
            : 0;

        if ($status === 'minified') {
            if ($optimized_length === 0 || $optimized_length >= $original_length) {
                $warnings[] = 'not_reduced';
                $optimized_content = $js_content;
                $optimized_length = $original_length;
                $reduction_ratio = 0;
                $status = 'passthrough_not_smaller';
            } elseif ($reduction_ratio < $threshold) {
                $allow_below = apply_filters(
                    'hunnt_ai_js_allow_below_threshold',
                    false,
                    $asset,
                    $reduction_ratio,
                    $threshold
                );

                if ($allow_below) {
                    $warnings[] = 'below_threshold_allowed';
                } else {
                    $retain_low_gain = apply_filters(
                        'hunnt_ai_js_retain_low_gain_minification',
                        true,
                        $asset,
                        $reduction_ratio,
                        $threshold
                    );

                    if ($retain_low_gain && $reduction_ratio > 0) {
                        $warnings[] = 'below_threshold_retained';
                        $status = 'minified_low_gain';
                    } else {
                        $warnings[] = 'below_threshold';
                        $optimized_content = $js_content;
                        $optimized_length = $original_length;
                        $reduction_ratio = 0;
                        $status = 'passthrough_threshold';
                    }
                }
            }
        }

        $optimized_content = apply_filters(
            'hunnt_ai_js_optimized_content',
            $optimized_content,
            $asset,
            $status,
            $reduction_ratio
        );

        $optimized_length = strlen($optimized_content);
        $hash = $this->generate_hash($optimized_content . '|' . $status);
        $filename = sprintf('hunnt-ai-js-%s-%s.js', $asset['handle'], $hash);
        $file_out = $output_dir . '/' . $filename;

        if (@file_put_contents($file_out, $optimized_content) === false) {
            return [
                'success' => false,
                'error'   => [
                    'error_code' => 'FILE_WRITE_FAILED',
                    'message'    => 'Could not write optimized file',
                ],
            ];
        }

        return [
            'success' => true,
            'data'    => [
                'handle'            => $asset['handle'],
                'source'            => $asset['src'],
                'optimized_path'    => $file_out,
                'optimized_url'     => $output_url . '/' . $filename,
                'hash'              => $hash,
                'deps'              => $asset['deps'] ?? [],
                'ver'               => $asset['ver'] ?? false,
                'in_footer'         => $asset['in_footer'] ?? 0,
                'original_bytes'    => $original_length,
                'optimized_bytes'   => $optimized_length,
                'saved_bytes'       => max(0, $original_length - $optimized_length),
                'reduction_ratio'   => $reduction_ratio,
                'threshold_ratio'   => $threshold,
                'status'            => $status,
                'warnings'          => array_values(array_unique($warnings)),
                'inline_before'     => $asset['inline_before'] ?? [],
                'inline_after'      => $asset['inline_after'] ?? [],
                'inline_data'       => $asset['inline_data'] ?? '',
                'textdomain'        => $asset['textdomain'] ?? null,
                'translations_path' => $asset['translations_path'] ?? null,
            ],
        ];
    }

    /**
     * Minify JavaScript using matthiasmullie/minify
     *
     * @param string $js
     * @return string
     */
    private function minify_js($js) {
        if (!is_string($js) || $js === '') {
            return '';
        }

        if ($this->minifier === null) {
            return $js;
        }

        try {
            $minifier = new \MatthiasMullie\Minify\JS($js);
            $minified = $minifier->minify();
            
            return is_string($minified) && $minified !== '' ? $minified : $js;
        } catch (\Exception $e) {
            error_log('Hunnt AI JS: Minification failed - ' . $e->getMessage());
            return $js;
        }
    }

    /**
     * Write JS optimization log
     *
     * @param array $js_assets
     * @param array $optimized
     * @param array $skipped
     * @param array $log_items
     */
    private function write_js_log(array $js_assets, array $optimized, array $skipped, array $log_items) {
        $upload_dir = wp_upload_dir();
        $log_file = $upload_dir['basedir'] . '/hunnt-ai-optimized/js-errors.json';

        file_put_contents(
            $log_file,
            wp_json_encode(
                [
                    'timestamp'      => current_time('mysql'),
                    'total_assets'   => count($js_assets),
                    'processed_count' => count($optimized),
                    'skipped_count'  => count($skipped),
                    'optimized'      => $log_items,
                    'skipped'        => $skipped,
                ],
                JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES
            )
        );
    }

    /**
     * Convert URL to file path
     *
     * @param string $url
     * @return string|null
     */
    private function url_to_path($url) {
        if (empty($url)) {
            return null;
        }

        $site_url = site_url('/');
        $home_url = home_url('/');
        
        $url = str_replace(['https://', 'http://'], '', $url);
        $site_url = str_replace(['https://', 'http://'], '', $site_url);
        $home_url = str_replace(['https://', 'http://'], '', $home_url);

        if (strpos($url, $site_url) === 0) {
            $relative = substr($url, strlen($site_url));
            $path = ABSPATH . ltrim($relative, '/');
            return file_exists($path) ? wp_normalize_path($path) : null;
        }

        if ($home_url !== $site_url && strpos($url, $home_url) === 0) {
            $relative = substr($url, strlen($home_url));
            $path = ABSPATH . ltrim($relative, '/');
            return file_exists($path) ? wp_normalize_path($path) : null;
        }

        return null;
    }

    /**
     * Detect if JavaScript content is an ES6 module
     * 
     * ES6 modules use import/export statements and cannot be concatenated
     * with traditional scripts as they require type="module" attribute
     *
     * @param string $content JavaScript file content
     * @return bool True if ES6 module detected
     */
    private function is_es6_module($content) {
        // Remove comments to avoid false positives
        $content_no_comments = preg_replace([
            '#/\*.*?\*/#s',    // Multi-line comments
            '#//.*$#m'          // Single-line comments
        ], '', $content);

        // Check for ES6 import/export statements at the start of lines
        // These patterns match real module syntax, not strings or comments
        $es6_patterns = [
            '/^\s*import\s+/m',                    // import something
            '/^\s*export\s+/m',                    // export something
            '/^\s*import\s*\{/m',                  // import { x }
            '/^\s*export\s+default\s+/m',         // export default
            '/^\s*export\s*\{/m',                  // export { x }
            '/\bimport\s*\([\'"].*[\'"]\)/m'      // dynamic import()
        ];

        foreach ($es6_patterns as $pattern) {
            if (preg_match($pattern, $content_no_comments)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Generate hash for cache busting
     *
     * @param string $content
     * @return string
     */
    public function generate_hash($content) {
        return substr(md5($content), 0, 12);
    }
}
