<?php
/**
 * CSS Processor - Native CSS combination, path resolution, and minification
 * 
 * Handles:
 * - CSS file combination
 * - Relative path resolution (url(), @import)
 * - Background images, fonts, and other asset URLs
 * - Basic CSS minification without external dependencies
 * 
 * @package Hunnt_AI
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Hunnt_AI_CSS_Processor {
    /**
     * Tracks Font Awesome canonical stylesheet and recorded font-face hashes
     * to avoid duplicate heavy payloads while preserving font fallbacks.
     *
     * @var string|null
     */
    private static $fontawesome_primary_handle = null;

    /**
     * @var array<string, true>
     */
    private static $fontawesome_font_face_hashes = [];

    /**
     * Tracks extracted font files keyed by content hash to avoid duplicate disk writes.
     *
     * @var array<string, string>
     */
    private static $extracted_font_map = [];

    /**
     * Minimum reduction ratio required to keep a minified CSS asset.
     *
     * @var float
     */
    private $min_reduction_ratio = 0.0;

    /**
     * Whether .min.css sources should be passed through unchanged by default.
     *
     * @var bool
     */
    private $skip_minified_sources = true;

    /**
     * Cached list of handles that should bypass minification.
     *
     * @var array|null
     */
    private $passthrough_handles = null;

    public function __construct() {
        $base_ratio = function_exists('hunnt_ai_get_assets_min_ratio')
            ? hunnt_ai_get_assets_min_ratio()
            : 0.15;

        $this->min_reduction_ratio = (float) apply_filters(
            'hunnt_ai_css_min_reduction_ratio',
            $base_ratio,
            null
        );

        $this->skip_minified_sources = (bool) apply_filters(
            'hunnt_ai_css_skip_minified_sources',
            $this->skip_minified_sources
        );
    }

    /**
     * Retrieve handles that should bypass CSS minification.
     *
     * @return array<int, string>
     */
    private function get_passthrough_handles() {
        if ($this->passthrough_handles !== null) {
            return $this->passthrough_handles;
        }

        $handles = get_option('hunnt_ai_css_passthrough_handles', []);

        if (!is_array($handles)) {
            $handles = [];
        }

        $handles = array_filter(
            array_map(
                static function ($value) {
                    if (!is_string($value)) {
                        return null;
                    }

                    $value = sanitize_key(trim($value));
                    return $value !== '' ? $value : null;
                },
                $handles
            )
        );

        $handles = array_values(array_unique($handles));

        $this->passthrough_handles = array_values(
            array_unique(
                (array) apply_filters('hunnt_ai_css_passthrough_handles', $handles)
            )
        );

        return $this->passthrough_handles;
    }

    /**
     * Determine if the provided handle should bypass minification.
     *
     * @param string $handle
     * @return bool
     */
    private function is_handle_passthrough($handle) {
        if (!is_string($handle) || $handle === '') {
            return false;
        }

        return in_array($handle, $this->get_passthrough_handles(), true);
    }

    /**
     * Evaluate directive for how to treat the CSS asset.
     *
     * @param array  $asset
     * @param string $css_content
     * @return array{mode:string,reason:?string}
     */
    private function evaluate_asset_directive(array $asset, $css_content) {
        $handle = isset($asset['handle']) ? (string) $asset['handle'] : '';
        $src    = isset($asset['src']) ? (string) $asset['src'] : '';

        if (apply_filters('hunnt_ai_css_skip_asset', false, $asset, $css_content)) {
            return [ 'mode' => 'skip', 'reason' => 'filtered_skip' ];
        }

        if ($this->is_handle_passthrough($handle)) {
            return [ 'mode' => 'passthrough', 'reason' => 'handle_passthrough' ];
        }

        if (apply_filters('hunnt_ai_css_force_passthrough_asset', false, $asset, $css_content)) {
            return [ 'mode' => 'passthrough', 'reason' => 'filtered_passthrough' ];
        }

        if ($this->skip_minified_sources && $this->is_probably_minified_source($src, $css_content)) {
            return [ 'mode' => 'passthrough', 'reason' => 'already_minified' ];
        }

        return [ 'mode' => 'minify', 'reason' => null ];
    }

    /**
     * Best-effort heuristic to detect already minified CSS sources.
     *
     * @param string $src
     * @param string $css_content
     * @return bool
     */
    private function is_probably_minified_source($src, $css_content) {
        if (is_string($src) && preg_match('/\.min\.css($|\?)/i', $src)) {
            return true;
        }

        if (!is_string($css_content) || $css_content === '') {
            return false;
        }

        $line_breaks = substr_count($css_content, "\n");
        $length      = strlen($css_content);

        if ($length > 0 && ($line_breaks / max($length, 1)) < 0.0025) {
            return true;
        }

        return false;
    }

    /**
     * Resolve minimum reduction ratio per asset.
     *
     * @param array $asset
     * @return float
     */
    private function get_min_reduction_ratio(array $asset) {
        $ratio = $this->min_reduction_ratio;
        $filtered = apply_filters('hunnt_ai_css_min_reduction_ratio', $ratio, $asset);

        if (is_numeric($filtered)) {
            $ratio = max(0.0, min(1.0, (float) $filtered));
        }

        return $ratio;
    }
    
    /**
     * Process and combine CSS files
     * Production-grade file processing with comprehensive error tracking
     * 
     * @param array $css_assets Array of CSS asset information
     * @return string|false Combined CSS content or false on failure
     */
    public function combine_css($css_assets) {
        if (empty($css_assets)) {
            return false;
        }
        
        $combined_css = '';
        $processed_count = 0;
        $failed_files = [];
        $debug_info = [];
        $success_files = [];
        
        foreach ($css_assets as $asset) {
            $file_path = $this->url_to_path($asset['src']);
            
            if (!$file_path) {
                $failed_files[] = $asset['handle'];
                $debug_info[] = [
                    'handle' => $asset['handle'],
                    'src' => $asset['src'],
                    'error' => 'Could not convert URL to file path',
                    'error_code' => 'URL_CONVERSION_FAILED'
                ];
                continue;
            }
            
            if (!file_exists($file_path)) {
                $failed_files[] = $asset['handle'];
                $debug_info[] = [
                    'handle' => $asset['handle'],
                    'src' => $asset['src'],
                    'path' => $file_path,
                    'error' => 'File does not exist',
                    'error_code' => 'FILE_NOT_FOUND'
                ];
                continue;
            }
            
            if (!is_readable($file_path)) {
                $failed_files[] = $asset['handle'];
                $debug_info[] = [
                    'handle' => $asset['handle'],
                    'src' => $asset['src'],
                    'path' => $file_path,
                    'error' => 'File is not readable (permission denied)',
                    'error_code' => 'FILE_NOT_READABLE'
                ];
                continue;
            }
            
            $css_content = @file_get_contents($file_path);
            
            if ($css_content === false) {
                $failed_files[] = $asset['handle'];
                $debug_info[] = [
                    'handle' => $asset['handle'],
                    'src' => $asset['src'],
                    'path' => $file_path,
                    'error' => 'Could not read file content',
                    'error_code' => 'FILE_READ_FAILED'
                ];
                continue;
            }
            
            // Check if file is empty
            if (strlen(trim($css_content)) === 0) {
                $failed_files[] = $asset['handle'];
                $debug_info[] = [
                    'handle' => $asset['handle'],
                    'src' => $asset['src'],
                    'path' => $file_path,
                    'error' => 'File is empty',
                    'error_code' => 'EMPTY_FILE',
                    'file_size' => filesize($file_path)
                ];
                continue;
            }
            
        // Strip any HTML tags that might be in the CSS file
        // Some WordPress inline styles include <style> tags
        $css_content = preg_replace('/<style[^>]*>|<\/style>/i', '', $css_content);
        $css_content = preg_replace('/<\/?[a-z][^>]*>/i', '', $css_content);

        // Check bracket balance and fix if needed
        $open_count = substr_count($css_content, '{');
        $close_count = substr_count($css_content, '}');
        
        if ($open_count !== $close_count) {
            $diff = $close_count - $open_count;
            if ($diff > 0) {
                // More closing than opening - remove extra closing brackets from the end
                $css_content = preg_replace('/\}\s*$/', '', $css_content, $diff);
                error_log("[CSS Processor] Fixed {$asset['handle']}: Removed $diff extra closing bracket(s)");
            } elseif ($diff < 0) {
                // More opening than closing - add closing brackets at the end
                $css_content .= str_repeat('}', abs($diff));
                error_log("[CSS Processor] Fixed {$asset['handle']}: Added " . abs($diff) . " missing closing bracket(s)");
            }
        }

        // Resolve relative paths to absolute URLs
        $css_content = $this->resolve_css_paths($css_content, $asset['src']);

        if ($this->should_include_source_comment('css_combined')) {
            $combined_css .= "\n/* Source: {$asset['handle']} - {$asset['src']} */\n";
        }

        $combined_css .= $css_content . "\n";
            
            $processed_count++;
            $success_files[] = [
                'handle' => $asset['handle'],
                'src' => $asset['src'],
                'path' => $file_path,
                'size' => filesize($file_path),
                'content_length' => strlen($css_content)
            ];
        }
        
        // Save detailed error log
        $upload_dir = wp_upload_dir();
        $log_file = $upload_dir['basedir'] . '/hunnt-ai-optimized/css-errors.json';
        if (!file_exists(dirname($log_file))) {
            wp_mkdir_p(dirname($log_file));
        }
        
        file_put_contents($log_file, wp_json_encode([
            'timestamp' => current_time('mysql'),
            'total_assets' => count($css_assets),
            'processed_count' => $processed_count,
            'failed_count' => count($failed_files),
            'combined_content_length' => strlen($combined_css),
            'success_files' => $success_files,
            'failed_files' => $debug_info
        ], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
        
        if ($processed_count === 0) {
            return false;
        }
        
        // Log any failed files
        if (!empty($failed_files) && defined('WP_DEBUG') && WP_DEBUG) {
            error_log('Hunnt AI CSS: Failed to process files: ' . implode(', ', $failed_files));
        }
        
        // Debug: Log combined CSS length before minification
        $combined_length_before = strlen($combined_css);
        
    // Basic minification
    $minified_css = $this->minify_css($combined_css);

    // Debug: Log after minification
    $minified_length = strlen($minified_css);
        
        // If minification returns empty but we had content, return original
        if ($minified_length === 0 && $combined_length_before > 0) {
            error_log("Hunnt AI CSS: Minification returned empty result, using original. Before: $combined_length_before bytes");
            return $combined_css; // Return un-minified version
        }

        if ($minified_length >= $combined_length_before) {
            return $combined_css;
        }

        return $minified_css;
    }

    /**
     * Optimize each CSS asset individually and write minified copies
     *
     * @param array  $css_assets Array of collected CSS assets
     * @param string $output_dir Destination directory for optimized files
     * @param string $output_url Public URL for the destination directory
     * @return array{optimized: array<string, array<string, mixed>>, skipped: array<int, array<string, mixed>>}
     */
    public function optimize_assets(array $css_assets, $output_dir, $output_url) {
        if (empty($css_assets)) {
            return [
                'optimized' => [],
                'skipped'   => [],
            ];
        }

        if (!file_exists($output_dir)) {
            wp_mkdir_p($output_dir);
        }

        $optimized = [];
        $skipped   = [];
        $log_items = [];

        foreach ($css_assets as $asset) {
            $result = $this->optimize_single_asset($asset, $output_dir, $output_url);

            if ($result['success']) {
                $handle = $asset['handle'];
                $optimized[$handle] = $result['data'];
                $log_items[] = [
                    'handle'          => $handle,
                    'src'             => $asset['src'],
                    'optimized_path'  => $result['data']['optimized_path'],
                    'optimized_url'   => $result['data']['optimized_url'],
                    'hash'            => $result['data']['hash'],
                    'original_bytes'  => $result['data']['original_bytes'],
                    'optimized_bytes' => $result['data']['optimized_bytes'],
                    'saved_bytes'     => $result['data']['saved_bytes'],
                    'reduction_ratio' => $result['data']['reduction_ratio'],
                    'status'          => $result['data']['status'],
                    'warnings'        => $result['data']['warnings'],
                    'threshold_ratio' => $result['data']['threshold_ratio'],
                ];
            } else {
                $skipped[] = array_merge([
                    'handle' => $asset['handle'],
                    'src'    => $asset['src'],
                    'type'   => 'css',
                ], $result['error']);
            }
        }

        $this->write_css_log($css_assets, $optimized, $skipped, $log_items);

        return [
            'optimized' => $optimized,
            'skipped'   => $skipped,
        ];
    }

    /**
     * Aggregate optimized CSS assets into consolidated bundles grouped by media type.
     *
     * @param array  $optimized_css Optimized assets keyed by handle (output of optimize_assets)
     * @param string $output_dir    Destination directory for bundle files
     * @param string $output_url    Public URL for the destination directory
     * @return array{bundles: array<string, array<string, mixed>>, skipped: array<int, array<string,mixed>>}
     */
    public function aggregate_bundles(array $optimized_css, $output_dir, $output_url) {
        if (empty($optimized_css)) {
            return [
                'bundles' => [],
                'skipped' => [],
            ];
        }

        if (!file_exists($output_dir)) {
            wp_mkdir_p($output_dir);
        }

        $groups = [];

        foreach ($optimized_css as $handle => $data) {
            $media = isset($data['media']) && $data['media'] ? (string) $data['media'] : 'all';
            $bucket = sanitize_key($media !== '' ? $media : 'all');
            if ($bucket === '') {
                $bucket = 'all';
            }

            if (!isset($groups[$bucket])) {
                $groups[$bucket] = [
                    'media'   => $media,
                    'assets'  => [],
                ];
            }

            $groups[$bucket]['assets'][] = array_merge($data, [
                'handle' => $handle,
            ]);
        }

        $bundles = [];
        $skipped = [];

        foreach ($groups as $bucket => $group) {
            $bundle_handle = sprintf('hunnt-ai-css-bundle-%s', $bucket);
            $bundle_content = '';
            $original_bytes = 0;
            $intermediate_bytes = 0;
            $bundle_saved_bytes = 0;
            $bundle_warnings = [];
            $bundle_statuses = [];
            $bundle_handles = [];
            $bundle_deps = [];

            foreach ($group['assets'] as $asset) {
                $path = $asset['optimized_path'] ?? '';

                if (!is_string($path) || $path === '' || !file_exists($path)) {
                    $skipped[] = [
                        'handle' => $asset['handle'],
                        'type'   => 'css',
                        'reason' => 'bundle_source_missing',
                        'path'   => $path,
                        'bucket' => $bucket,
                    ];
                    continue;
                }

                $chunk = @file_get_contents($path);

                if ($chunk === false) {
                    $skipped[] = [
                        'handle' => $asset['handle'],
                        'type'   => 'css',
                        'reason' => 'bundle_source_unreadable',
                        'path'   => $path,
                        'bucket' => $bucket,
                    ];
                    continue;
                }

                if ($this->should_include_source_comment('css_bundle')) {
                    $bundle_content .= "\n/* Source: {$asset['handle']} */\n";
                }

                $bundle_content .= $chunk . "\n";

                if (!empty($asset['inline_after']) && is_array($asset['inline_after'])) {
                    foreach ($asset['inline_after'] as $inline_css) {
                        $inline_css = trim((string) $inline_css);
                        if ($inline_css === '') {
                            continue;
                        }
                        $bundle_content .= $inline_css . "\n";
                    }
                }

                $original_bytes += (int) ($asset['original_bytes'] ?? strlen($chunk));
                $intermediate_bytes += strlen($chunk);
                $bundle_saved_bytes += (int) ($asset['saved_bytes'] ?? 0);

                $bundle_handles[] = $asset['handle'];

                if (!empty($asset['warnings'])) {
                    $bundle_warnings = array_merge($bundle_warnings, (array) $asset['warnings']);
                }

                if (!empty($asset['status'])) {
                    $bundle_statuses[] = (string) $asset['status'];
                }

                if (!empty($asset['deps']) && is_array($asset['deps'])) {
                    foreach ($asset['deps'] as $dep) {
                        if (!is_string($dep) || $dep === '') {
                            continue;
                        }
                        $bundle_deps[$dep] = true;
                    }
                }
            }

            $bundle_content = trim($bundle_content);

            if ($bundle_content === '' || empty($bundle_handles)) {
                continue;
            }

            $bundle_content = $this->finalize_bundle_css($bundle_content);

            if ($bundle_content === '') {
                continue;
            }

            $optimized_bytes = strlen($bundle_content);

            $bundle_handles_map = array_fill_keys($bundle_handles, true);
            $deps = [];

            if (!empty($bundle_deps)) {
                foreach (array_keys($bundle_deps) as $dep) {
                    if (isset($bundle_handles_map[$dep])) {
                        continue; // Dependency now satisfied inside the bundle
                    }

                    $deps[] = $dep;
                }
            }

            $bundle_reduction_ratio = $original_bytes > 0
                ? max(0, 1 - ($optimized_bytes / $original_bytes))
                : 0;

            $bundle_threshold = apply_filters(
                'hunnt_ai_css_bundle_min_reduction_ratio',
                $this->min_reduction_ratio,
                $bundle_handle,
                $group,
                $bundle_reduction_ratio
            );

            if (!is_numeric($bundle_threshold)) {
                $bundle_threshold = $this->min_reduction_ratio;
            }

            $bundle_threshold = max(0.0, min(1.0, (float) $bundle_threshold));

            if ($bundle_reduction_ratio < $bundle_threshold) {
                $allow_bundle = apply_filters(
                    'hunnt_ai_css_bundle_allow_below_threshold',
                    false,
                    $bundle_handle,
                    $bundle_reduction_ratio,
                    $bundle_threshold,
                    $group
                );

                if ($allow_bundle) {
                    $bundle_warnings[] = 'bundle_below_threshold_allowed';
                } else {
                    $retain_bundle = apply_filters(
                        'hunnt_ai_css_bundle_retain_low_gain',
                        true,
                        $bundle_handle,
                        $bundle_reduction_ratio,
                        $bundle_threshold,
                        $group
                    );

                    if ($retain_bundle && $bundle_reduction_ratio > 0) {
                        $bundle_warnings[] = 'bundle_below_threshold_retained';
                        $bundle_statuses[] = 'bundle_low_gain';
                    } else {
                        $skipped[] = [
                            'handle'                 => $bundle_handle,
                            'type'                   => 'css',
                            'reason'                 => 'bundle_below_threshold',
                            'bucket'                 => $bucket,
                            'bundle_reduction_ratio' => $bundle_reduction_ratio,
                            'bundle_threshold_ratio' => $bundle_threshold,
                        ];
                        continue;
                    }
                }
            }

            $hash = $this->generate_hash($bundle_content . $bucket);
            $filename = sprintf('hunnt-ai-css-bundle-%s-%s.css', $bucket, $hash);
            $file_out = trailingslashit($output_dir) . $filename;

            if (@file_put_contents($file_out, $bundle_content) === false) {
                $skipped[] = [
                    'handle' => $bundle_handle,
                    'type'   => 'css',
                    'reason' => 'bundle_write_failed',
                    'path'   => $file_out,
                    'bucket' => $bucket,
                ];
                continue;
            }

            $bundles[$bundle_handle] = [
                'handle'           => $bundle_handle,
                'optimized_path'   => $file_out,
                'optimized_url'    => trailingslashit($output_url) . $filename,
                'hash'             => $hash,
                'media'            => $group['media'],
                'deps'             => array_values($deps),
                'original_handles' => array_values($bundle_handles),
                'original_bytes'   => $original_bytes,
                'optimized_bytes'  => $optimized_bytes,
                'intermediate_bytes' => $intermediate_bytes,
                'saved_bytes'      => max(0, $original_bytes - $optimized_bytes),
                'bundle_saved_bytes' => $bundle_saved_bytes,
                'reduction_ratio'  => $bundle_reduction_ratio,
                'threshold_ratio'  => $bundle_threshold,
                'warnings'        => array_values(array_unique($bundle_warnings)),
                'statuses'        => array_values(array_unique($bundle_statuses)),
            ];
        }

        return [
            'bundles' => $bundles,
            'skipped' => $skipped,
        ];
    }

    /**
     * Optimize a single CSS asset.
     *
     * @param array  $asset
     * @param string $output_dir
     * @param string $output_url
     * @return array{success:bool, data?:array<string,mixed>, error?:array<string,mixed>}
     */
    private function optimize_single_asset(array $asset, $output_dir, $output_url) {
        $file_path = $this->url_to_path($asset['src']);
        if (!$file_path) {
            return [
                'success' => false,
                'error'   => [
                    'error_code' => 'URL_CONVERSION_FAILED',
                    'message'    => __('Could not convert URL to file path', 'hunnt-ai'),
                ],
            ];
        }

        if (!file_exists($file_path)) {
            return [
                'success' => false,
                'error'   => [
                    'error_code' => 'FILE_NOT_FOUND',
                    'message'    => __('File does not exist', 'hunnt-ai'),
                    'path'       => $file_path,
                ],
            ];
        }

        if (!is_readable($file_path)) {
            return [
                'success' => false,
                'error'   => [
                    'error_code' => 'FILE_NOT_READABLE',
                    'message'    => __('File is not readable (permission denied)', 'hunnt-ai'),
                    'path'       => $file_path,
                ],
            ];
        }

        $css_content = @file_get_contents($file_path);
        if ($css_content === false) {
            return [
                'success' => false,
                'error'   => [
                    'error_code' => 'FILE_READ_FAILED',
                    'message'    => __('Could not read file content', 'hunnt-ai'),
                    'path'       => $file_path,
                ],
            ];
        }

        if (strlen(trim($css_content)) === 0) {
            return [
                'success' => false,
                'error'   => [
                    'error_code' => 'EMPTY_FILE',
                    'message'    => __('File is empty', 'hunnt-ai'),
                    'path'       => $file_path,
                    'file_size'  => filesize($file_path),
                ],
            ];
        }

        // Clean HTML artifacts and balance braces similar to combine_css
        $css_content = preg_replace('/<style[^>]*>|<\/style>/i', '', $css_content);
        $css_content = preg_replace('/<\/?[a-z][^>]*>/i', '', $css_content);

        $open_count = substr_count($css_content, '{');
        $close_count = substr_count($css_content, '}');
        if ($open_count !== $close_count) {
            $diff = $close_count - $open_count;
            if ($diff > 0) {
                $css_content = preg_replace('/\}\s*$/', '', $css_content, $diff);
            } elseif ($diff < 0) {
                $css_content .= str_repeat('}', abs($diff));
            }
        }

        $css_content = $this->resolve_css_paths($css_content, $asset['src']);

        $prepared_css = trim($css_content);

        if ($this->should_include_source_comment('css_individual')) {
            $prepared_css = "/* Source: {$asset['handle']} - {$asset['src']} */\n" . $prepared_css;
        }

        $directive = $this->evaluate_asset_directive($asset, $prepared_css);

        if ($directive['mode'] === 'skip') {
            $reason = isset($directive['reason']) ? $directive['reason'] : 'skip';

            return [
                'success' => false,
                'error'   => [
                    'error_code' => strtoupper($reason),
                    'message'    => sprintf(__('Asset skipped (%s)', 'hunnt-ai'), $reason),
                ],
            ];
        }

        $original_length = strlen($prepared_css);
        $optimized_css = $prepared_css;
        $status = 'passthrough';
        $warnings = [];
        $threshold = $this->get_min_reduction_ratio($asset);

        if ($directive['mode'] === 'minify') {
            $status = 'minified';
            $optimized_css = $this->minify_css($prepared_css);

            if ($optimized_css === '') {
                $warnings[] = 'minify_empty_result';
                $optimized_css = $prepared_css;
                $status = 'fallback_original';
            } else {
                $optimized_css = trim($optimized_css);
                $optimized_css = $this->maybe_reduce_fontawesome($asset, $optimized_css);
                $optimized_css = trim($optimized_css);
                $optimized_css = $this->maybe_externalize_embedded_fonts($asset, $optimized_css, $output_dir, $output_url);
                $optimized_css = trim($optimized_css);
            }
        } else {
            if (!empty($directive['reason'])) {
                $warnings[] = 'passthrough_' . $directive['reason'];
            }
        }

        $optimized_css = trim($optimized_css);
        $optimized_length = strlen($optimized_css);

        if ($optimized_length === 0) {
            $warnings[] = 'empty_output';
            $optimized_css = $prepared_css;
            $optimized_length = $original_length;
            $status = 'fallback_original';
        }

        $reduction_ratio = $original_length > 0
            ? max(0, 1 - ($optimized_length / $original_length))
            : 0;

        if ($status === 'minified') {
            if ($optimized_length >= $original_length) {
                $warnings[] = 'not_reduced';
                $optimized_css = $prepared_css;
                $optimized_length = $original_length;
                $reduction_ratio = 0;
                $status = 'passthrough_not_smaller';
            } elseif ($reduction_ratio < $threshold) {
                $allow_below = apply_filters(
                    'hunnt_ai_css_allow_below_threshold',
                    false,
                    $asset,
                    $reduction_ratio,
                    $threshold
                );

                if ($allow_below) {
                    $warnings[] = 'below_threshold_allowed';
                } else {
                    $retain_low_gain = apply_filters(
                        'hunnt_ai_css_retain_low_gain_minification',
                        true,
                        $asset,
                        $reduction_ratio,
                        $threshold
                    );

                    if ($retain_low_gain && $reduction_ratio > 0) {
                        $warnings[] = 'below_threshold_retained';
                        $status = 'minified_low_gain';
                    } else {
                        $warnings[] = 'below_threshold';
                        $optimized_css = $prepared_css;
                        $optimized_length = $original_length;
                        $reduction_ratio = 0;
                        $status = 'passthrough_threshold';
                    }
                }
            }
        }

        $optimized_css = apply_filters(
            'hunnt_ai_css_optimized_content',
            $optimized_css,
            $asset,
            $status,
            $reduction_ratio
        );

        $optimized_css = trim($optimized_css);
        $optimized_length = strlen($optimized_css);

        $hash      = $this->generate_hash($optimized_css . $asset['src'] . '|' . $status);
        $safe_slug = sanitize_key($asset['handle']);
        $filename  = 'hunnt-ai-css-' . $safe_slug . '-' . $hash . '.css';
        $file_out  = trailingslashit($output_dir) . $filename;

        if (@file_put_contents($file_out, $optimized_css) === false) {
            return [
                'success' => false,
                'error'   => [
                    'error_code' => 'WRITE_FAILED',
                    'message'    => __('Failed to write optimized CSS file', 'hunnt-ai'),
                    'path'       => $file_out,
                ],
            ];
        }

        return [
            'success' => true,
            'data'    => [
                'handle'          => $asset['handle'],
                'source'          => $asset['src'],
                'optimized_path'  => $file_out,
                'optimized_url'   => trailingslashit($output_url) . $filename,
                'hash'            => $hash,
                'media'           => $asset['media'],
                'deps'            => $asset['deps'],
                'ver'             => $asset['ver'],
                'original_bytes'  => $original_length,
                'optimized_bytes' => $optimized_length,
                'saved_bytes'     => max(0, $original_length - $optimized_length),
                'reduction_ratio' => $reduction_ratio,
                'threshold_ratio' => $threshold,
                'status'          => $status,
                'warnings'        => array_values(array_unique($warnings)),
                'inline_after'    => isset($asset['inline_after']) ? array_values((array) $asset['inline_after']) : [],
            ],
        ];
    }

    /**
     * Persist a JSON log of CSS optimization results.
     *
     * @param array $css_assets Original assets
     * @param array $optimized  Optimized assets keyed by handle
     * @param array $skipped    Skipped entries with reasons
     * @param array $log_items  Summaries for successful optimizations
     */
    private function write_css_log(array $css_assets, array $optimized, array $skipped, array $log_items) {
        $upload_dir = wp_upload_dir();
        $log_file   = trailingslashit($upload_dir['basedir']) . 'hunnt-ai-optimized/css-errors.json';

        if (!file_exists(dirname($log_file))) {
            wp_mkdir_p(dirname($log_file));
        }

        $report = [
            'timestamp'          => current_time('mysql'),
            'total_assets'       => count($css_assets),
            'optimized_count'    => count($optimized),
            'skipped_count'      => count($skipped),
            'optimized_files'    => array_values($log_items),
            'skipped'            => $skipped,
        ];

        file_put_contents($log_file, wp_json_encode($report, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
    }

    /**
     * Detect and reduce duplicated Font Awesome payloads while keeping font fallbacks.
     *
     * @param array  $asset Current asset descriptor.
     * @param string $css   Minified CSS content.
     * @return string Possibly reduced CSS output.
     */
    private function maybe_reduce_fontawesome(array $asset, $css) {
        if (!$this->is_fontawesome_stylesheet($asset, $css)) {
            return $css;
        }

        $font_face_blocks = $this->extract_font_face_blocks($css);

        if (empty($font_face_blocks)) {
            return $css;
        }

        if (self::$fontawesome_primary_handle === null) {
            self::$fontawesome_primary_handle = $asset['handle'];
            $this->register_font_face_blocks($font_face_blocks);
            return $css;
        }

        if (self::$fontawesome_primary_handle === $asset['handle']) {
            $this->register_font_face_blocks($font_face_blocks);
            return $css;
        }

        $new_blocks = $this->filter_new_font_face_blocks($font_face_blocks);

        if (!empty($new_blocks)) {
            $this->register_font_face_blocks($new_blocks);
            return sprintf(
                '/*! hunnt-ai fontawesome fallback retained for %s */%s',
                $asset['handle'],
                implode('', $new_blocks)
            );
        }

        return sprintf(
            '/*! hunnt-ai fontawesome deduplicated: using canonical %s */',
            self::$fontawesome_primary_handle
        );
    }

    /**
     * Determine whether the given stylesheet carries Font Awesome payloads.
     *
     * @param array  $asset Asset info.
     * @param string $css   CSS contents.
     * @return bool
     */
    private function is_fontawesome_stylesheet(array $asset, $css) {
        $handle_flag = isset($asset['handle']) && is_string($asset['handle']) && (stripos($asset['handle'], 'fontawesome') !== false || stripos($asset['handle'], 'font-awesome') !== false);
        $content_flag = stripos($css, 'Font Awesome') !== false && substr_count($css, '.fa-') > 50;

        return $handle_flag || $content_flag;
    }

    /**
     * Extract @font-face declarations from minified CSS.
     *
     * @param string $css CSS content.
     * @return array<int, string>
     */
    private function extract_font_face_blocks($css) {
        $matches = [];
        preg_match_all('/@font-face\s*\{.*?\}/i', $css, $matches);

        if (empty($matches[0])) {
            return [];
        }

        return array_map('trim', $matches[0]);
    }

    /**
     * Filter font-face blocks that have not yet been registered.
     *
     * @param array<int, string> $blocks
     * @return array<int, string>
     */
    private function filter_new_font_face_blocks(array $blocks) {
        $unique = [];

        foreach ($blocks as $block) {
            $hash = md5(preg_replace('/\s+/', '', $block));
            if (!isset(self::$fontawesome_font_face_hashes[$hash])) {
                $unique[] = $block;
            }
        }

        return $unique;
    }

    /**
     * Register font-face blocks to avoid duplicate preservation later in the request.
     *
     * @param array<int, string> $blocks
     * @return void
     */
    private function register_font_face_blocks(array $blocks) {
        foreach ($blocks as $block) {
            $hash = md5(preg_replace('/\s+/', '', $block));
            self::$fontawesome_font_face_hashes[$hash] = true;
        }
    }

    /**
     * Normalize bundle CSS content by stripping redundant charset declarations
     * and running a final minification pass to squeeze additional bytes.
     *
     * @param string $css Raw bundle CSS content
     * @return string Optimized bundle CSS content
     */
    private function finalize_bundle_css($css) {
        if (!is_string($css) || $css === '') {
            return '';
        }

        // Remove any @charset statements (not valid mid-file and redundant when bundled).
        if (stripos($css, '@charset') !== false) {
            $css = preg_replace('/@charset\s+[^;]+;?/i', '', $css);
        }

        $css = trim($css);

        if ($css === '') {
            return '';
        }

        $minified = $this->minify_css($css);

        if (is_string($minified) && $minified !== '' && strlen($minified) <= strlen($css)) {
            $css = $minified;
        }

        return trim($css);
    }

    /**
     * Replace embedded font data URIs with external files to reduce CSS payload size.
     *
     * @param array  $asset      Asset metadata (handle, src, etc.)
     * @param string $css        Minified CSS content
     * @param string $output_dir Directory where optimized assets are stored
     * @param string $output_url Public URL for the optimized directory
     * @return string CSS content with externalized fonts where applicable
     */
    private function maybe_externalize_embedded_fonts(array $asset, $css, $output_dir, $output_url) {
        if (!is_string($css) || $css === '') {
            return $css;
        }

        if (stripos($css, 'data:application/font') === false && stripos($css, 'data:font/') === false) {
            return $css;
        }

        $pattern = "/url\((['\"]?)data:(?<mime>(?:application|font)\/(?:font-woff2?|font-woff|x-font-woff2?|x-font-woff|x-font-truetype|x-font-opentype|woff2?|woff|opentype|truetype|octet-stream))[^,]*,\s*(?<data>[^\)]+?)\1\)/i";

        $output_dir = trailingslashit($output_dir);
        $output_url = trailingslashit($output_url);

        $css = preg_replace_callback(
            $pattern,
            function ($matches) use ($asset, $output_dir, $output_url) {
                $quote = $matches[1] ?? '';
                $mime  = isset($matches['mime']) ? strtolower($matches['mime']) : '';
                $data  = $matches['data'] ?? '';

                $extension = $this->get_font_extension_from_mime($mime);
                if ($extension === null) {
                    return $matches[0];
                }

                $clean_data = preg_replace('/\s+/', '', $data);
                if ($clean_data === '') {
                    return $matches[0];
                }

                $binary = base64_decode($clean_data, true);
                if ($binary === false) {
                    return $matches[0];
                }

                $hash = substr(md5($binary), 0, 12);

                if (isset(self::$extracted_font_map[$hash])) {
                    $filename = self::$extracted_font_map[$hash];
                } else {
                    $filename = sprintf('hunnt-ai-font-%s.%s', $hash, $extension);
                    $target_path = $output_dir . $filename;

                    if (!file_exists($target_path)) {
                        if (@file_put_contents($target_path, $binary) === false) {
                            return $matches[0];
                        }
                    }

                    self::$extracted_font_map[$hash] = $filename;
                }

                $replacement_url = $output_url . $filename;

                return 'url(' . $quote . $replacement_url . $quote . ')';
            },
            $css
        );

        return $css;
    }

    /**
     * Map MIME types to font file extensions.
     *
     * @param string $mime
     * @return string|null
     */
    private function get_font_extension_from_mime($mime) {
        $map = [
            'application/font-woff2'    => 'woff2',
            'application/font-woff'     => 'woff',
            'application/x-font-woff2'  => 'woff2',
            'application/x-font-woff'   => 'woff',
            'font/woff2'                => 'woff2',
            'font/woff'                 => 'woff',
            'application/x-font-truetype' => 'ttf',
            'application/font-truetype' => 'ttf',
            'font/ttf'                  => 'ttf',
            'font/truetype'             => 'ttf',
            'application/x-font-opentype' => 'otf',
            'font/otf'                  => 'otf',
            'font/opentype'             => 'otf',
            'application/octet-stream'  => 'woff',
        ];

        return $map[$mime] ?? null;
    }
    
    /**
     * Resolve relative paths in CSS (url(), @import, etc.)
     * Converts paths like ../images/demo.png to absolute URLs
     * 
     * @param string $css_content CSS content
     * @param string $original_url Original CSS file URL
     * @return string CSS with resolved paths
     */
    /**
     * Resolve relative paths in CSS (url(), @import, etc.)
     * Converts paths like ../images/demo.png to absolute URLs
     * 
     * @param string $css_content CSS content
     * @param string $original_url Original CSS file URL
     * @return string CSS with resolved paths
     */
    private function resolve_css_paths($css_content, $original_url) {
        // Safety check
        if (empty($css_content) || !is_string($css_content)) {
            return $css_content;
        }
        
        // Get the base URL (directory of the CSS file)
        $base_url = $this->get_base_url($original_url);
        
        if (empty($base_url)) {
            return $css_content; // Return original if base URL invalid
        }
        
        // Handle url() declarations (background images, fonts, etc.)
        $css_content = preg_replace_callback(
            '/url\s*\(\s*[\'"]?([^\)\'"]+)[\'"]?\s*\)/i',
            function($matches) use ($base_url) {
                $url = trim($matches[1]);
                
                // Skip data URIs
                if (preg_match('/^data:/i', $url)) {
                    return $matches[0];
                }
                
                // Skip absolute URLs
                if (preg_match('/^(https?:)?\/\//i', $url)) {
                    return $matches[0];
                }
                
                // Skip root-relative URLs
                if (strpos($url, '/') === 0) {
                    return $matches[0];
                }
                
                // Convert relative to absolute
                $absolute_url = $this->resolve_relative_url($url, $base_url);
                return "url('{$absolute_url}')";
            },
            $css_content
        );
        
        // Handle @import statements
        $css_content = preg_replace_callback(
            '/@import\s+[\'"]([^\'"]+)[\'"]/i',
            function($matches) use ($base_url) {
                $url = $matches[1];
                
                // Skip absolute URLs
                if (preg_match('/^(https?:)?\/\//i', $url)) {
                    return $matches[0];
                }
                
                // Skip root-relative URLs
                if (strpos($url, '/') === 0) {
                    return $matches[0];
                }
                
                $absolute_url = $this->resolve_relative_url($url, $base_url);
                return "@import '{$absolute_url}'";
            },
            $css_content
        );
        
        return $css_content;
    }
    
    /**
     * Get base URL from a file URL (directory path)
     * 
     * @param string $url File URL
     * @return string Base URL
     */
    private function get_base_url($url) {
        // Remove query string
        $url = preg_replace('/\?.*$/', '', $url);
        
        // Get directory
        $parts = explode('/', $url);
        array_pop($parts); // Remove filename
        
        return implode('/', $parts);
    }
    
    /**
     * Resolve relative URL to absolute based on base URL
     * Handles ../ and ./ navigation
     * 
     * @param string $relative_url Relative URL (e.g., ../images/logo.png)
     * @param string $base_url Base URL (directory of CSS file)
     * @return string Absolute URL
     */
    private function resolve_relative_url($relative_url, $base_url) {
        // Remove query strings and fragments from relative URL
        $clean_url = preg_replace('/[?#].*$/', '', $relative_url);
        
        // Split base URL into parts
        $base_parts = explode('/', trim($base_url, '/'));
        
        // Split relative URL into parts
        $relative_parts = explode('/', $clean_url);
        
        // Process each part of the relative URL
        foreach ($relative_parts as $part) {
            if ($part === '..') {
                // Go up one directory
                if (!empty($base_parts)) {
                    array_pop($base_parts);
                }
            } elseif ($part === '.' || $part === '') {
                // Current directory or empty, skip
                continue;
            } else {
                // Add to path
                $base_parts[] = $part;
            }
        }
        
        // Reconstruct URL
        $resolved = implode('/', $base_parts);
        
        // Ensure it starts with protocol or //
        if (!preg_match('/^(https?:)?\/\//i', $resolved)) {
            // Add protocol
            $site_url = site_url();
            $parsed = parse_url($site_url);
            $protocol = isset($parsed['scheme']) ? $parsed['scheme'] : 'https';
            $host = isset($parsed['host']) ? $parsed['host'] : '';
            
            $resolved = $protocol . '://' . $host . '/' . $resolved;
        }
        
        return $resolved;
    }
    
    /**
     * Convert URL to filesystem path
     * 
     * @param string $url Asset URL
     * @return string|false Filesystem path or false
     */
    /**
     * Convert asset URL to local file system path
     * Production-grade conversion following WordPress standards
     * 
     * @param string $url Asset URL
     * @return string|false Absolute file path or false on failure
     */
    private function url_to_path($url) {
        // Skip empty URLs
        if (empty($url)) {
            return false;
        }
        
        // Remove query string and fragments first
        $url = preg_replace('/[?#].*$/', '', $url);
        
        // Handle relative URLs - convert to absolute
        if (strpos($url, '//') === false && strpos($url, 'http') === false) {
            // Remove leading slash if present
            $url = ltrim($url, '/');
            $url = site_url('/' . $url);
        }
        
        // Get WordPress URLs and directories with fallbacks
        // WP_CONTENT_URL and WP_CONTENT_DIR are preferred but may not be defined
        if (defined('WP_CONTENT_URL') && defined('WP_CONTENT_DIR')) {
            $content_url = untrailingslashit(WP_CONTENT_URL);
            $content_dir = untrailingslashit(WP_CONTENT_DIR);
        } else {
            // Fallback: Calculate from ABSPATH
            $content_url = untrailingslashit(site_url('/wp-content'));
            $content_dir = untrailingslashit(ABSPATH . 'wp-content');
        }
        
        $site_url = untrailingslashit(site_url());
        $abspath = untrailingslashit(ABSPATH);
        
        // Normalize URL for comparison (handle http/https)
        $url_normalized = preg_replace('#^https?://#i', '', strtolower($url));
        $content_url_normalized = preg_replace('#^https?://#i', '', strtolower($content_url));
        $site_url_normalized = preg_replace('#^https?://#i', '', strtolower($site_url));
        
        $file_path = false;
        
        // Try wp-content URL first (most common case for themes/plugins)
        if (strpos($url_normalized, $content_url_normalized) === 0) {
            // Extract path after wp-content
            $url_parts = preg_replace('#^https?://[^/]+#i', '', $url);
            $content_path = preg_replace('#^https?://[^/]+#i', '', $content_url);
            
            if (strpos($url_parts, $content_path) !== false) {
                $relative_path = str_replace($content_path, '', $url_parts);
                $file_path = $content_dir . $relative_path;
            }
        }
        // Try site URL
        elseif (strpos($url_normalized, $site_url_normalized) === 0) {
            $url_parts = preg_replace('#^https?://[^/]+#i', '', $url);
            $site_path = preg_replace('#^https?://[^/]+#i', '', $site_url);
            
            if (strpos($url_parts, $site_path) !== false) {
                $relative_path = str_replace($site_path, '', $url_parts);
                $file_path = $abspath . $relative_path;
            }
        }
        // Handle protocol-relative URLs (//example.com/path)
        elseif (strpos($url, '//') === 0) {
            $site_host = parse_url($site_url, PHP_URL_HOST);
            if ($site_host && stripos($url, '//' . $site_host) === 0) {
                $protocol = is_ssl() ? 'https:' : 'http:';
                return $this->url_to_path($protocol . $url);
            }
            return false; // External protocol-relative URL
        }
        
        // No valid path found
        if (!$file_path) {
            return false;
        }
        
        // Normalize path separators for current OS
        $file_path = str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $file_path);
        
        // Resolve real path (handles symlinks, relative paths like ../)
        $real_path = @realpath($file_path);
        
        // Validation checks
        if (!$real_path || !file_exists($real_path)) {
            return false; // File doesn't exist
        }
        
        // Security: Ensure file is within WordPress installation
        $real_path_normalized = str_replace('\\', '/', $real_path);
        $content_dir_normalized = str_replace('\\', '/', $content_dir);
        $abspath_normalized = str_replace('\\', '/', $abspath);
        
        $is_allowed = false;
        if (strpos($real_path_normalized, $content_dir_normalized) === 0 || 
            strpos($real_path_normalized, $abspath_normalized) === 0) {
            $is_allowed = true;
        }
        
        if (!$is_allowed) {
            return false; // Path is outside WordPress installation
        }
        
        // Additional security: Check file type
        $extension = strtolower(pathinfo($real_path, PATHINFO_EXTENSION));
        
        if ($extension !== 'css') {
            return false; // Not a CSS file
        }
        
        // Additional check: file is readable
        if (!is_readable($real_path)) {
            return false;
        }
        
        return $real_path;
    }
    
    /**
     * Basic CSS minification using native PHP
     * Removes comments, whitespace, and unnecessary characters
     * 
     * @param string $css CSS content
     * @return string Minified CSS
     */
    /**
     * Basic CSS minification
     * Removes comments, whitespace, and unnecessary characters
     * 
     * @param string $css CSS content
     * @return string Minified CSS
     */
    /**
     * Minify CSS using matthiasmullie/minify
     *
     * @param string $css
     * @return string
     */
    private function minify_css($css) {
        if (empty($css) || !is_string($css)) {
            return '';
        }

        // Load matthiasmullie/minify
        $autoload = dirname(__DIR__, 2) . '/vendor/autoload.php';
        if (file_exists($autoload)) {
            require_once $autoload;
        }

        // Use matthiasmullie/minify if available
        if (class_exists('\\MatthiasMullie\\Minify\\CSS')) {
            try {
                $minifier = new \MatthiasMullie\Minify\CSS($css);
                $minified = $minifier->minify();
                
                if (is_string($minified) && $minified !== '') {
                    return $minified;
                }
            } catch (\Exception $e) {
                error_log('Hunnt AI CSS: Minification failed - ' . $e->getMessage());
            }
        }

        // Return original if minification fails
        return $css;
    }

    /**
     * Decide whether to include asset source comments for the given context.
     *
     * @param string $context Context identifier (e.g. css_combined, css_individual)
     * @return bool
     */
    private function should_include_source_comment($context) {
        static $cache = [];

        if (!array_key_exists($context, $cache)) {
            $cache[$context] = (bool) apply_filters('hunnt_ai_include_source_comments', false, $context, 'css');
        }

        return $cache[$context];
    }
    
    /**
     * Generate hash for cache busting
     * 
     * @param string $content File content
     * @return string Hash
     */
    public function generate_hash($content) {
        return substr(md5($content), 0, 12);
    }
}
