<?php
/**
 * Asset Optimizer - Main orchestrator for CSS/JS optimization
 * 
 * Native PHP implementation without external dependencies
 * Collects, filters, and manages optimization of theme/plugin assets
 * Excludes WordPress core files (wp-admin, wp-includes)
 * 
 * @package Hunnt_AI
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Hunnt_AI_Asset_Optimizer {
    
    /**
     * @var string Path to manifest file
     */
    private $manifest_file;
    
    /**
     * @var string Path to optimized assets directory
     */
    private $optimized_dir;
    
    /**
     * @var string URL to optimized assets directory
     */
    private $optimized_url;
    
    /**
     * @var string Path to log file
     */
    private $log_file;

    /**
     * @var array<string, array<int, string>> List of asset handle patterns to exclude per type
     */
    /**
     * Default exclude patterns per asset type.
     * Excludes critical libraries that other scripts depend on.
     */
    private $excluded_handles = [
        'css' => [],
        'js'  => [
            // WooCommerce Cookie libraries
            'js-cookie',           // Alias for wc-js-cookie
            'wc-js-cookie',        // Actual JS Cookie library
            'jquery-cookie',       // Alias for wc-jquery-cookie (deprecated)
            'wc-jquery-cookie',    // Old jQuery cookie library (deprecated)
            
            // Core jQuery libraries
            'jquery',              // Core jQuery library
            'jquery-migrate',      // jQuery migration helper
            'jquery-blockui',      // jQuery BlockUI plugin
        ],
    ];

    /**
     * @var array<string, array<int, array<string, mixed>>> Skipped asset metadata collected during scanning
     */
    private $skipped_assets = [
        'css' => [],
        'js'  => [],
    ];
    
    /**
     * Constructor - Initialize paths and create directories
     */
    public function __construct() {
        $upload_dir = wp_upload_dir();
        $base_dir = trailingslashit($upload_dir['basedir']) . 'hunnt-ai-optimized';
        $base_url = trailingslashit($upload_dir['baseurl']) . 'hunnt-ai-optimized';
        
        $this->optimized_dir = $base_dir;
        $this->optimized_url = $base_url;
        $this->manifest_file = $base_dir . '/manifest.json';
        $this->log_file = $base_dir . '/optimization-log.json';

        $this->excluded_handles['css'] = apply_filters(
            'hunnt_ai_css_excluded_handles',
            $this->excluded_handles['css']
        );

        $this->excluded_handles['js'] = apply_filters(
            'hunnt_ai_js_excluded_handles',
            $this->excluded_handles['js']
        );
        
        // Ensure directories exist
        if (!file_exists($base_dir)) {
            wp_mkdir_p($base_dir);
        }
    }
    
    /**
     * Collect all enqueued CSS/JS (excluding WP core)
     * 
     * @return array ['css' => [], 'js' => []]
     */
    public function collect_assets() {
        global $wp_styles, $wp_scripts;
        
        // Reset skipped assets tracking for this collection run
        $this->skipped_assets = [
            'css' => [],
            'js'  => [],
        ];

        $assets = [
            'css' => [],
            'js' => []
        ];
        
        // Collect CSS
        if (!empty($wp_styles->queue)) {
            foreach ($wp_styles->queue as $handle) {
                if (!isset($wp_styles->registered[$handle])) {
                    continue;
                }
                
                $style = $wp_styles->registered[$handle];
                $src = $style->src;

                $matched_pattern = $this->match_excluded_handle($handle, 'css');
                if ($matched_pattern) {
                    $this->skipped_assets['css'][] = [
                        'handle' => $handle,
                        'src' => $src,
                        'pattern' => $matched_pattern,
                        'type' => 'css',
                        'reason' => 'excluded_handle_pattern'
                    ];
                    continue;
                }
                
                // Skip external and WP core files
                if ($this->should_process_asset($src, 'css')) {
                    $assets['css'][] = [
                        'handle'       => $handle,
                        'src'          => $src,
                        'deps'         => $style->deps,
                        'ver'          => $style->ver,
                        'media'        => $style->args,
                        'inline_after' => $this->normalize_inline_entries(
                            method_exists($wp_styles, 'get_data') ? $wp_styles->get_data($handle, 'after') : []
                        ),
                    ];
                }
            }
        }
        
        // Collect JS
        if (!empty($wp_scripts->queue)) {
            foreach ($wp_scripts->queue as $handle) {
                if (!isset($wp_scripts->registered[$handle])) {
                    continue;
                }
                
                $script = $wp_scripts->registered[$handle];
                $src = $script->src;

                $matched_pattern = $this->match_excluded_handle($handle, 'js');
                if ($matched_pattern) {
                    $this->skipped_assets['js'][] = [
                        'handle' => $handle,
                        'src' => $src,
                        'pattern' => $matched_pattern,
                        'type' => 'js',
                        'reason' => 'excluded_handle_pattern'
                    ];
                    continue;
                }
                
                // Skip external and WP core files
                if ($this->should_process_asset($src, 'js')) {
                    $assets['js'][] = [
                        'handle'            => $handle,
                        'src'               => $src,
                        'deps'              => $script->deps,
                        'ver'               => $script->ver,
                        'in_footer'         => isset($script->extra['group']) ? $script->extra['group'] : 0,
                        'inline_before'     => $this->normalize_inline_entries(
                            method_exists($wp_scripts, 'get_data') ? $wp_scripts->get_data($handle, 'before') : []
                        ),
                        'inline_after'      => $this->normalize_inline_entries(
                            method_exists($wp_scripts, 'get_data') ? $wp_scripts->get_data($handle, 'after') : []
                        ),
                        'inline_data'       => $this->normalize_inline_string(
                            method_exists($wp_scripts, 'get_data') ? $wp_scripts->get_data($handle, 'data') : ''
                        ),
                        'textdomain'        => property_exists($script, 'textdomain') ? $script->textdomain : null,
                        'translations_path' => property_exists($script, 'translations_path') ? $script->translations_path : null,
                    ];
                }
            }
        }
        
        $assets['skipped'] = $this->skipped_assets;
        
        return $assets;
    }

    /**
     * Retrieve skipped assets metadata from last collection run
     *
     * @return array
     */
    public function get_skipped_assets() {
        return $this->skipped_assets;
    }

    /**
     * Determine if a handle should be excluded based on configured patterns
     *
     * @param string $handle Asset handle
     * @param string $type Asset type ('css' or 'js')
     * @return string|null Matched pattern or null when not excluded
     */
    private function match_excluded_handle($handle, $type) {
        $patterns = $this->excluded_handles[$type] ?? [];

        if (empty($handle) || empty($patterns)) {
            return null;
        }

        foreach ($patterns as $pattern) {
            $regex = '/^' . str_replace(['\*', '\?'], ['.*', '.'], preg_quote($pattern, '/')) . '$/i';
            if (preg_match($regex, $handle)) {
                return $pattern;
            }
        }

        return null;
    }
    
    /**
     * Normalize inline snippets captured from WP_Scripts/WP_Styles.
     *
     * @param mixed $value Raw inline data (string|array|false)
     * @return array<int, string>
     */
    private function normalize_inline_entries($value) {
        if (empty($value)) {
            return [];
        }

        if (is_string($value)) {
            $value = [$value];
        }

        if (!is_array($value)) {
            return [];
        }

        $normalized = [];

        foreach ($value as $snippet) {
            if (!is_string($snippet)) {
                continue;
            }

            $trimmed = trim($snippet);

            if ($trimmed === '') {
                continue;
            }

            $normalized[] = $trimmed;
        }

        return $normalized;
    }

    /**
     * Normalize inline data blocks (e.g., localized settings scripts).
     *
     * @param mixed $value Raw inline data
     * @return string
     */
    private function normalize_inline_string($value) {
        if (!is_string($value)) {
            return '';
        }

        $trimmed = trim($value);

        return $trimmed !== '' ? $trimmed : '';
    }
    
    /**
     * Determine if asset should be processed
     * Production-grade filtering following WordPress plugin standards
     * Only process local theme and plugin assets, exclude WordPress core and external CDNs
     * 
     * @param string $src Asset URL
     * @param string $type 'css' or 'js'
     * @return bool
     */
    private function should_process_asset($src, $type) {
        // Skip empty sources
        if (empty($src)) {
            return false;
        }
        
        // Remove query strings for analysis
        $src_clean = preg_replace('/[?#].*$/', '', $src);
        
        // Convert relative URLs to absolute for consistent checking
        if (strpos($src, '//') === false && strpos($src, 'http') === false) {
            $src = site_url('/' . ltrim($src, '/'));
            $src_clean = preg_replace('/[?#].*$/', '', $src);
        }
        
        // Get WordPress URLs (normalized for comparison)
        $content_url = untrailingslashit(WP_CONTENT_URL);
        $site_url = untrailingslashit(site_url());
        
        // Normalize for protocol-agnostic comparison
        $src_normalized = preg_replace('#^https?://#', '', $src_clean);
        $content_url_normalized = preg_replace('#^https?://#', '', $content_url);
        $site_url_normalized = preg_replace('#^https?://#', '', $site_url);
        
        // Check if asset is on the same domain
        $is_local = false;
        
        if (strpos($src_normalized, $content_url_normalized) === 0 || 
            strpos($src_normalized, $site_url_normalized) === 0) {
            $is_local = true;
        }
        
        // Handle protocol-relative URLs (//example.com/path)
        if (strpos($src, '//') === 0) {
            $site_host = parse_url($site_url, PHP_URL_HOST);
            if ($site_host && strpos($src, '//' . $site_host) === 0) {
                $is_local = true;
            }
        }
        
        // Skip external URLs (CDNs, third-party resources)
        if (!$is_local) {
            return false;
        }
        
        // Exclude WordPress core directories
        $excluded_paths = [
            '/wp-includes/',
            '/wp-admin/',
            '/wp-content/mu-plugins/', // Must-use plugins (usually core functionality)
            '/wp-content/uploads/hunnt-ai-optimized/', // Skip plugin output to avoid recursion
        ];
        
        foreach ($excluded_paths as $path) {
            if (strpos($src_clean, $path) !== false) {
                return false;
            }
        }
        
        // Only include theme and plugin assets
        $allowed_paths = [
            '/wp-content/themes/',
            '/wp-content/plugins/',
            '/wp-content/uploads/',
        ];

        /**
         * Allow third-parties to modify the allowed asset path prefixes.
         *
         * @param array  $allowed_paths Current allowed path prefixes.
         * @param string $src_clean     Normalized source path (without query string).
         * @param string $type          Asset type (css|js).
         */
        $allowed_paths = (array) apply_filters('hunnt_ai_asset_allowed_paths', $allowed_paths, $src_clean, $type);
        
        $is_allowed = false;
        foreach ($allowed_paths as $path) {
            if (strpos($src_clean, $path) !== false) {
                $is_allowed = true;
                break;
            }
        }
        
        if (!$is_allowed) {
            return false;
        }
        
        // Additional file type validation
        $extension = strtolower(pathinfo($src_clean, PATHINFO_EXTENSION));
        
        if ($type === 'css' && $extension !== 'css') {
            return false;
        }
        
        if ($type === 'js' && $extension !== 'js') {
            return false;
        }
        
        return true;
    }
    
    /**
     * Save manifest for rollback and state tracking
     * 
     * @param array $original_assets Original collected assets
     * @param array $optimized_info Optimized file information
     */
    public function save_manifest($original_assets, $optimized_info) {
        $manifest = [
            'timestamp' => current_time('mysql'),
            'version' => defined('HUNNT_AI_VERSION') ? HUNNT_AI_VERSION : '1.0.0',
            'original' => $original_assets,
            'optimized' => $optimized_info,
            'site_url' => home_url()
        ];
        
        $this->persist_manifest($manifest);
    }

    /**
     * Persist a fully constructed manifest array to disk.
     *
     * @param array $manifest Manifest payload.
     */
    public function persist_manifest(array $manifest) {
        file_put_contents(
            $this->manifest_file,
            wp_json_encode($manifest, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES)
        );
    }
    
    /**
     * Load manifest
     * 
     * @return array|null
     */
    public function load_manifest() {
        if (!file_exists($this->manifest_file)) {
            return null;
        }
        
        $content = file_get_contents($this->manifest_file);
        return json_decode($content, true);
    }
    
    /**
     * Get optimized directory path
     * 
     * @return string
     */
    public function get_optimized_dir() {
        return $this->optimized_dir;
    }
    
    /**
     * Get optimized directory URL
     * 
     * @return string
     */
    public function get_optimized_url() {
        return $this->optimized_url;
    }
    
    /**
     * Clear optimization cache and remove all optimized files
     */
    /**
     * Clear all optimization cache and temporary files
     * Production-grade cleanup removes all generated files, logs, and manifests
     * 
     * @return bool True if cleanup successful, false otherwise
     */
    public function clear_cache() {
        $removed_files = [];
        $failed_files = [];
        
        if (file_exists($this->optimized_dir)) {
            // Remove generated CSS/JS assets (legacy combined and per-handle files)
            $patterns = [
                '/combined-*.css',
                '/combined-*.js',
                '/hunnt-ai-css-*.css',
                '/hunnt-ai-js-*.js',
            ];

            foreach ($patterns as $pattern) {
                $files = glob($this->optimized_dir . $pattern, GLOB_NOSORT);
                if (!$files) {
                    continue;
                }

                foreach ($files as $file) {
                    if (@unlink($file)) {
                        $removed_files[] = basename($file);
                    } else {
                        $failed_files[] = basename($file);
                    }
                }
            }
            
            // Remove manifest file
            if (file_exists($this->manifest_file)) {
                if (@unlink($this->manifest_file)) {
                    $removed_files[] = 'manifest.json';
                } else {
                    $failed_files[] = 'manifest.json';
                }
            }
            
            // Remove error log files
            $error_logs = [
                $this->optimized_dir . '/error-log.json',
                $this->optimized_dir . '/css-errors.json',
                $this->optimized_dir . '/js-errors.json',
                $this->optimized_dir . '/optimization-log.json'
            ];
            
            foreach ($error_logs as $log_file) {
                if (file_exists($log_file)) {
                    if (@unlink($log_file)) {
                        $removed_files[] = basename($log_file);
                    } else {
                        $failed_files[] = basename($log_file);
                    }
                }
            }
            
            // Remove log directory if exists
            $log_dir = $this->optimized_dir . '/logs';
            if (file_exists($log_dir) && is_dir($log_dir)) {
                // Remove all files in logs directory
                $log_files = glob($log_dir . '/*');
                if ($log_files) {
                    foreach ($log_files as $log_file) {
                        @unlink($log_file);
                    }
                }
                // Remove the logs directory
                @rmdir($log_dir);
            }
            
            // Try to remove main optimization directory if empty
            $remaining_files = glob($this->optimized_dir . '/*');
            if (empty($remaining_files)) {
                @rmdir($this->optimized_dir);
            }
        }
        
        // Log the clear action
        $this->log_action('clear_cache', [
            'removed_files' => $removed_files,
            'failed_files' => $failed_files,
            'total_removed' => count($removed_files),
            'total_failed' => count($failed_files)
        ]);
        
        return empty($failed_files);
    }
    
    /**
     * Log optimization action
     * 
     * @param string $action Action name
     * @param array $data Additional data
     */
    public function log_action($action, $data = []) {
        $log_dir = dirname($this->log_file);
        if (!file_exists($log_dir) || !is_dir($log_dir)) {
            wp_mkdir_p($log_dir);
        }

        $log_entry = [
            'timestamp' => current_time('mysql'),
            'action' => $action,
            'data' => $data
        ];
        
        $existing_log = [];
        if (file_exists($this->log_file)) {
            $content = file_get_contents($this->log_file);
            $existing_log = json_decode($content, true);
            if (!is_array($existing_log)) {
                $existing_log = [];
            }
        }
        
        $existing_log[] = $log_entry;
        
        // Keep only last 100 entries
        if (count($existing_log) > 100) {
            $existing_log = array_slice($existing_log, -100);
        }
        
        file_put_contents(
            $this->log_file,
            wp_json_encode($existing_log, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES)
        );
    }
    
    /**
     * Check if optimization is currently active
     * 
     * @return bool
     */
    public function is_optimization_active() {
        return file_exists($this->manifest_file) && get_option('hunnt_ai_assets_minify_enable', false);
    }
}
