<?php
/**
 * Technical SEO Helper Functions
 * 
 * Provides reusable utility functions for managing Technical SEO features
 * Follows WordPress coding standards and SOLID principles
 * 
 * @package Hunnt_AI
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Get all technical SEO option keys
 * 
 * @return array List of option keys used by technical SEO
 */
function hunnt_ai_get_technical_seo_option_keys() {
    return [
        'hunnt_ai_technical_seo_enabled',
        'hunnt_ai_basic_cache_enabled',
        'hunnt_ai_html_minify_enabled',
        'hunnt_ai_enable_gzip',
        'hunnt_ai_assets_minify_enable',
        'hunnt_ai_assets_minified',
    ];
}

/**
 * Get technical SEO feature configuration
 * 
 * @return array Feature configurations with metadata
 */
function hunnt_ai_get_technical_seo_features() {
    return [
        'cache' => [
            'name'        => __('Cache Headers', 'hunnt-ai'),
            'description' => __('Sends cache control headers to improve page load times', 'hunnt-ai'),
            'option_key'  => 'hunnt_ai_basic_cache_enabled',
            'icon'        => 'dashicons-performance',
        ],
        'minify_css' => [
            'name'        => __('Minify CSS', 'hunnt-ai'),
            'description' => __('Combines and minifies CSS files to reduce file size', 'hunnt-ai'),
            'option_key'  => 'hunnt_ai_assets_minify_enable',
            'icon'        => 'dashicons-media-code',
        ],
        'minify_js' => [
            'name'        => __('Minify JavaScript', 'hunnt-ai'),
            'description' => __('Combines and minifies JavaScript files to reduce file size', 'hunnt-ai'),
            'option_key'  => 'hunnt_ai_assets_minify_enable',
            'icon'        => 'dashicons-media-code',
        ],
        'minify_html' => [
            'name'        => __('Minify HTML', 'hunnt-ai'),
            'description' => __('Compresses HTML output by removing unnecessary whitespace', 'hunnt-ai'),
            'option_key'  => 'hunnt_ai_html_minify_enabled',
            'icon'        => 'dashicons-editor-code',
        ],
        'gzip' => [
            'name'        => __('Gzip Compression', 'hunnt-ai'),
            'description' => __('Enables server-side gzip compression for faster delivery', 'hunnt-ai'),
            'option_key'  => 'hunnt_ai_enable_gzip',
            'icon'        => 'dashicons-archive',
        ],
    ];
}

/**
 * Check if a specific technical SEO feature is enabled
 * 
 * @param string $feature_key Feature key (cache, minify_css, etc.)
 * @return bool True if feature is enabled
 */
function hunnt_ai_is_feature_enabled($feature_key) {
    $features = hunnt_ai_get_technical_seo_features();
    
    if (!isset($features[$feature_key])) {
        return false;
    }
    
    $option_key = $features[$feature_key]['option_key'];
    
    // Special handling for CSS/JS minification
    if (in_array($feature_key, ['minify_css', 'minify_js'], true)) {
        $paths = hunnt_ai_get_uploads_optimized_paths();
        $file_key = $feature_key === 'minify_css' ? 'css' : 'js';
        return file_exists($paths[$file_key]);
    }
    
    // Special handling for gzip
    if ($feature_key === 'gzip') {
        return function_exists('hunnt_ai_check_gzip_active') 
            ? hunnt_ai_check_gzip_active() 
            : (bool) get_option($option_key, false);
    }
    
    return (bool) get_option($option_key, false);
}

/**
 * Sanitize and validate technical SEO feature keys
 * 
 * @param array $features Array of feature keys
 * @return array Validated feature keys
 */
function hunnt_ai_sanitize_feature_keys($features) {
    if (!is_array($features)) {
        return [];
    }
    
    $valid_features = array_keys(hunnt_ai_get_technical_seo_features());
    $sanitized = [];
    
    foreach ($features as $feature) {
        $feature = sanitize_text_field($feature);
        if (in_array($feature, $valid_features, true)) {
            $sanitized[] = $feature;
        }
    }
    
    return array_unique($sanitized);
}

/**
 * Log technical SEO actions for debugging
 * 
 * @param string $action Action performed (enable, disable, apply)
 * @param array  $data   Additional data to log
 * @return void
 */
function hunnt_ai_log_technical_seo_action($action, $data = []) {
    if (!WP_DEBUG || !WP_DEBUG_LOG) {
        return;
    }
    
    $log_entry = sprintf(
        '[Hunnt AI Technical SEO] Action: %s | User: %s | Time: %s | Data: %s',
        $action,
        get_current_user_id(),
        current_time('Y-m-d H:i:s'),
        wp_json_encode($data)
    );
    
    error_log($log_entry);
}

/**
 * Get technical SEO performance metrics
 * 
 * @return array Performance metrics
 */
function hunnt_ai_get_technical_seo_metrics() {
    $status = hunnt_ai_get_technical_seo_status();
    $enabled_count = count(array_filter($status));
    $total_count = count($status);
    
    return [
        'enabled_count'    => $enabled_count,
        'total_count'      => $total_count,
        'enabled_percent'  => $total_count > 0 ? round(($enabled_count / $total_count) * 100) : 0,
        'is_fully_enabled' => $enabled_count === $total_count,
        'is_globally_enabled' => hunnt_ai_is_technical_seo_enabled(),
        'features'         => $status,
    ];
}

/**
 * Clear all technical SEO caches and optimized files
 * 
 * @return bool Success status
 */
function hunnt_ai_clear_technical_seo_cache() {
    $cleared = false;
    
    // Clear optimized CSS/JS files
    $paths = hunnt_ai_get_uploads_optimized_paths();
    
    if (file_exists($paths['css'])) {
        $cleared = @unlink($paths['css']) || $cleared;
    }
    
    if (file_exists($paths['js'])) {
        $cleared = @unlink($paths['js']) || $cleared;
    }
    
    // Clear WordPress object cache if available
    if (function_exists('wp_cache_flush')) {
        wp_cache_flush();
        $cleared = true;
    }
    
    hunnt_ai_log_technical_seo_action('cache_cleared', [
        'success' => $cleared,
    ]);
    
    return $cleared;
}

/**
 * Validate technical SEO requirements
 * 
 * @return array Validation results with missing requirements
 */
function hunnt_ai_validate_technical_seo_requirements() {
    $requirements = [
        'php_version' => [
            'required' => '7.4',
            'current'  => PHP_VERSION,
            'met'      => version_compare(PHP_VERSION, '7.4', '>='),
        ],
        'zlib_extension' => [
            'required' => true,
            'current'  => extension_loaded('zlib'),
            'met'      => extension_loaded('zlib'),
        ],
        'writable_uploads' => [
            'required' => true,
            'current'  => is_writable(wp_upload_dir()['basedir']),
            'met'      => is_writable(wp_upload_dir()['basedir']),
        ],
    ];
    
    $missing_modules = function_exists('hunnt_ai_check_modules') 
        ? hunnt_ai_check_modules() 
        : [];
    
    $requirements['php_modules'] = [
        'required' => [],
        'current'  => $missing_modules,
        'met'      => empty($missing_modules),
    ];
    
    return [
        'all_met'      => array_reduce($requirements, function($carry, $req) {
            return $carry && $req['met'];
        }, true),
        'requirements' => $requirements,
    ];
}

/**
 * Get user-friendly status message for technical SEO
 * 
 * @return string Status message
 */
function hunnt_ai_get_technical_seo_status_message() {
    if (!hunnt_ai_is_technical_seo_enabled()) {
        return __('Technical SEO is currently disabled. Enable it to start optimizing your site.', 'hunnt-ai');
    }
    
    $metrics = hunnt_ai_get_technical_seo_metrics();
    
    if ($metrics['is_fully_enabled']) {
        return __('All technical SEO optimizations are active and running!', 'hunnt-ai');
    }
    
    if ($metrics['enabled_count'] === 0) {
        return __('Technical SEO is enabled but no optimizations are active yet. Click "Fix Now" to apply.', 'hunnt-ai');
    }
    
    return sprintf(
        __('%d of %d optimizations are active. Click "Fix Now" to enable all features.', 'hunnt-ai'),
        $metrics['enabled_count'],
        $metrics['total_count']
    );
}
