<?php

function hunnt_ai_get_category($category_name = "", $lang = "en", $taxonomy = 'category')
{
    if (empty($category_name)) {
        return null;
    }

    // 1️⃣ Try to get category by name
    $term = get_term_by('name', $category_name, $taxonomy);

    if (!$term) {
        // 2️⃣ Insert new term
        $new_term = wp_insert_term($category_name, $taxonomy);

        if (!is_wp_error($new_term)) {
            $term_id = $new_term['term_id'];

            // 3️⃣ Assign language if Polylang is active
            if (function_exists('pll_set_term_language')) {
                pll_set_term_language($term_id, $lang);
            }

            // Reload full term object
            $term = get_term($term_id, $taxonomy);
        } else {
            // Error inserting
            return $new_term; // WP_Error
        }
    }

    return (array) $term; // return as array
}

function hunnt_ai_categories_list() {
    $categories_by_lang = [];

    // Get all categories
    $categories = get_terms([
        'taxonomy'   => 'category',
        'hide_empty' => false,
    ]);

    if (empty($categories) || is_wp_error($categories)) {
        return $categories_by_lang; // empty array
    }

    foreach ($categories as $category) {
        // Skip "uncategorized"
        if (strtolower($category->name) === "uncategorized") {
            continue;
        }

        $lang = 'default'; // fallback

        // Polylang check
        if (function_exists('pll_get_term_language')) {
            $cat_lang = pll_get_term_language($category->term_id);
            if (!empty($cat_lang)) {
                $lang = $cat_lang; // e.g. 'ar', 'en'
            }
        }

        // Add category to the proper language key
        if (!isset($categories_by_lang[$lang])) {
            $categories_by_lang[$lang] = [];
        }

        $categories_by_lang[$lang][] = [
            'id'   => $category->term_id,
            'name' => $category->name,
            'slug' => urldecode($category->slug),
        ];
    }

    return $categories_by_lang;
}

function hunnt_ai_get_post_language($post_id) {
    if (empty($post_id)) {
        return 'default';
    }

    // 1️⃣ Polylang
    if (function_exists('pll_get_post_language')) {
        $lang = pll_get_post_language($post_id);
        if (!empty($lang)) {
            return $lang;
        }
    }

    // 2️⃣ WPML
    if (function_exists('apply_filters')) {
        $lang_details = apply_filters('wpml_post_language_details', null, $post_id);
        if (!empty($lang_details['language_code'])) {
            return $lang_details['language_code'];
        }
    }

    // 3️⃣ Default fallback
    return 'default';
}


function hunnt_ai_get_post_categories($post_id) {
    $categories = get_the_terms($post_id, 'category');
    $result = [];

    if (!empty($categories) && !is_wp_error($categories)) {
        foreach ($categories as $cat) {
            // Skip "uncategorized"
            if (strtolower($cat->name) === "uncategorized") {
                continue;
            }
            $result[] = [
                'id'   => $cat->term_id,
                'name' => $cat->name,
                'slug' => urldecode($cat->slug),
            ];
        }
    }
    return $result;
}

/**
 * Get all active site languages from multilingual plugins
 *
 * @return array Array of languages in format: ['code' => 'Language Name']
 */
function hunnt_ai_get_site_languages() {
    $languages = [];

    // Polylang
    if (function_exists('pll_get_the_languages')) {
        $langs = pll_get_the_languages(['raw' => 1]);
        if (!empty($langs)) {
            foreach ($langs as $code => $lang) {
                $languages[$code] = $lang['name'];
            }
        }
    }

    // WPML
    elseif (defined('ICL_LANGUAGE_CODE') && function_exists('apply_filters')) {
        $langs = apply_filters('wpml_active_languages', null, ['skip_missing' => 0]);
        if (!empty($langs)) {
            foreach ($langs as $code => $lang) {
                $languages[$code] = $lang['native_name'];
            }
        }
    }

    // Fallback: default site language
    if (empty($languages)) {
        $languages[get_locale()] = get_bloginfo('language');
    }

    return $languages;
}
