<?php
// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

require_once HUNNT_AI_PATH . 'includes/optimization/asset-optimizer.php';
require_once HUNNT_AI_PATH . 'includes/optimization/css-processor.php';
require_once HUNNT_AI_PATH . 'includes/optimization/js-processor.php';

function hunnt_ai_get_uploads_optimized_paths() {
    $upload_dir = wp_upload_dir();
    $base_dir = trailingslashit($upload_dir['basedir']) . 'hunnt-ai-optimized';
    $base_url = trailingslashit($upload_dir['baseurl']) . 'hunnt-ai-optimized';

    return [
        'dir'      => $base_dir,
        'url'      => $base_url,
        'manifest' => $base_dir . '/manifest.json',
        'css'      => $base_dir . '/hunnt-ai-css-*.css',
        'js'       => $base_dir . '/hunnt-ai-js-*.js',
    ];
}

function hunnt_ai_get_minify_manifest() {
    $optimizer = new Hunnt_AI_Asset_Optimizer();
    $manifest = $optimizer->load_manifest();

    return is_array($manifest) ? $manifest : [];
}

function hunnt_ai_normalize_minify_manifest(?array $manifest = null): array {
    if (!is_array($manifest)) {
        $manifest = [];
    }

    if (!isset($manifest['timestamp'])) {
        $manifest['timestamp'] = current_time('mysql');
    }

    if (!isset($manifest['version'])) {
        $manifest['version'] = defined('HUNNT_AI_VERSION') ? HUNNT_AI_VERSION : '1.0.0';
    }

    if (!isset($manifest['site_url'])) {
        $manifest['site_url'] = home_url();
    }

    if (!isset($manifest['original']) || !is_array($manifest['original'])) {
        $manifest['original'] = [];
    }

    $manifest['original']['css'] = isset($manifest['original']['css']) && is_array($manifest['original']['css'])
        ? $manifest['original']['css']
        : [];
    $manifest['original']['js'] = isset($manifest['original']['js']) && is_array($manifest['original']['js'])
        ? $manifest['original']['js']
        : [];

    if (!isset($manifest['original']['skipped']) || !is_array($manifest['original']['skipped'])) {
        $manifest['original']['skipped'] = [];
    }

    $manifest['original']['skipped']['css'] = isset($manifest['original']['skipped']['css']) && is_array($manifest['original']['skipped']['css'])
        ? $manifest['original']['skipped']['css']
        : [];
    $manifest['original']['skipped']['js'] = isset($manifest['original']['skipped']['js']) && is_array($manifest['original']['skipped']['js'])
        ? $manifest['original']['skipped']['js']
        : [];

    if (!isset($manifest['optimized']) || !is_array($manifest['optimized'])) {
        $manifest['optimized'] = [];
    }

    $manifest['optimized']['css'] = isset($manifest['optimized']['css']) && is_array($manifest['optimized']['css'])
        ? $manifest['optimized']['css']
        : [];
    $manifest['optimized']['js'] = isset($manifest['optimized']['js']) && is_array($manifest['optimized']['js'])
        ? $manifest['optimized']['js']
        : [];

    $manifest['optimized']['css_skipped'] = isset($manifest['optimized']['css_skipped']) && is_array($manifest['optimized']['css_skipped'])
        ? $manifest['optimized']['css_skipped']
        : [];
    $manifest['optimized']['js_skipped'] = isset($manifest['optimized']['js_skipped']) && is_array($manifest['optimized']['js_skipped'])
        ? $manifest['optimized']['js_skipped']
        : [];

    if (!isset($manifest['optimized']['bundles']) || !is_array($manifest['optimized']['bundles'])) {
        $manifest['optimized']['bundles'] = [];
    }

    $manifest['optimized']['bundles']['css'] = isset($manifest['optimized']['bundles']['css']) && is_array($manifest['optimized']['bundles']['css'])
        ? $manifest['optimized']['bundles']['css']
        : [];
    $manifest['optimized']['bundles']['js'] = isset($manifest['optimized']['bundles']['js']) && is_array($manifest['optimized']['bundles']['js'])
        ? $manifest['optimized']['bundles']['js']
        : [];

    if (!isset($manifest['optimized']['meta']) || !is_array($manifest['optimized']['meta'])) {
        $manifest['optimized']['meta'] = [];
    }

    if (!isset($manifest['optimized']['meta']['css_collected'])) {
        $manifest['optimized']['meta']['css_collected'] = count($manifest['optimized']['css']);
    }

    if (!isset($manifest['optimized']['meta']['js_collected'])) {
        $manifest['optimized']['meta']['js_collected'] = count($manifest['optimized']['js']);
    }

    if (!isset($manifest['optimized']['meta']['last_run'])) {
        $manifest['optimized']['meta']['last_run'] = current_time('mysql');
    }

    return $manifest;
}

function hunnt_ai_minify_runtime_capture(
    Hunnt_AI_Asset_Optimizer $optimizer,
    array $manifest,
    array $assets_snapshot
): array {
    $updated = false;

    $optimized_dir = $optimizer->get_optimized_dir();
    $optimized_url = $optimizer->get_optimized_url();

    $css_assets_map = [];
    foreach ($assets_snapshot['css'] ?? [] as $asset) {
        if (!is_array($asset) || empty($asset['handle'])) {
            continue;
        }
        $css_assets_map[$asset['handle']] = $asset;
    }

    $js_assets_map = [];
    foreach ($assets_snapshot['js'] ?? [] as $asset) {
        if (!is_array($asset) || empty($asset['handle'])) {
            continue;
        }
        $js_assets_map[$asset['handle']] = $asset;
    }

    $known_css = [];

    if (!empty($manifest['optimized']['css'])) {
        $known_css = array_merge($known_css, array_keys($manifest['optimized']['css']));
    }

    foreach (($manifest['optimized']['bundles']['css'] ?? []) as $bundle) {
        if (!empty($bundle['original_handles']) && is_array($bundle['original_handles'])) {
            $known_css = array_merge($known_css, $bundle['original_handles']);
        }
    }

    foreach (($manifest['optimized']['css_skipped'] ?? []) as $skip) {
        if (!empty($skip['handle'])) {
            $known_css[] = $skip['handle'];
        }
    }

    foreach (($manifest['original']['skipped']['css'] ?? []) as $skip) {
        if (!empty($skip['handle'])) {
            $known_css[] = $skip['handle'];
        }
    }

    foreach (($assets_snapshot['skipped']['css'] ?? []) as $skip) {
        if (!empty($skip['handle'])) {
            $known_css[] = $skip['handle'];
        }
    }

    $known_css = array_values(array_unique(array_filter($known_css, 'is_string')));

    $missing_css_handles = array_diff(array_keys($css_assets_map), $known_css);

    $css_updates = [];
    $css_skipped = [];

    if (!empty($missing_css_handles)) {
        $css_subset = [];
        foreach ($missing_css_handles as $handle) {
            if (isset($css_assets_map[$handle])) {
                $css_subset[] = $css_assets_map[$handle];
            }
        }

        if (!empty($css_subset)) {
            $css_processor = new Hunnt_AI_CSS_Processor();
            $result = $css_processor->optimize_assets($css_subset, $optimized_dir, $optimized_url);

            if (!empty($result['optimized'])) {
                $css_updates = $result['optimized'];
                $updated = true;
            }

            if (!empty($result['skipped'])) {
                $css_skipped = $result['skipped'];
                $updated = true;
            }
        }
    }

    $known_js = [];

    if (!empty($manifest['optimized']['js'])) {
        $known_js = array_merge($known_js, array_keys($manifest['optimized']['js']));
    }

    foreach (($manifest['optimized']['bundles']['js'] ?? []) as $bundle) {
        if (!empty($bundle['original_handles']) && is_array($bundle['original_handles'])) {
            $known_js = array_merge($known_js, $bundle['original_handles']);
        }
    }

    foreach (($manifest['optimized']['js_skipped'] ?? []) as $skip) {
        if (!empty($skip['handle'])) {
            $known_js[] = $skip['handle'];
        }
    }

    foreach (($manifest['original']['skipped']['js'] ?? []) as $skip) {
        if (!empty($skip['handle'])) {
            $known_js[] = $skip['handle'];
        }
    }

    foreach (($assets_snapshot['skipped']['js'] ?? []) as $skip) {
        if (!empty($skip['handle'])) {
            $known_js[] = $skip['handle'];
        }
    }

    $known_js = array_values(array_unique(array_filter($known_js, 'is_string')));

    $missing_js_handles = array_diff(array_keys($js_assets_map), $known_js);

    $js_updates = [];
    $js_skipped = [];

    if (!empty($missing_js_handles)) {
        $js_subset = [];
        foreach ($missing_js_handles as $handle) {
            if (isset($js_assets_map[$handle])) {
                $js_subset[] = $js_assets_map[$handle];
            }
        }

        if (!empty($js_subset)) {
            $js_processor = new Hunnt_AI_JS_Processor();
            $result = $js_processor->optimize_assets($js_subset, $optimized_dir, $optimized_url);

            if (!empty($result['optimized'])) {
                $js_updates = $result['optimized'];
                $updated = true;
            }

            if (!empty($result['skipped'])) {
                $js_skipped = $result['skipped'];
                $updated = true;
            }
        }
    }

    if (!$updated) {
        return [
            'manifest' => $manifest,
            'updated'  => false,
        ];
    }

    $original_css_map = [];
    foreach ($manifest['original']['css'] as $entry) {
        if (!empty($entry['handle'])) {
            $original_css_map[$entry['handle']] = $entry;
        }
    }

    foreach ($css_assets_map as $handle => $entry) {
        if (isset($css_updates[$handle])) {
            $original_css_map[$handle] = $entry;
        }
    }

    $original_js_map = [];
    foreach ($manifest['original']['js'] as $entry) {
        if (!empty($entry['handle'])) {
            $original_js_map[$entry['handle']] = $entry;
        }
    }

    foreach ($js_assets_map as $handle => $entry) {
        if (isset($js_updates[$handle])) {
            $original_js_map[$handle] = $entry;
        }
    }

    $manifest['original']['css'] = array_values($original_css_map);
    $manifest['original']['js'] = array_values($original_js_map);

    $css_skip_map = [];
    foreach ($manifest['optimized']['css_skipped'] as $skip) {
        if (!empty($skip['handle'])) {
            $css_skip_map[$skip['handle']] = $skip;
        }
    }

    foreach ($css_skipped as $skip) {
        if (!empty($skip['handle'])) {
            $css_skip_map[$skip['handle']] = $skip;
        }
    }

    $manifest['optimized']['css_skipped'] = array_values($css_skip_map);

    $js_skip_map = [];
    foreach ($manifest['optimized']['js_skipped'] as $skip) {
        if (!empty($skip['handle'])) {
            $js_skip_map[$skip['handle']] = $skip;
        }
    }

    foreach ($js_skipped as $skip) {
        if (!empty($skip['handle'])) {
            $js_skip_map[$skip['handle']] = $skip;
        }
    }

    $manifest['optimized']['js_skipped'] = array_values($js_skip_map);

    foreach ($css_updates as $handle => $data) {
        $manifest['optimized']['css'][$handle] = $data;
    }

    foreach ($js_updates as $handle => $data) {
        $manifest['optimized']['js'][$handle] = $data;
    }

    $manifest['optimized']['meta']['css_collected'] = count($manifest['optimized']['css']);
    $manifest['optimized']['meta']['js_collected'] = count($manifest['optimized']['js']);
    $manifest['optimized']['meta']['last_run'] = current_time('mysql');
    $manifest['timestamp'] = current_time('mysql');

    if (!empty($assets_snapshot['skipped']['css'])) {
        $skip_map = [];
        foreach ($manifest['original']['skipped']['css'] as $skip) {
            if (!empty($skip['handle'])) {
                $skip_map[$skip['handle']] = $skip;
            }
        }
        foreach ($assets_snapshot['skipped']['css'] as $skip) {
            if (!empty($skip['handle'])) {
                $skip_map[$skip['handle']] = $skip;
            }
        }
        $manifest['original']['skipped']['css'] = array_values($skip_map);
    }

    if (!empty($assets_snapshot['skipped']['js'])) {
        $skip_map = [];
        foreach ($manifest['original']['skipped']['js'] as $skip) {
            if (!empty($skip['handle'])) {
                $skip_map[$skip['handle']] = $skip;
            }
        }
        foreach ($assets_snapshot['skipped']['js'] as $skip) {
            if (!empty($skip['handle'])) {
                $skip_map[$skip['handle']] = $skip;
            }
        }
        $manifest['original']['skipped']['js'] = array_values($skip_map);
    }

    $optimizer->persist_manifest($manifest);

    return [
        'manifest' => $manifest,
        'updated'  => true,
    ];
}

function hunnt_ai_is_cache_enabled() {
    if (defined('WP_CACHE') && WP_CACHE) {
        return true;
    }
    if (function_exists('wp_using_ext_object_cache') && wp_using_ext_object_cache()) {
        return true;
    }
    if (get_option('hunnt_ai_basic_cache_enabled')) {
        return true;
    }
    // Avoid slow self-HTTP calls by default. Opt-in via filter.
    if (apply_filters('hunnt_ai_detect_cache_via_http', false)) {
        $response = wp_remote_head(site_url(), [ 'timeout' => 1, 'redirection' => 0 ]);
        if (!is_wp_error($response)) {
            $headers = wp_remote_retrieve_headers($response);
            if (!empty($headers['cache-control'])) {
                return true;
            }
        }
    }
    return false;
}

function hunnt_ai_static_cache_is_enabled(): bool {
    if (!class_exists('Hunnt_AI_Static_Cache_Manager')) {
        return false;
    }

    return Hunnt_AI_Static_Cache_Manager::is_enabled();
}

function hunnt_ai_enable_static_cache(): bool {
    if (!class_exists('Hunnt_AI_Static_Cache_Manager')) {
        return false;
    }

    if (function_exists('hunnt_ai_is_technical_seo_enabled') && !hunnt_ai_is_technical_seo_enabled()) {
        return false;
    }

    Hunnt_AI_Static_Cache_Manager::update_enabled(true);

    return Hunnt_AI_Static_Cache_Manager::is_enabled();
}

function hunnt_ai_disable_static_cache(): bool {
    if (!class_exists('Hunnt_AI_Static_Cache_Manager')) {
        return false;
    }

    Hunnt_AI_Static_Cache_Manager::update_enabled(false);

    return true;
}

function hunnt_ai_static_cache_purge_all(): bool {
    if (!class_exists('Hunnt_AI_Static_Cache_Manager')) {
        return false;
    }

    return Hunnt_AI_Static_Cache_Manager::purge_all();
}

function hunnt_ai_static_cache_warm_urls(array $urls = []): array {
    if (!class_exists('Hunnt_AI_Static_Cache_Manager')) {
        return [];
    }

    if (function_exists('hunnt_ai_is_technical_seo_enabled') && !hunnt_ai_is_technical_seo_enabled()) {
        return [];
    }

    if (empty($urls)) {
        $urls = [home_url('/')];
    }

    return Hunnt_AI_Static_Cache_Manager::warm_urls($urls);
}

function hunnt_ai_get_static_cache_summary(): array {
    if (class_exists('Hunnt_AI_Static_Cache_Helper')) {
        Hunnt_AI_Static_Cache_Helper::ensure_directories();
    }

    $enabled = hunnt_ai_static_cache_is_enabled();
    $wp_cache = defined('WP_CACHE') && WP_CACHE;
    $dropin_path = WP_CONTENT_DIR . '/advanced-cache.php';
    $dropin_active = false;

    if (file_exists($dropin_path)) {
        $contents = file_get_contents($dropin_path);
        if ($contents !== false && stripos($contents, Hunnt_AI_Static_Cache_Helper::DROPIN_SIGNATURE) !== false) {
            $dropin_active = true;
        }
    }

    $inventory = class_exists('Hunnt_AI_Static_Cache_Manager')
        ? Hunnt_AI_Static_Cache_Manager::get_cache_inventory()
        : ['count' => 0, 'bytes' => 0];

    $config = [];
    $config_path = Hunnt_AI_Static_Cache_Helper::get_config_path();
    if (is_readable($config_path)) {
        $maybe = include $config_path;
        if (is_array($maybe)) {
            $config = $maybe;
        }
    }

    $ttl = isset($config['ttl']) ? (int) $config['ttl'] : Hunnt_AI_Static_Cache_Helper::get_default_ttl();
    $generated = isset($config['generated']) ? (int) $config['generated'] : 0;
    $technical_seo_enabled = function_exists('hunnt_ai_is_technical_seo_enabled') ? (bool) hunnt_ai_is_technical_seo_enabled() : true;

    $directories = class_exists('Hunnt_AI_Static_Cache_Helper')
        ? Hunnt_AI_Static_Cache_Helper::get_directories_status()
        : [];

    $has_write_access = true;
    foreach ($directories as $meta) {
        if (empty($meta['writable'])) {
            $has_write_access = false;
            break;
        }
    }

    $last_error = class_exists('Hunnt_AI_Static_Cache_Helper')
        ? Hunnt_AI_Static_Cache_Helper::get_last_fs_issue()
        : null;

    if ($last_error) {
        $has_write_access = false;
    }

    if (!$technical_seo_enabled) {
        $enabled = false;
    }

    return [
        'enabled'        => $enabled,
        'technical_seo_enabled' => $technical_seo_enabled,
        'wp_cache'       => $wp_cache,
        'dropin_active'  => $dropin_active,
        'cache_dir'      => Hunnt_AI_Static_Cache_Helper::get_cache_root_dir(),
        'ttl'            => $ttl,
        'last_generated' => $generated,
        'inventory'      => $inventory,
        'directories'    => $directories,
        'has_write_access' => $has_write_access,
        'last_error'     => $last_error,
    ];
}

function hunnt_ai_is_minify_css_enabled() {
    $manifest = hunnt_ai_get_minify_manifest();
    return !empty($manifest['optimized']['css']);
}

function hunnt_ai_is_minify_js_enabled() {
    $manifest = hunnt_ai_get_minify_manifest();
    return !empty($manifest['optimized']['js']);
}

function hunnt_ai_is_asset_aggregation_enabled() {
    return (bool) get_option('hunnt_ai_assets_aggregate_enable', 1);
}

function hunnt_ai_is_minify_html_enabled() {
    return (bool) get_option('hunnt_ai_html_minify_enabled', false);
}

function hunnt_ai_get_assets_min_ratio(): float {
    $default_ratio = apply_filters('hunnt_ai_default_assets_min_ratio', 0.15);

    $ratio = max(0.0, min(1.0, (float) $default_ratio));

    return (float) apply_filters('hunnt_ai_assets_min_ratio_value', $ratio);
}

function hunnt_ai_asset_is_minified(array $asset): bool {
    if (!isset($asset['status'])) {
        return false;
    }

    $status = (string) $asset['status'];

    return in_array($status, ['minified', 'minified_low_gain'], true);
}

function hunnt_ai_asset_passthrough_reason(array $asset): string {
    $status = isset($asset['status']) ? (string) $asset['status'] : '';

    if (in_array($status, ['minified', 'minified_low_gain'], true)) {
        return 'minified';
    }

    $warnings = [];

    if (isset($asset['warnings']) && is_array($asset['warnings'])) {
        $warnings = array_map('strval', $asset['warnings']);
    }

    if (in_array('passthrough_already_minified', $warnings, true)) {
        return 'already_minified';
    }

    if ($status === 'passthrough_threshold') {
        return 'below_threshold';
    }

    if ($status === 'passthrough_not_smaller') {
        return 'not_smaller';
    }

    if ($status === 'fallback_original') {
        return 'fallback';
    }

    if ($status === 'passthrough') {
        if (in_array('passthrough_handle_passthrough', $warnings, true)) {
            return 'handle_passthrough';
        }

        if (in_array('passthrough_filtered_passthrough', $warnings, true)) {
            return 'filtered_passthrough';
        }
    }

    return $status !== '' ? $status : 'other';
}

function hunnt_ai_is_gzip_enabled() {
    if (function_exists('hunnt_ai_check_gzip_active')) {
        return hunnt_ai_check_gzip_active();
    }
    return false;
}

function hunnt_ai_get_technical_seo_status() {
    // If Technical SEO is globally disabled, return all features as disabled
    if (!hunnt_ai_is_technical_seo_enabled()) {
        return [
            'cache'       => false,
            'minify_css'  => false,
            'minify_js'   => false,
            'minify_html' => false,
            'gzip'        => false,
            'static_cache'=> false,
        ];
    }
    
    // Otherwise, return actual status of each feature
    return [
        'cache'       => hunnt_ai_is_cache_enabled(),
        'minify_css'  => hunnt_ai_is_minify_css_enabled(),
        'minify_js'   => hunnt_ai_is_minify_js_enabled(),
        'minify_html' => hunnt_ai_is_minify_html_enabled(),
        'gzip'        => hunnt_ai_is_gzip_enabled(),
        'static_cache'=> hunnt_ai_static_cache_is_enabled(),
    ];
}

function hunnt_ai_send_basic_cache_headers() {
    if (is_admin()) {
        return;
    }
    // Only set if not already set by other plugins/server
    if (!headers_sent()) {
        if (!headers_list() || !preg_grep('/^Cache-Control:/i', headers_list())) {
            header('Cache-Control: public, max-age=600'); // 10 minutes
        }
    }
}

function hunnt_ai_enable_basic_cache_headers() {
    if (!get_option('hunnt_ai_basic_cache_enabled')) {
        update_option('hunnt_ai_basic_cache_enabled', 1);
    }
    add_action('send_headers', 'hunnt_ai_send_basic_cache_headers');
}

function hunnt_ai_minify_html_buffer_callback($html) {
    try {
        if (trim($html) === '') return $html;
        // Preserve pre, textarea, script, style blocks
        $placeholders = [];
        $blocks = ['pre', 'textarea', 'script', 'style'];
        foreach ($blocks as $tag) {
            $pattern = '#<'.$tag.'\b[^>]*>.*?</'.$tag.'>#si';
            $html = preg_replace_callback($pattern, function($m) use (&$placeholders, $tag) {
                $key = '%%HUNNT_AI_BLOCK_'.strtoupper($tag).'_'.count($placeholders).'%%';
                $placeholders[$key] = $m[0];
                return $key;
            }, $html);
        }
        // Remove HTML comments except IE conditionals
        $html = preg_replace('/<!--(?!\s*\[if|<!|>).*?-->/', '', $html);
        // Collapse whitespace
        $html = preg_replace('/\s{2,}/', ' ', $html);
        $html = preg_replace('/>\s+</', '><', $html);
        // Restore placeholders
        if (!empty($placeholders)) {
            $html = strtr($html, $placeholders);
        }
        return $html;
    } catch (\Throwable $e) {
        return $html; // Safety
    }
}

function hunnt_ai_enable_html_minify() {
    if (!get_option('hunnt_ai_html_minify_enabled')) {
        update_option('hunnt_ai_html_minify_enabled', 1);
    }
    add_action('template_redirect', function() {
        if (is_admin()) return;
        if (!apply_filters('hunnt_ai_minify_html_should_run', true)) return;
        if (!headers_sent()) {
            ob_start('hunnt_ai_minify_html_buffer_callback');
        }
    });
}

function hunnt_ai_apply_minify_assets() {
    global $hunnt_ai_minify_building_depth;

    if (!isset($hunnt_ai_minify_building_depth) || !is_int($hunnt_ai_minify_building_depth)) {
        $hunnt_ai_minify_building_depth = 0;
    }

    $hunnt_ai_minify_building_depth++;

    $upload_dir = wp_upload_dir();
    $log_file = $upload_dir['basedir'] . '/hunnt-ai-optimized/error-log.json';

    try {
        if (function_exists('hunnt_ai_runtime_minify_prepare_environment')) {
            hunnt_ai_runtime_minify_prepare_environment();
        }

        $optimizer      = new Hunnt_AI_Asset_Optimizer();
        $css_processor  = new Hunnt_AI_CSS_Processor();
        $js_processor   = new Hunnt_AI_JS_Processor();

        $assets = $optimizer->collect_assets();

        $run_timestamp = current_time('mysql');

        $debug_info = [
            'timestamp'           => $run_timestamp,
            'css_count'           => count($assets['css']),
            'js_count'            => count($assets['js']),
            'css_handles'         => array_column($assets['css'], 'handle'),
            'js_handles'          => array_column($assets['js'], 'handle'),
            'skipped_collections' => $assets['skipped'] ?? [],
        ];

        $optimized_info = [
            'css'         => [],
            'js'          => [],
            'css_skipped' => $assets['skipped']['css'] ?? [],
            'js_skipped'  => $assets['skipped']['js'] ?? [],
            'meta'        => [
                'css_collected' => count($assets['css']),
                'js_collected'  => count($assets['js']),
                'last_run'      => $run_timestamp,
            ],
        ];

        if (!empty($assets['css'])) {
            $css_result = $css_processor->optimize_assets(
                $assets['css'],
                $optimizer->get_optimized_dir(),
                $optimizer->get_optimized_url()
            );

            $optimized_info['css'] = $css_result['optimized'];
            $optimized_info['css_skipped'] = array_merge($optimized_info['css_skipped'], $css_result['skipped']);

            if (!empty($css_result['optimized'])) {
                $debug_info['css_optimized_handles'] = array_keys($css_result['optimized']);
            } else {
                $debug_info['css_optimization_warning'] = __('CSS optimization completed but no files written', 'hunnt-ai');
            }
        } else {
            $debug_info['css_empty'] = __('No CSS assets detected', 'hunnt-ai');
        }

        if (!empty($assets['js'])) {
            $js_result = $js_processor->optimize_assets(
                $assets['js'],
                $optimizer->get_optimized_dir(),
                $optimizer->get_optimized_url()
            );

            $optimized_info['js'] = $js_result['optimized'];
            $optimized_info['js_skipped'] = array_merge($optimized_info['js_skipped'], $js_result['skipped']);

            if (!empty($js_result['optimized'])) {
                $debug_info['js_optimized_handles'] = array_keys($js_result['optimized']);
            } else {
                $debug_info['js_optimization_warning'] = __('JS optimization completed but no files written', 'hunnt-ai');
            }
        } else {
            $debug_info['js_empty'] = __('No JS assets detected', 'hunnt-ai');
        }

        $optimized_info['bundles'] = [
            'css' => [],
            'js'  => [],
        ];

        if (hunnt_ai_is_asset_aggregation_enabled()) {
            if (!empty($optimized_info['css'])) {
                $css_bundles = $css_processor->aggregate_bundles(
                    $optimized_info['css'],
                    $optimizer->get_optimized_dir(),
                    $optimizer->get_optimized_url()
                );

                if (!empty($css_bundles['bundles'])) {
                    $optimized_info['bundles']['css'] = $css_bundles['bundles'];
                    $debug_info['css_bundle_handles'] = array_keys($css_bundles['bundles']);
                }

                if (!empty($css_bundles['skipped'])) {
                    $optimized_info['css_skipped'] = array_merge($optimized_info['css_skipped'], $css_bundles['skipped']);
                }
            }

            if (!empty($optimized_info['js'])) {
                $js_bundles = $js_processor->aggregate_bundles(
                    $optimized_info['js'],
                    $optimizer->get_optimized_dir(),
                    $optimizer->get_optimized_url()
                );

                if (!empty($js_bundles['bundles'])) {
                    $optimized_info['bundles']['js'] = $js_bundles['bundles'];
                    $debug_info['js_bundle_handles'] = array_keys($js_bundles['bundles']);
                }

                if (!empty($js_bundles['skipped'])) {
                    $optimized_info['js_skipped'] = array_merge($optimized_info['js_skipped'], $js_bundles['skipped']);
                }
            }

            if (empty($optimized_info['bundles']['css']) && empty($optimized_info['bundles']['js'])) {
                $debug_info['aggregation_warning'] = __('Asset aggregation enabled but no bundles were generated.', 'hunnt-ai');
            }
        }

        $summary = [
            'css_optimized' => count($optimized_info['css']),
            'js_optimized'  => count($optimized_info['js']),
            'css_skipped'   => count($optimized_info['css_skipped']),
            'js_skipped'    => count($optimized_info['js_skipped']),
            'css_bundles'   => count($optimized_info['bundles']['css']),
            'js_bundles'    => count($optimized_info['bundles']['js']),
        ];

        $bytes_stats = [
            'css' => [
                'original'  => 0,
                'optimized' => 0,
            ],
            'js'  => [
                'original'  => 0,
                'optimized' => 0,
            ],
        ];

        $eligible_stats = [
            'css' => [
                'original'  => 0,
                'optimized' => 0,
            ],
            'js'  => [
                'original'  => 0,
                'optimized' => 0,
            ],
        ];

        $passthrough_stats = [
            'css' => [
                'total'                => 0,
                'already_minified'     => 0,
                'below_threshold'      => 0,
                'not_smaller'          => 0,
                'fallback'             => 0,
                'handle_passthrough'   => 0,
                'filtered_passthrough' => 0,
                'other'                => 0,
            ],
            'js'  => [
                'total'                => 0,
                'already_minified'     => 0,
                'below_threshold'      => 0,
                'not_smaller'          => 0,
                'fallback'             => 0,
                'handle_passthrough'   => 0,
                'filtered_passthrough' => 0,
                'other'                => 0,
            ],
        ];

        $bundle_bytes = [
            'css' => [
                'original'  => 0,
                'optimized' => 0,
            ],
            'js'  => [
                'original'  => 0,
                'optimized' => 0,
            ],
        ];

        foreach ($optimized_info['css'] as $data) {
            $original = isset($data['original_bytes']) ? (int) $data['original_bytes'] : 0;
            $optimized = isset($data['optimized_bytes']) ? (int) $data['optimized_bytes'] : $original;

            $bytes_stats['css']['original']  += $original;
            $bytes_stats['css']['optimized'] += $optimized;

            if (hunnt_ai_asset_is_minified($data)) {
                $eligible_stats['css']['original']  += $original;
                $eligible_stats['css']['optimized'] += $optimized;
            } else {
                $reason = hunnt_ai_asset_passthrough_reason($data);
                $passthrough_stats['css']['total'] += $original;

                if (isset($passthrough_stats['css'][$reason])) {
                    $passthrough_stats['css'][$reason] += $original;
                } else {
                    $passthrough_stats['css']['other'] += $original;
                }
            }
        }

        foreach ($optimized_info['js'] as $data) {
            $original = isset($data['original_bytes']) ? (int) $data['original_bytes'] : 0;
            $optimized = isset($data['optimized_bytes']) ? (int) $data['optimized_bytes'] : $original;

            $bytes_stats['js']['original']  += $original;
            $bytes_stats['js']['optimized'] += $optimized;

            if (hunnt_ai_asset_is_minified($data)) {
                $eligible_stats['js']['original']  += $original;
                $eligible_stats['js']['optimized'] += $optimized;
            } else {
                $reason = hunnt_ai_asset_passthrough_reason($data);
                $passthrough_stats['js']['total'] += $original;

                if (isset($passthrough_stats['js'][$reason])) {
                    $passthrough_stats['js'][$reason] += $original;
                } else {
                    $passthrough_stats['js']['other'] += $original;
                }
            }
        }

        foreach ($optimized_info['bundles']['css'] as $bundle) {
            $bundle_bytes['css']['original']  += (int) ($bundle['original_bytes'] ?? 0);
            $bundle_bytes['css']['optimized'] += (int) ($bundle['optimized_bytes'] ?? 0);
        }

        foreach ($optimized_info['bundles']['js'] as $bundle) {
            $bundle_bytes['js']['original']  += (int) ($bundle['original_bytes'] ?? 0);
            $bundle_bytes['js']['optimized'] += (int) ($bundle['optimized_bytes'] ?? 0);
        }

        $summary['css_bytes_original']   = $bytes_stats['css']['original'];
        $summary['css_bytes_optimized']  = $bytes_stats['css']['optimized'];
        $summary['css_bytes_saved']      = max(0, $bytes_stats['css']['original'] - $bytes_stats['css']['optimized']);
        $summary['css_reduction_ratio']  = $bytes_stats['css']['original'] > 0
            ? round(1 - ($bytes_stats['css']['optimized'] / $bytes_stats['css']['original']), 6)
            : 0;

        $summary['css_bytes_unoptimizable']       = $passthrough_stats['css']['total'];
        $summary['css_passthrough_breakdown']     = $passthrough_stats['css'];
        $summary['css_passthrough_bytes']         = $passthrough_stats['css']['total'];
        $summary['css_effective_bytes_original']  = $eligible_stats['css']['original'];
        $summary['css_effective_bytes_optimized'] = $eligible_stats['css']['optimized'];
        $summary['css_effective_bytes_saved']     = max(0, $eligible_stats['css']['original'] - $eligible_stats['css']['optimized']);
        $summary['css_effective_reduction_ratio'] = $eligible_stats['css']['original'] > 0
            ? round(1 - ($eligible_stats['css']['optimized'] / $eligible_stats['css']['original']), 6)
            : 0;

        $summary['js_bytes_original']    = $bytes_stats['js']['original'];
        $summary['js_bytes_optimized']   = $bytes_stats['js']['optimized'];
        $summary['js_bytes_saved']       = max(0, $bytes_stats['js']['original'] - $bytes_stats['js']['optimized']);
        $summary['js_reduction_ratio']   = $bytes_stats['js']['original'] > 0
            ? round(1 - ($bytes_stats['js']['optimized'] / $bytes_stats['js']['original']), 6)
            : 0;

        $summary['js_bytes_unoptimizable']       = $passthrough_stats['js']['total'];
        $summary['js_passthrough_breakdown']     = $passthrough_stats['js'];
        $summary['js_passthrough_bytes']         = $passthrough_stats['js']['total'];
        $summary['js_effective_bytes_original']  = $eligible_stats['js']['original'];
        $summary['js_effective_bytes_optimized'] = $eligible_stats['js']['optimized'];
        $summary['js_effective_bytes_saved']     = max(0, $eligible_stats['js']['original'] - $eligible_stats['js']['optimized']);
        $summary['js_effective_reduction_ratio'] = $eligible_stats['js']['original'] > 0
            ? round(1 - ($eligible_stats['js']['optimized'] / $eligible_stats['js']['original']), 6)
            : 0;

        $total_passthrough_bytes = $passthrough_stats['css']['total'] + $passthrough_stats['js']['total'];
        $summary['total_passthrough_bytes'] = $total_passthrough_bytes;
        $summary['passthrough_breakdown'] = [
            'css' => $passthrough_stats['css'],
            'js'  => $passthrough_stats['js'],
        ];

        $total_effective_original  = $eligible_stats['css']['original'] + $eligible_stats['js']['original'];
        $total_effective_optimized = $eligible_stats['css']['optimized'] + $eligible_stats['js']['optimized'];
        $summary['total_effective_bytes_original']  = $total_effective_original;
        $summary['total_effective_bytes_optimized'] = $total_effective_optimized;
        $summary['total_effective_bytes_saved']     = max(0, $total_effective_original - $total_effective_optimized);
        $summary['total_effective_reduction_ratio'] = $total_effective_original > 0
            ? round(1 - ($total_effective_optimized / $total_effective_original), 6)
            : 0;

        if ($summary['css_bundles'] > 0) {
            $summary['css_bundle_bytes_original']  = $bundle_bytes['css']['original'];
            $summary['css_bundle_bytes_optimized'] = $bundle_bytes['css']['optimized'];
            $summary['css_bundle_bytes_saved']     = max(0, $bundle_bytes['css']['original'] - $bundle_bytes['css']['optimized']);
        }

        if ($summary['js_bundles'] > 0) {
            $summary['js_bundle_bytes_original']  = $bundle_bytes['js']['original'];
            $summary['js_bundle_bytes_optimized'] = $bundle_bytes['js']['optimized'];
            $summary['js_bundle_bytes_saved']     = max(0, $bundle_bytes['js']['original'] - $bundle_bytes['js']['optimized']);
        }

        $total_original  = $bytes_stats['css']['original'] + $bytes_stats['js']['original'];
        $total_optimized = $bytes_stats['css']['optimized'] + $bytes_stats['js']['optimized'];

        $summary['total_bytes_saved']    = max(0, $total_original - $total_optimized);
        $summary['total_reduction_ratio'] = $total_original > 0
            ? round(1 - ($total_optimized / $total_original), 6)
            : 0;

        $minimum_target_ratio = hunnt_ai_get_assets_min_ratio();
        $minimum_target_ratio = apply_filters(
            'hunnt_ai_assets_minimum_reduction_ratio',
            $minimum_target_ratio,
            $summary,
            $optimized_info
        );

        if (!is_numeric($minimum_target_ratio)) {
            $minimum_target_ratio = hunnt_ai_get_assets_min_ratio();
        }

        $minimum_target_ratio = max(0.0, min(1.0, (float) $minimum_target_ratio));
        $reduction_basis = 'total';
        $meets_reduction_target = $summary['total_reduction_ratio'] >= $minimum_target_ratio;

        if ($summary['total_effective_bytes_original'] > 0) {
            $reduction_basis = 'effective';
            $meets_reduction_target = $summary['total_effective_reduction_ratio'] >= $minimum_target_ratio;
        } elseif ($total_original > 0 && $summary['total_passthrough_bytes'] > 0) {
            $reduction_basis = 'passthrough';
            $meets_reduction_target = true;
        }

        $summary['minimum_target_ratio'] = $minimum_target_ratio;
        $summary['reduction_basis'] = $reduction_basis;
        $summary['meets_reduction_target'] = $meets_reduction_target;

        $optimized_info['meta']['bundles'] = [
            'css' => $summary['css_bundles'],
            'js'  => $summary['js_bundles'],
        ];
        $optimized_info['meta']['summary'] = $summary;
        $optimized_info['meta']['targets'] = [
            'minimum_ratio' => $minimum_target_ratio,
            'met'           => $summary['meets_reduction_target'],
            'basis'         => $reduction_basis,
        ];

        $has_optimized_assets = ($summary['css_optimized'] + $summary['js_optimized']) > 0;
        $success = $has_optimized_assets && $summary['meets_reduction_target'];

        $debug_info['optimized_info'] = $optimized_info;
        $debug_info['success'] = $success;
        $debug_info['minimum_target_ratio'] = $minimum_target_ratio;
        $debug_info['total_reduction_ratio'] = $summary['total_reduction_ratio'];
        $debug_info['total_effective_reduction_ratio'] = $summary['total_effective_reduction_ratio'];
        $debug_info['total_passthrough_bytes'] = $summary['total_passthrough_bytes'];
        $debug_info['reduction_basis'] = $reduction_basis;

        if (!$has_optimized_assets) {
            $debug_info['failure_reason'] = 'no_assets_optimized';
        } elseif (!$summary['meets_reduction_target']) {
            $debug_info['failure_reason'] = 'minimum_reduction_not_met';
        }

        $last_debug_snapshot = [
            'timestamp'             => $run_timestamp,
            'success'               => $success,
            'failure_reason'        => $debug_info['failure_reason'] ?? null,
            'minimum_target_ratio'  => $minimum_target_ratio,
            'total_reduction_ratio' => $summary['total_reduction_ratio'],
            'total_effective_reduction_ratio' => $summary['total_effective_reduction_ratio'] ?? 0,
            'reduction_basis'      => $reduction_basis,
            'css_reduction_ratio'   => $summary['css_reduction_ratio'] ?? 0,
            'js_reduction_ratio'    => $summary['js_reduction_ratio'] ?? 0,
            'total_bytes_saved'     => $summary['total_bytes_saved'] ?? 0,
            'css_bytes_saved'       => $summary['css_bytes_saved'] ?? 0,
            'js_bytes_saved'        => $summary['js_bytes_saved'] ?? 0,
            'total_effective_bytes_saved' => $summary['total_effective_bytes_saved'] ?? 0,
        ];

        update_option('hunnt_ai_assets_last_summary', $summary);
        update_option('hunnt_ai_assets_last_debug', $last_debug_snapshot);

        $optimizer->log_action('apply_minify_assets', $debug_info);

        if ($success) {
            $optimizer->save_manifest($assets, $optimized_info);
            update_option('hunnt_ai_assets_minify_enable', 1);
            update_option('hunnt_ai_assets_minified', time());
            hunnt_ai_schedule_technical_seo_warmup();
        } else {
            $optimizer->clear_cache();
            update_option('hunnt_ai_assets_minify_enable', 0);
            delete_option('hunnt_ai_assets_minified');
        }

        if (!file_exists(dirname($log_file))) {
            wp_mkdir_p(dirname($log_file));
        }

        file_put_contents($log_file, wp_json_encode([
            'last_run'  => $debug_info,
            'timestamp' => current_time('mysql'),
        ], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));

        return $success;

    } catch (Exception $e) {
        $error_data = [
            'timestamp' => current_time('mysql'),
            'error'     => $e->getMessage(),
            'file'      => $e->getFile(),
            'line'      => $e->getLine(),
            'trace'     => $e->getTraceAsString(),
        ];

        if (!file_exists(dirname($log_file))) {
            wp_mkdir_p(dirname($log_file));
        }

        file_put_contents($log_file, wp_json_encode([
            'exception' => $error_data,
            'timestamp' => current_time('mysql'),
        ], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));

        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('Hunnt AI: Failed to apply minify assets: ' . $e->getMessage());
        }

        return false;

    } catch (Throwable $e) {
        $error_data = [
            'timestamp' => current_time('mysql'),
            'error'     => $e->getMessage(),
            'file'      => $e->getFile(),
            'line'      => $e->getLine(),
            'trace'     => $e->getTraceAsString(),
        ];

        if (!file_exists(dirname($log_file))) {
            wp_mkdir_p(dirname($log_file));
        }

        file_put_contents($log_file, wp_json_encode([
            'throwable' => $error_data,
            'timestamp' => current_time('mysql'),
        ], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));

        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('Hunnt AI: Throwable error in apply minify assets: ' . $e->getMessage());
        }

        return false;
    } finally {
        $hunnt_ai_minify_building_depth--;
        if ($hunnt_ai_minify_building_depth <= 0) {
            $hunnt_ai_minify_building_depth = 0;
        }
    }
}

function hunnt_ai_manifest_has_assets(string $type): bool {
    $manifest = hunnt_ai_get_minify_manifest();

    if ($type === 'css' && !empty($manifest['optimized']['css'])) {
        return true;
    }

    if ($type === 'css' && !empty($manifest['optimized']) && !empty($manifest['optimized']['bundles']) && !empty($manifest['optimized']['bundles']['css'])) {
        return true;
    }

    if ($type === 'js' && !empty($manifest['optimized']['js'])) {
        return true;
    }

    if ($type === 'js' && !empty($manifest['optimized']) && !empty($manifest['optimized']['bundles']) && !empty($manifest['optimized']['bundles']['js'])) {
        return true;
    }

    if (!empty($manifest['entries'])) {
        foreach ($manifest['entries'] as $entry) {
            if (($entry['type'] ?? '') === $type && !empty($entry['path']) && file_exists($entry['path'])) {
                return true;
            }
        }
    }

    if (!empty($manifest['skipped'])) {
        foreach ($manifest['skipped'] as $entry) {
            if (($entry['type'] ?? '') === $type) {
                if (($entry['reason'] ?? '') === 'already_minified') {
                    return true;
                }
                if (!empty($entry['local_path']) && file_exists($entry['local_path'])) {
                    return true;
                }
            }
        }
    }

    return false;
}

function hunnt_ai_get_minify_summary(): array {
    $manifest  = hunnt_ai_get_minify_manifest();
    $optimized = $manifest['optimized'] ?? [];

    $summary = [
        'generated_at' => $manifest['timestamp'] ?? null,
        'updated_at'   => $manifest['optimized']['meta']['last_run'] ?? ($manifest['timestamp'] ?? null),
        'totals'       => [
            'css' => ['entries' => 0, 'skipped' => 0, 'failures' => 0],
            'js'  => ['entries' => 0, 'skipped' => 0, 'failures' => 0],
            'css_bundles' => ['entries' => 0],
            'js_bundles'  => ['entries' => 0],
        ],
        'skipped'      => [],
        'failures'     => [],
    ];

    if (!empty($optimized['css'])) {
        $summary['totals']['css']['entries'] = count($optimized['css']);
    }

    if (!empty($optimized['js'])) {
        $summary['totals']['js']['entries'] = count($optimized['js']);
    }

    if (!empty($optimized['bundles']) && !empty($optimized['bundles']['css'])) {
        $summary['totals']['css_bundles']['entries'] = count($optimized['bundles']['css']);
    }

    if (!empty($optimized['bundles']) && !empty($optimized['bundles']['js'])) {
        $summary['totals']['js_bundles']['entries'] = count($optimized['bundles']['js']);
    }

    if (!empty($optimized['css_skipped'])) {
        $summary['totals']['css']['skipped'] = count($optimized['css_skipped']);
        $summary['skipped'] = array_merge($summary['skipped'], $optimized['css_skipped']);
    }

    if (!empty($optimized['js_skipped'])) {
        $summary['totals']['js']['skipped'] = count($optimized['js_skipped']);
        $summary['skipped'] = array_merge($summary['skipped'], $optimized['js_skipped']);
    }

    // Backward compatibility: process legacy manifest structures when present.
    if (!empty($manifest['entries'])) {
        foreach ($manifest['entries'] as $entry) {
            $type = $entry['type'] ?? null;
            if ($type && isset($summary['totals'][$type])) {
                $summary['totals'][$type]['entries']++;
            }
        }
    }

    if (!empty($manifest['skipped'])) {
        foreach ($manifest['skipped'] as $entry) {
            $type = $entry['type'] ?? null;
            if ($type && isset($summary['totals'][$type])) {
                $summary['totals'][$type]['skipped']++;
            }
            $summary['skipped'][] = [
                'handle' => $entry['handle'] ?? null,
                'type'   => $type,
                'reason' => $entry['reason'] ?? 'unknown',
                'path'   => $entry['local_path'] ?? null,
            ];
        }
    }

    if (!empty($manifest['failures'])) {
        foreach ($manifest['failures'] as $entry) {
            $type = $entry['type'] ?? null;
            if ($type && isset($summary['totals'][$type])) {
                $summary['totals'][$type]['failures']++;
            }
            $summary['failures'][] = [
                'type'    => $type,
                'path'    => $entry['local_path'] ?? null,
                'message' => $entry['message'] ?? 'Unknown failure',
            ];
        }
    }

    return $summary;
}

function hunnt_ai_get_technical_seo_error_log(): ?array {
    $upload_dir = wp_upload_dir();
    $log_file   = trailingslashit($upload_dir['basedir']) . 'hunnt-ai-optimized/error-log.json';

    if (!file_exists($log_file)) {
        return null;
    }

    $contents = file_get_contents($log_file);

    if ($contents === false) {
        return null;
    }

    $decoded = json_decode($contents, true);

    return is_array($decoded) ? $decoded : null;
}

function hunnt_ai_get_minify_log_snapshot(): ?array {
    $paths = hunnt_ai_get_uploads_optimized_paths();
    if (empty($paths['dir'])) {
        return null;
    }
    $log_file = trailingslashit($paths['dir']) . 'logs/minify-log.json';

    if (!file_exists($log_file)) {
        return null;
    }

    $contents = file_get_contents($log_file);

    if ($contents === false) {
        return null;
    }

    $decoded = json_decode($contents, true);

    return is_array($decoded) ? $decoded : null;
}

function hunnt_ai_get_technical_seo_warmup_state(): array {
    $next_timestamp = function_exists('wp_next_scheduled')
        ? wp_next_scheduled('hunnt_ai_technical_seo_warmup')
        : false;

    $last_timestamp = (int) get_option('hunnt_ai_technical_seo_last_warmup', 0);
    $last_result    = get_option('hunnt_ai_technical_seo_last_warmup_result', '');

    $now = current_time('timestamp');

    return [
        'next_timestamp'   => $next_timestamp ? (int) $next_timestamp : null,
        'next_relative'    => $next_timestamp ? human_time_diff($now, (int) $next_timestamp) : null,
        'last_timestamp'   => $last_timestamp ?: null,
        'last_relative'    => $last_timestamp ? human_time_diff($last_timestamp, $now) : null,
        'last_result'      => $last_result ?: null,
        'schedule_enabled' => $next_timestamp !== false,
    ];
}

function hunnt_ai_get_technical_seo_diagnostics(): array {
    $enabled       = hunnt_ai_is_technical_seo_enabled();
    $status        = hunnt_ai_get_technical_seo_status();
    $minimum_ratio = hunnt_ai_get_assets_min_ratio();
    $summary       = hunnt_ai_get_minify_summary();
    $last_summary  = get_option('hunnt_ai_assets_last_summary', []);
    $last_debug    = get_option('hunnt_ai_assets_last_debug', []);
    $manifest      = hunnt_ai_get_minify_manifest();
    $warmup_state  = hunnt_ai_get_technical_seo_warmup_state();

    $manifest_meta = [];
    if (!empty($manifest['optimized']['meta'])) {
        $manifest_meta = $manifest['optimized']['meta'];
    }

    $error_log = hunnt_ai_get_technical_seo_error_log();
    $minify_log = hunnt_ai_get_minify_log_snapshot();

    return [
        'enabled'        => $enabled,
        'status'         => $status,
        'minimum_ratio'  => $minimum_ratio,
        'summary'        => $summary,
        'last_summary'   => is_array($last_summary) ? $last_summary : [],
        'last_debug'     => is_array($last_debug) ? $last_debug : [],
        'manifest_meta'  => $manifest_meta,
        'warmup'         => $warmup_state,
        'logs'           => [
            'error'  => $error_log,
            'minify' => $minify_log,
        ],
    ];
}

function hunnt_ai_store_minify_log(array $log): ?string {
    $paths = hunnt_ai_get_uploads_optimized_paths();
    $baseDir = $paths['dir'] ?? '';

    if (empty($baseDir)) {
        return null;
    }

    $logDir = trailingslashit($baseDir) . 'logs/';

    if (!is_dir($logDir) && !wp_mkdir_p($logDir)) {
        return null;
    }

    $filename = $logDir . 'minify-log.json';

    $record = [
        'stored_at_gmt' => current_time('mysql', true),
        'data'          => $log,
    ];

    $payload = wp_json_encode($record, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);

    if ($payload === false) {
        return null;
    }

    $result = file_put_contents($filename, $payload);

    return $result === false ? null : $filename;
}

function hunnt_ai_apply_gzip() {
    // Include and run gzip init once
    if (!get_option('hunnt_ai_enable_gzip')) {
        update_option('hunnt_ai_enable_gzip', 1);
    }
    $gzip_file = HUNNT_AI_PATH . 'includes/init/gzip-init.php';
    if (file_exists($gzip_file)) {
        include_once $gzip_file;
        return true;
    }
    return false;
}
/**
 * Check if technical SEO is globally enabled
 * 
 * @return bool
 */

function hunnt_ai_schedule_technical_seo_warmup() {
    if (!function_exists('wp_next_scheduled') || !function_exists('wp_schedule_event')) {
        return;
    }

    if (!wp_next_scheduled('hunnt_ai_technical_seo_warmup')) {
        $timestamp = time() + HOUR_IN_SECONDS;
        wp_schedule_event($timestamp, 'twicedaily', 'hunnt_ai_technical_seo_warmup');
    }
}

function hunnt_ai_clear_technical_seo_warmup() {
    if (!function_exists('wp_next_scheduled') || !function_exists('wp_unschedule_event')) {
        return;
    }

    $timestamp = wp_next_scheduled('hunnt_ai_technical_seo_warmup');
    while ($timestamp) {
        wp_unschedule_event($timestamp, 'hunnt_ai_technical_seo_warmup');
        $timestamp = wp_next_scheduled('hunnt_ai_technical_seo_warmup');
    }
}

function hunnt_ai_handle_technical_seo_warmup() {
    if (!hunnt_ai_is_technical_seo_enabled()) {
        return;
    }

    if (!get_option('hunnt_ai_assets_minify_enable')) {
        return;
    }

    $result = true;

    if (function_exists('hunnt_ai_runtime_minify_warm_cache')) {
        $result = (bool) hunnt_ai_runtime_minify_warm_cache();
    }

    update_option('hunnt_ai_technical_seo_last_warmup', current_time('timestamp'));
    update_option('hunnt_ai_technical_seo_last_warmup_result', $result ? 'success' : 'failed');
}

add_action('hunnt_ai_technical_seo_warmup', 'hunnt_ai_handle_technical_seo_warmup');
function hunnt_ai_is_technical_seo_enabled() {
    return (bool) get_option('hunnt_ai_technical_seo_enabled', true);
}

/**
 * Enable technical SEO globally
 * 
 * @return bool Success status
 */
function hunnt_ai_enable_technical_seo() {
    $was_enabled = hunnt_ai_is_technical_seo_enabled();
    $updated = update_option('hunnt_ai_technical_seo_enabled', 1);

    if ($updated) {
        hunnt_ai_schedule_technical_seo_warmup();

        if (function_exists('wp_schedule_single_event')) {
            wp_schedule_single_event(time() + 5, 'hunnt_ai_technical_seo_warmup');
        }
    }

    if (!$was_enabled || $updated) {
        if (function_exists('hunnt_ai_enable_static_cache')) {
            hunnt_ai_enable_static_cache();
        }
    }

    return hunnt_ai_is_technical_seo_enabled();
}

/**
 * Disable technical SEO globally
 * Removes all technical SEO optimizations and deletes minified files
 * 
 * @return bool Success status
 */
function hunnt_ai_disable_technical_seo() {
    hunnt_ai_clear_technical_seo_warmup();

    // Disable cache
    delete_option('hunnt_ai_basic_cache_enabled');
    
    // Disable HTML minification
    delete_option('hunnt_ai_html_minify_enabled');
    
    // Disable Gzip
    delete_option('hunnt_ai_enable_gzip');
    
    // Disable assets minification
    delete_option('hunnt_ai_assets_minify_enable');
    delete_option('hunnt_ai_assets_minified');
    
    // DELETE runtime minified assets directory to ensure clean state
    $paths = hunnt_ai_get_uploads_optimized_paths();
    if (!empty($paths['dir']) && file_exists($paths['dir'])) {
        $iterator = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($paths['dir'], \FilesystemIterator::SKIP_DOTS),
            \RecursiveIteratorIterator::CHILD_FIRST
        );

        foreach ($iterator as $fileInfo) {
            $path = $fileInfo->getPathname();
            if ($fileInfo->isDir()) {
                @rmdir($path);
            } else {
                @unlink($path);
            }
        }

        @rmdir($paths['dir']);
        error_log('HUNNT AI: Cleared runtime minified asset directory: ' . $paths['dir']);
    }
    
    if (function_exists('hunnt_ai_minify_reset_manifest')) {
        hunnt_ai_minify_reset_manifest();
    }

    // Clear any WordPress object cache
    if (function_exists('wp_cache_flush')) {
        wp_cache_flush();
    }
    
    if (function_exists('hunnt_ai_disable_static_cache')) {
        hunnt_ai_disable_static_cache();
    }

    // Set global flag to disabled
    $result = update_option('hunnt_ai_technical_seo_enabled', 0);
    
    // Notify backend about status change
    if (function_exists('hunnt_ai_notify_technical_seo_status')) {
        hunnt_ai_notify_technical_seo_status([
            'cache'       => false,
            'minify_css'  => false,
            'minify_js'   => false,
            'minify_html' => false,
            'gzip'        => false,
        ]);
    }
    
    error_log('HUNNT AI: Technical SEO disabled and all optimizations removed');
    
    return $result;
}

/**
 * Bootstrap technical SEO features if enabled
 * 
 * @return void
 */
function hunnt_ai_bootstrap_technical_seo() {
    // Exit early if technical SEO is globally disabled
    if (!hunnt_ai_is_technical_seo_enabled()) {
        return;
    }
    
    if (get_option('hunnt_ai_basic_cache_enabled')) {
        add_action('send_headers', 'hunnt_ai_send_basic_cache_headers');
    }
    
    if (get_option('hunnt_ai_html_minify_enabled')) {
        add_action('template_redirect', function() {
            if (is_admin()) return;
            if (!headers_sent()) {
                ob_start('hunnt_ai_minify_html_buffer_callback');
            }
        });
    }
    
    if (get_option('hunnt_ai_enable_gzip')) {
        $gzip_file = HUNNT_AI_PATH . 'includes/init/gzip-init.php';
        if (file_exists($gzip_file)) {
            include_once $gzip_file;
        }
    }

    if (get_option('hunnt_ai_assets_minify_enable')) {
        hunnt_ai_schedule_technical_seo_warmup();
        // Hook ONLY at wp_enqueue_scripts for ALL bundle registration (header + footer)
        // Priority 999999 ensures we run after all themes/plugins have enqueued their assets
        // WordPress automatically handles printing header bundles in <head> and footer bundles before </body>
        add_action('wp_enqueue_scripts', 'hunnt_ai_replace_with_optimized_assets', 999999);
    }
    
}
add_action('plugins_loaded', 'hunnt_ai_bootstrap_technical_seo');

function hunnt_ai_check_subscription_active() {
    if (!function_exists('hunnt_ai_get_subscription_status')) {
        return [ 'active' => false, 'message' => __('API client missing', 'hunnt-ai') ];
    }
    $sub = hunnt_ai_get_subscription_status();
    // Prefer feature gate if available
    $hasTech = false;
    if (function_exists('hunnt_ai_get_technical_seo_access')) {
        $tech = hunnt_ai_get_technical_seo_access();
        $hasTech = !empty($tech['has']);
    } elseif (!empty($sub['features']) && isset($sub['features']['technical_seo'])) {
        $hasTech = (bool)$sub['features']['technical_seo'];
    }
    $active = (bool)$sub['active'] || $hasTech;
    $message = $active ? 'active' : ($sub['status'] ?: 'inactive');
    return [ 'active' => $active, 'message' => $message, 'raw' => $sub['raw'] ?? null ];
}

function hunnt_ai_notify_technical_seo_status($status) {
    if (!function_exists('hunnt_ai_api_request')) return;
    hunnt_ai_api_request('POST', 'subscription/technical-seo', [
        'status' => $status,
        'site'   => site_url(),
    ]);
}

/**
 * Ensure News Kit Elementor widgets always have a global data container before inline scripts run.
 */
function hunnt_ai_output_nekit_widget_data_guard() {
    static $printed = false;

    if ($printed || is_admin()) {
        return;
    }

    $printed = true;

    echo "<script>(window.nekitWidgetData = window.nekitWidgetData || []);</script>\n";
}

function hunnt_ai_replace_with_optimized_assets() {
    if (is_admin()) {
        return;
    }

    global $hunnt_ai_minify_building_depth;

    if (!empty($hunnt_ai_minify_building_depth)) {
        return;
    }

    $optimizer = new Hunnt_AI_Asset_Optimizer();
    $manifest  = hunnt_ai_normalize_minify_manifest($optimizer->load_manifest());

    $assets_snapshot = $optimizer->collect_assets();
    $runtime_update = hunnt_ai_minify_runtime_capture($optimizer, $manifest, $assets_snapshot);

    if (!empty($runtime_update['manifest'])) {
        $manifest = hunnt_ai_normalize_minify_manifest($runtime_update['manifest']);
    }

    $optimized_css = $manifest['optimized']['css'] ?? [];
    $optimized_js  = $manifest['optimized']['js'] ?? [];
    $bundled_css   = $manifest['optimized']['bundles']['css'] ?? [];
    $bundled_js    = $manifest['optimized']['bundles']['js'] ?? [];

    if (empty($optimized_css) && empty($optimized_js) && empty($bundled_css) && empty($bundled_js)) {
        return;
    }
    $aggregation_enabled = hunnt_ai_is_asset_aggregation_enabled();

    global $wp_styles, $wp_scripts;

    if (!($wp_styles instanceof \WP_Styles)) {
        $wp_styles = wp_styles();
    }

    if (!($wp_scripts instanceof \WP_Scripts)) {
        $wp_scripts = wp_scripts();
    }

    $aggregated_css_handles = [];
    $aggregated_js_handles  = [];

    if ($aggregation_enabled) {
        // CSS bundles (all go in <head>)
        if (!empty($bundled_css)) {
            foreach ($bundled_css as $bundle_handle => $bundle) {
                // Skip if already enqueued (WordPress's own duplicate prevention)
                if (wp_style_is($bundle_handle, 'enqueued')) {
                    continue;
                }

                $url  = $bundle['optimized_url'] ?? '';
                $path = $bundle['optimized_path'] ?? '';

                if (empty($url) || (is_string($path) && $path !== '' && !file_exists($path))) {
                    continue;
                }

                $deps  = is_array($bundle['deps']) ? array_values(array_unique(array_filter($bundle['deps'], 'strlen'))) : [];
                $media = !empty($bundle['media']) ? $bundle['media'] : 'all';
                $ver   = $bundle['hash'] ?? (defined('HUNNT_AI_VERSION') ? HUNNT_AI_VERSION : null);

                wp_register_style($bundle_handle, $url, $deps, $ver, $media);
                wp_enqueue_style($bundle_handle);

                $bundle_handles = isset($bundle['original_handles']) && is_array($bundle['original_handles'])
                    ? $bundle['original_handles']
                    : [];

                foreach ($bundle_handles as $original_handle) {
                    // Simply dequeue and deregister - DO NOT re-enqueue
                    // The bundle already contains this file's content
                    if (wp_style_is($original_handle, 'registered') || wp_style_is($original_handle, 'enqueued')) {
                        wp_dequeue_style($original_handle);
                        wp_deregister_style($original_handle);
                    }

                    $aggregated_css_handles[$original_handle] = true;
                }
            }
        }

        // JS bundles (WordPress handles header vs footer based on $in_footer parameter)
        if (!empty($bundled_js)) {
            foreach ($bundled_js as $bundle_handle => $bundle) {
                $url  = $bundle['optimized_url'] ?? '';
                $path = $bundle['optimized_path'] ?? '';

                if (empty($url) || (is_string($path) && $path !== '' && !file_exists($path))) {
                    continue;
                }

                $deps      = is_array($bundle['deps']) ? array_values(array_unique(array_filter($bundle['deps'], 'strlen'))) : [];
                $in_footer = !empty($bundle['in_footer']);
                $ver       = $bundle['hash'] ?? (defined('HUNNT_AI_VERSION') ? HUNNT_AI_VERSION : null);

                $missing_deps = [];

                foreach ($deps as $dep) {
                    if (!wp_script_is($dep, 'registered') && !wp_script_is($dep, 'enqueued')) {
                        $missing_deps[] = $dep;
                    }
                }

                if (!empty($missing_deps)) {
                    $optimizer->log_action('bundle_missing_dependencies', [
                        'bundle'       => $bundle_handle,
                        'missing_deps' => $missing_deps,
                        'deps'         => $deps,
                        'in_footer'    => $in_footer,
                    ]);

                    do_action(
                        'hunnt_ai_js_bundle_missing_dependencies',
                        $bundle_handle,
                        $missing_deps,
                        $bundle
                    );

                    continue;
                }

                wp_register_script($bundle_handle, $url, $deps, $ver, $in_footer);
                wp_enqueue_script($bundle_handle);

                if (!empty($bundle['translations']) && function_exists('wp_set_script_translations')) {
                    foreach ((array) $bundle['translations'] as $translation) {
                        $textdomain = isset($translation['textdomain']) ? (string) $translation['textdomain'] : '';
                        $translation_path = isset($translation['path']) ? (string) $translation['path'] : '';

                        if ($textdomain === '') {
                            continue;
                        }

                        if ($translation_path !== '') {
                            wp_set_script_translations($bundle_handle, $textdomain, $translation_path);
                        } else {
                            wp_set_script_translations($bundle_handle, $textdomain);
                        }
                    }
                }

                $bundle_handles = isset($bundle['original_handles']) && is_array($bundle['original_handles'])
                    ? $bundle['original_handles']
                    : [];

                foreach ($bundle_handles as $original_handle) {
                    // Simply dequeue and deregister - DO NOT re-enqueue
                    // The bundle already contains this file's content
                    if (wp_script_is($original_handle, 'registered') || wp_script_is($original_handle, 'enqueued')) {
                        wp_dequeue_script($original_handle);
                        wp_deregister_script($original_handle);
                    }

                    $aggregated_js_handles[$original_handle] = true;
                }
            }
        }
    }

    foreach (array_keys($aggregated_css_handles) as $handle) {
        unset($optimized_css[$handle]);
    }

    foreach (array_keys($aggregated_js_handles) as $handle) {
        unset($optimized_js[$handle]);
    }

    foreach ($optimized_css as $handle => $data) {
        $url  = $data['optimized_url'] ?? '';
        $path = $data['optimized_path'] ?? '';
        if (empty($url) || (is_string($path) && $path !== '' && !file_exists($path))) {
            continue;
        }

        $deps   = is_array($data['deps']) ? $data['deps'] : [];
        $media  = !empty($data['media']) ? $data['media'] : 'all';
        $ver    = $data['hash'] ?? (defined('HUNNT_AI_VERSION') ? HUNNT_AI_VERSION : null);

        if (isset($wp_styles->registered[$handle])) {
            $style = $wp_styles->registered[$handle];
            $style->src  = $url;
            $style->deps = $deps;
            $style->ver  = $ver;
            $style->args = $media;
        } else {
            wp_register_style($handle, $url, $deps, $ver, $media);
        }

        if (!in_array($handle, (array) $wp_styles->queue, true)) {
            wp_enqueue_style($handle);
        }

        if (!empty($data['inline_after']) && function_exists('wp_add_inline_style')) {
            foreach ((array) $data['inline_after'] as $inline_css) {
                if (!is_string($inline_css)) {
                    continue;
                }

                $inline_css = trim($inline_css);

                if ($inline_css === '') {
                    continue;
                }

                wp_add_inline_style($handle, $inline_css);
            }
        }

        wp_enqueue_style($handle);
    }

    foreach ($optimized_js as $handle => $data) {
        $url  = $data['optimized_url'] ?? '';
        $path = $data['optimized_path'] ?? '';
        if (empty($url) || (is_string($path) && $path !== '' && !file_exists($path))) {
            continue;
        }

        $deps      = is_array($data['deps']) ? $data['deps'] : [];
        $in_footer = !empty($data['in_footer']);
        $ver       = $data['hash'] ?? (defined('HUNNT_AI_VERSION') ? HUNNT_AI_VERSION : null);

        if (isset($wp_scripts->registered[$handle])) {
            $script = $wp_scripts->registered[$handle];
            $script->src  = $url;
            $script->deps = $deps;
            $script->ver  = $ver;

            if ($in_footer) {
                $script->extra['group'] = 1;
            } else {
                unset($script->extra['group']);
            }
        } else {
            wp_register_script($handle, $url, $deps, $ver, $in_footer);
        }

        if (!in_array($handle, (array) $wp_scripts->queue, true)) {
            wp_enqueue_script($handle);
        }

        if (!empty($data['textdomain']) && function_exists('wp_set_script_translations')) {
            $textdomain = $data['textdomain'];
            $translation_path = isset($data['translations_path']) ? $data['translations_path'] : null;

            if (is_string($textdomain) && $textdomain !== '') {
                if (is_string($translation_path) && $translation_path !== '') {
                    wp_set_script_translations($handle, $textdomain, $translation_path);
                } else {
                    wp_set_script_translations($handle, $textdomain);
                }
            }
        }

        if (!empty($data['inline_data'])) {
            if (function_exists('wp_script_add_data')) {
                wp_script_add_data($handle, 'data', $data['inline_data']);
            } else {
                wp_add_inline_script($handle, $data['inline_data'], 'before');
            }
        }

        if (!empty($data['inline_before'])) {
            foreach ((array) $data['inline_before'] as $snippet) {
                if (!is_string($snippet)) {
                    continue;
                }

                $snippet = trim($snippet);

                if ($snippet === '') {
                    continue;
                }

                wp_add_inline_script($handle, $snippet, 'before');
            }
        }

        if (!empty($data['inline_after'])) {
            foreach ((array) $data['inline_after'] as $snippet) {
                if (!is_string($snippet)) {
                    continue;
                }

                $snippet = trim($snippet);

                if ($snippet === '') {
                    continue;
                }

                wp_add_inline_script($handle, $snippet, 'after');
            }
        }
    }

    if (!empty($optimized_js)) {
        foreach ($optimized_js as $handle => $data) {
            if (!empty($data['in_footer'])) {
                wp_script_add_data($handle, 'group', 1);
            }
        }
    }
}

