<?php
// Add sitemap discovery in robots.txt
add_filter('robots_txt', function($output, $public) {
    if ($public) {
        $output .= "\n# Hunnt AI Sitemaps\n";
        $output .= "Sitemap: " . home_url('/hunnt-ai/sitemap.xml') . "\n";
    }
    return $output;
}, 10, 2);

// Add sitemap URL to WordPress head
add_action('wp_head', function() {
    echo '<link rel="sitemap" type="application/xml" title="Hunnt AI Sitemap" href="' . home_url('/hunnt-ai/sitemap.xml') . '" />' . "\n";
});

// Add rewrite rules
add_action('init', function() {
    add_rewrite_rule('^hunnt-ai/sitemap\.xml$', 'index.php?hunnt_ai_sitemap=index', 'top');
    add_rewrite_rule('^hunnt-ai/sitemap-([a-zA-Z0-9_-]+)\.xml$', 'index.php?hunnt_ai_sitemap=$matches[1]', 'top');
});

// Register query var
add_filter('query_vars', function($vars) {
    $vars[] = 'hunnt_ai_sitemap';
    return $vars;
});

// Handle the sitemap request - EARLIER HOOK
add_action('template_redirect', function() {
    $sitemap_type = get_query_var('hunnt_ai_sitemap');

    if (!empty($sitemap_type)) {
        hunnt_ai_output_sitemap($sitemap_type);
    }
});

// Fallback detection for cases where rewrite rules don't work
add_action('parse_request', function($wp) {
    $request_uri = $_SERVER['REQUEST_URI'];

    if (strpos($request_uri, '/hunnt-ai/sitemap') !== false) {
        if (preg_match('#/hunnt-ai/sitemap(-([a-zA-Z0-9_-]+))?\.xml$#', $request_uri, $matches)) {
            $sitemap_type = isset($matches[2]) ? $matches[2] : 'index';
            hunnt_ai_output_sitemap($sitemap_type);
        }
    }
});

// Centralized sitemap output function with Google fixes
function hunnt_ai_output_sitemap($sitemap_type) {
    // Prevent multiple output
    if (defined('HUNNT_AI_SITEMAP_OUTPUT')) {
        return;
    }
    define('HUNNT_AI_SITEMAP_OUTPUT', true);

    // COMPLETELY clear all output
    while (ob_get_level() > 0) {
        ob_end_clean();
    }

    // Remove any filters that might modify output
    remove_all_actions('wp_head');
    remove_all_actions('wp_footer');
    remove_all_filters('wp_headers');

    $post_type = ($sitemap_type === 'index') ? null : $sitemap_type;
    $xml = hunnt_ai_generate_sitemap($post_type);

    // Set proper HTTP status FIRST
    if (function_exists('http_response_code')) {
        http_response_code(200);
    }

    // Clear any previous headers
    header_remove();

    // Set CRITICAL headers for Google
    header('HTTP/1.1 200 OK');
    header('Content-Type: application/xml; charset=utf-8');
    header('X-Robots-Tag: index, follow', true);
    header('Cache-Control: public, max-age=3600, s-maxage=3600');

    // Prevent any WordPress processing
    add_filter('wp_die_handler', function() use ($xml) {
        return function() use ($xml) {
            echo $xml;
            exit;
        };
    });

    echo $xml;
    exit;
}

// ------------------ Generate sitemap ------------------
function hunnt_ai_generate_sitemap($post_type = null) {
    if (!$post_type) {
        return hunnt_ai_generate_index_sitemap();
    }

    // Validate post type
    if (!post_type_exists($post_type)) {
        return '<?xml version="1.0" encoding="UTF-8"?><error>Post type not found</error>';
    }

    $post_type_object = get_post_type_object($post_type);
    if (!$post_type_object || !$post_type_object->public) {
        return '<?xml version="1.0" encoding="UTF-8"?><error>Post type not accessible</error>';
    }

    // Use get_posts with minimal filters
    $posts = get_posts([
        'post_type'      => $post_type,
        'post_status'    => 'publish',
        'posts_per_page' => -1,
        'orderby'        => 'modified',
        'order'          => 'DESC',
        'no_found_rows'  => true,
        'update_post_meta_cache' => false,
        'update_post_term_cache' => false,
        'suppress_filters' => true, // Important: suppress filters,
        'lang'=>''
    ]);

    $xml  = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
    $xml .= '<!-- Generated by Hunnt AI Sitemap -->' . "\n";
    $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";

    foreach ($posts as $post) {
        $loc = get_permalink($post->ID);

        if ($loc && !is_wp_error($loc)) {
            $xml .= "  <url>\n";
            $xml .= "    <loc>" . esc_url($loc) . "</loc>\n";
            $xml .= "    <lastmod>" . mysql2date('Y-m-d\TH:i:s+00:00', $post->post_modified) . "</lastmod>\n";
            $xml .= "    <changefreq>monthly</changefreq>\n";
            $xml .= "    <priority>0.8</priority>\n";
            $xml .= "  </url>\n";
        }
    }

    $xml .= '</urlset>';
    return $xml;
}

// ------------------ Sitemap index ------------------
function hunnt_ai_generate_index_sitemap() {
    $home = home_url();
    $today = date('Y-m-d\TH:i:s+00:00');

    $post_types = get_post_types(['public' => true], 'objects');
    $excluded_types = ['attachment', 'revision', 'nav_menu_item'];

    $xml  = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
    $xml .= '<!-- Hunnt AI Sitemap Index -->' . "\n";
    $xml .= '<sitemapindex xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";

    foreach ($post_types as $type => $obj) {
        if (in_array($type, $excluded_types)) continue;

        $post_count = wp_count_posts($type);
        if ($post_count->publish > 0) {
            $loc = esc_url("{$home}/hunnt-ai/sitemap-{$type}.xml");
            $xml .= "  <sitemap>\n";
            $xml .= "    <loc>{$loc}</loc>\n";
            $xml .= "    <lastmod>{$today}</lastmod>\n";
            $xml .= "  </sitemap>\n";
        }
    }

    $xml .= '</sitemapindex>';
    return $xml;
}

// Add to robots.txt
add_filter('robots_txt', function($output, $public) {
    if ($public) {
        $output .= "\n# Hunnt AI Sitemaps\n";
        $output .= "Sitemap: " . home_url('/hunnt-ai/sitemap.xml') . "\n";
    }
    return $output;
}, 10, 2);

// Flush rewrite rules
register_activation_hook(__FILE__, 'hunnt_ai_activate_plugin');
register_deactivation_hook(__FILE__, 'hunnt_ai_deactivate_plugin');

function hunnt_ai_activate_plugin() {
    flush_rewrite_rules();
}

function hunnt_ai_deactivate_plugin() {
    flush_rewrite_rules();
}

// Debug function - add ?debug_hunnt_sitemap to any URL to test
add_action('init', function() {
    if (isset($_GET['debug_hunnt_sitemap'])) {
        hunnt_ai_output_sitemap('post');
    }
});