<?php
/**
 * Generate minified assets by capturing WordPress enqueued scripts and styles
 * This runs ONCE when minification is enabled, not on every page load
 * Works with ANY WordPress theme - captures ALL enqueued assets
 */
function hunnt_ai_minify_match_handle($handle, array $patterns) {
    foreach ($patterns as $pattern) {
        if ($handle === $pattern || strpos($handle, $pattern) === 0) {
            return $pattern;
        }
    }

    return false;
}

function hunnt_ai_minify_context_hash() {
    $theme = wp_get_theme();
    $theme_key = $theme ? ($theme->get_stylesheet() . '|' . $theme->get('Version')) : 'unknown-theme';
    $locale = get_locale();
    $site_url = site_url();
    $plugin_version = defined('HUNNT_AI_VERSION') ? HUNNT_AI_VERSION : 'dev';
    $salt = apply_filters('hunnt_ai_minify_context_salt', '');

    return substr(md5($theme_key . '|' . $locale . '|' . $site_url . '|' . $plugin_version . '|' . $salt), 0, 12);
}

function hunnt_ai_generate_minified_assets() {
    $upload_dir = wp_upload_dir();
    $optimize_dir = trailingslashit($upload_dir['basedir']) . 'myplugin-optimized/';

    if (!file_exists($optimize_dir)) {
        if (!wp_mkdir_p($optimize_dir)) {
            error_log('HUNNT AI: Failed to create optimization directory: ' . $optimize_dir);
            return false;
        }
    }

    $context_hash = hunnt_ai_minify_context_hash();

    $manifest = [
        'generated_at' => current_time('mysql'),
        'context'      => [
            'hash'   => $context_hash,
            'theme'  => get_stylesheet(),
            'locale' => get_locale(),
        ],
        'css'          => [
            'file'     => "frontend-{$context_hash}.min.css",
            'included' => [],
            'excluded' => [],
        ],
        'js'           => [
            'header'                => [
                'file'         => "frontend-head-{$context_hash}.min.js",
                'included'     => [],
                'dependencies' => [],
            ],
            'footer'                => [
                'file'         => "frontend-footer-{$context_hash}.min.js",
                'included'     => [],
                'dependencies' => [],
            ],
            'excluded'              => [],
            'skipped'               => [],
            'external_dependencies' => [],
        ],
    ];

    try {
        set_transient('hunnt_ai_capture_assets', true, 300);

        ob_start();

        global $wp_styles, $wp_scripts;

        if (!did_action('wp_enqueue_scripts')) {
            do_action('wp_enqueue_scripts');
        }
        if (!did_action('wp_print_styles')) {
            do_action('wp_print_styles');
        }
        if (!did_action('wp_print_scripts')) {
            do_action('wp_print_scripts');
        }

        $resolve_dependencies = function ($dependency_object) {
            if (empty($dependency_object) || empty($dependency_object->queue)) {
                return [];
            }

            $ordered = [];
            $visited = [];
            $visiting = [];

            $registered = isset($dependency_object->registered) && is_array($dependency_object->registered)
                ? $dependency_object->registered
                : [];

            $visit = function ($handle) use (&$visit, &$ordered, &$visited, &$visiting, $registered) {
                if (isset($visited[$handle])) {
                    return;
                }

                if (isset($visiting[$handle])) {
                    error_log('HUNNT AI: Detected circular dependency while ordering assets: ' . $handle);
                    return;
                }

                if (!isset($registered[$handle])) {
                    return;
                }

                $visiting[$handle] = true;

                $dependencies = (array) ($registered[$handle]->deps ?? []);
                foreach ($dependencies as $dependency_handle) {
                    $visit($dependency_handle);
                }

                $visiting[$handle] = false;
                $visited[$handle] = true;
                $ordered[] = $handle;
            };

            foreach ($dependency_object->queue as $queued_handle) {
                $visit($queued_handle);
            }

            return $ordered;
        };

        $collect_inline_payload = function ($script, $key) {
            if (empty($script->extra[$key])) {
                return '';
            }

            $payload = $script->extra[$key];
            if (is_array($payload)) {
                $payload = implode("\n", array_filter(array_map('trim', $payload)));
            }

            return trim((string) $payload);
        };

        $combined_css = [];
        $processed_css = 0;
        $combined_js_groups = [
            'header' => [],
            'footer' => [],
        ];
        $processed_js = [
            'header' => 0,
            'footer' => 0,
        ];
        $js_group_entries = [
            'header' => [],
            'footer' => [],
        ];
        $js_external_dependencies = [];

        // CSS processing with dependency-aware ordering
        if (!empty($wp_styles) && !empty($wp_styles->queue)) {
            $style_handles = $resolve_dependencies($wp_styles);

            $default_excluded_css = [
                'admin-bar',
                'dashicons',
                'wp-block-library',
                'wp-block-library-theme',
                'elementor-icons',
                'elementor-common',
                'elementor-frontend',
                'elementor-pro',
                'elementor-post-',
                'customize-',
                'editor-',
                'fontawesome',
                'font-awesome',
                'woocommerce',
            ];

            $css_exclusion_patterns = apply_filters('hunnt_ai_minify_css_exclusions', $default_excluded_css, $style_handles);

            foreach ($style_handles as $handle) {
                if (!isset($wp_styles->registered[$handle])) {
                    continue;
                }

                $style = $wp_styles->registered[$handle];
                $src = $style->src;

                if ($matched = hunnt_ai_minify_match_handle($handle, $css_exclusion_patterns)) {
                    $manifest['css']['excluded'][$handle] = 'protected:' . $matched;
                    error_log("HUNNT AI: Excluded CSS from combination: {$handle} (rule: {$matched})");
                    continue;
                }

                if (apply_filters('hunnt_ai_minify_skip_css_handle', false, $handle, $style)) {
                    $manifest['css']['excluded'][$handle] = 'filtered';
                    continue;
                }

                if (empty($src)) {
                    $manifest['css']['excluded'][$handle] = 'empty_source';
                    continue;
                }

                if (strpos($src, '//') !== false && strpos($src, site_url()) === false) {
                    $manifest['css']['excluded'][$handle] = 'external_url';
                    error_log("HUNNT AI: Skipping external CSS: {$handle} ({$src})");
                    continue;
                }

                if (strpos($src, '//') === false) {
                    $src = site_url($src);
                }

                $http_args = apply_filters('hunnt_ai_minify_css_request_args', ['timeout' => 15, 'sslverify' => false], $handle, $style);
                $response = wp_remote_get($src, $http_args);
                if (is_wp_error($response)) {
                    $manifest['css']['excluded'][$handle] = 'fetch_error';
                    error_log("HUNNT AI: Failed to fetch CSS {$handle}: " . $response->get_error_message());
                    continue;
                }

                $css_content = wp_remote_retrieve_body($response);
                if (empty($css_content) || strlen($css_content) < 20) {
                    $manifest['css']['excluded'][$handle] = 'empty_body';
                    continue;
                }

                $css_content = apply_filters('hunnt_ai_minify_css_content', $css_content, $handle, $style);

                $combined_css[] = "/* Source: {$handle} */\n" . rtrim($css_content) . "\n";
                $manifest['css']['included'][] = $handle;
                $processed_css++;

                error_log("HUNNT AI: Combined CSS: {$handle} (" . strlen($css_content) . ' bytes)');
            }
        }

        // JS processing with dependency-aware ordering and inline data support
        if (!empty($wp_scripts) && !empty($wp_scripts->queue)) {
            $script_handles = $resolve_dependencies($wp_scripts);

            $default_protected_js = [
                'jquery',
                'jquery-core',
                'jquery-migrate',
                'wp-',
                'elementor',
                'elementor-pro',
                'elementor-frontend',
                'elementor-common',
                'elementor-dialog',
                'elementor-waypoints',
                'elementor-webpack-runtime',
                'react',
                'react-dom',
                'moment',
                'lodash',
                'underscore',
                'backbone',
                'admin-',
                'customize-',
                'editor-',
            ];

            $protected_patterns = apply_filters('hunnt_ai_minify_js_protected_handles', $default_protected_js, $script_handles);
            $user_exclusions = (array) apply_filters('hunnt_ai_minify_js_exclusions', [], $script_handles);

            foreach ($script_handles as $handle) {
                if (!isset($wp_scripts->registered[$handle])) {
                    continue;
                }

                $script = $wp_scripts->registered[$handle];
                $src = $script->src;
                $group = (!empty($script->extra['group']) && (int) $script->extra['group'] === 1) ? 'footer' : 'header';

                if ($matched = hunnt_ai_minify_match_handle($handle, $user_exclusions)) {
                    $manifest['js']['excluded'][$handle] = 'user:' . $matched;
                    error_log("HUNNT AI: User exclusion for script {$handle} (rule: {$matched})");
                    continue;
                }

                if ($matched = hunnt_ai_minify_match_handle($handle, $protected_patterns)) {
                    $manifest['js']['excluded'][$handle] = 'protected:' . $matched;
                    error_log("HUNNT AI: Protected script skipped {$handle} (rule: {$matched})");
                    continue;
                }

                if (apply_filters('hunnt_ai_minify_skip_js_handle', false, $handle, $script)) {
                    $manifest['js']['excluded'][$handle] = 'filtered';
                    continue;
                }

                $before_chunks = [];
                $after_chunks = [];

                if (!empty($script->extra)) {
                    $data_chunk = $collect_inline_payload($script, 'data');
                    if ($data_chunk !== '') {
                        $before_chunks[] = $data_chunk;
                    }

                    $before_inline = $collect_inline_payload($script, 'before');
                    if ($before_inline !== '') {
                        $before_chunks[] = $before_inline;
                    }

                    $after_inline = $collect_inline_payload($script, 'after');
                    if ($after_inline !== '') {
                        $after_chunks[] = $after_inline;
                    }
                }

                $dependencies = (array) ($script->deps ?? []);
                if (!empty($dependencies)) {
                    foreach ($dependencies as $dependency_handle) {
                        if (!in_array($dependency_handle, $script_handles, true) && !hunnt_ai_minify_match_handle($dependency_handle, $protected_patterns)) {
                            $js_external_dependencies[] = $dependency_handle;
                        }
                    }
                }

                if (empty($src)) {
                    if (empty($before_chunks) && empty($after_chunks)) {
                        $manifest['js']['skipped'][$handle] = 'empty_source';
                        continue;
                    }

                    $segment_parts = ["/* Source: {$handle} (inline) */"];
                    if (!empty($before_chunks)) {
                        $segment_parts[] = implode("\n", $before_chunks);
                    }
                    if (!empty($after_chunks)) {
                        $segment_parts[] = implode("\n", $after_chunks);
                    }

                    $js_group_entries[$group][] = [
                        'handle'       => $handle,
                        'segment'      => implode("\n", $segment_parts) . "\n",
                        'dependencies' => $dependencies,
                    ];
                    $manifest['js'][$group]['included'][] = $handle;
                    $processed_js[$group]++;
                    continue;
                }

                if (strpos($src, '//') !== false && strpos($src, site_url()) === false) {
                    $manifest['js']['excluded'][$handle] = 'external_url';
                    error_log("HUNNT AI: Skipping external JS: {$handle} ({$src})");
                    continue;
                }

                if (strpos($src, '//') === false) {
                    $src = site_url($src);
                }

                $http_args = apply_filters('hunnt_ai_minify_js_request_args', ['timeout' => 15, 'sslverify' => false], $handle, $script);
                $response = wp_remote_get($src, $http_args);
                if (is_wp_error($response)) {
                    $manifest['js']['skipped'][$handle] = 'fetch_error';
                    error_log("HUNNT AI: Failed to fetch JS {$handle}: " . $response->get_error_message());
                    continue;
                }

                $js_content = wp_remote_retrieve_body($response);
                if (empty($js_content) || strlen($js_content) < 20) {
                    $manifest['js']['skipped'][$handle] = 'empty_body';
                    continue;
                }

                $js_content = apply_filters('hunnt_ai_minify_js_content', $js_content, $handle, $script);

                $segment_parts = ["/* Source: {$handle} */"];

                if (!empty($before_chunks)) {
                    $segment_parts[] = implode("\n", $before_chunks);
                }

                $segment_parts[] = "try {\n" . rtrim($js_content) . "\n} catch (error) {\n  console.error('HUNNT AI: Error in script {$handle}:', error);\n}";

                if (!empty($after_chunks)) {
                    $segment_parts[] = implode("\n", $after_chunks);
                }

                $js_group_entries[$group][] = [
                    'handle'       => $handle,
                    'segment'      => implode("\n", $segment_parts) . "\n",
                    'dependencies' => $dependencies,
                ];
                $manifest['js'][$group]['included'][] = $handle;
                $processed_js[$group]++;

                error_log("HUNNT AI: Combined JS: {$handle} (" . strlen($js_content) . ' bytes)');
            }

            foreach (['header', 'footer'] as $group) {
                if (empty($js_group_entries[$group])) {
                    $manifest['js'][$group]['included'] = array_values(array_unique($manifest['js'][$group]['included']));
                    $manifest['js'][$group]['dependencies'] = [];
                    $manifest['js'][$group]['file'] = null;
                    continue;
                }

                $group_segments = [];
                $group_dependencies = [];

                foreach ($js_group_entries[$group] as $entry) {
                    $group_segments[] = $entry['segment'];

                    foreach ($entry['dependencies'] as $dependency_handle) {
                        if (in_array($dependency_handle, $manifest['js'][$group]['included'], true)) {
                            continue;
                        }

                        if (hunnt_ai_minify_match_handle($dependency_handle, $protected_patterns)) {
                            continue;
                        }

                        if (!in_array($dependency_handle, $group_dependencies, true)) {
                            $group_dependencies[] = $dependency_handle;
                        }
                    }
                }

                $manifest['js'][$group]['included'] = array_values(array_unique($manifest['js'][$group]['included']));
                $manifest['js'][$group]['dependencies'] = array_values($group_dependencies);
                $combined_js_groups[$group] = $group_segments;
            }
        }

        ob_end_clean();
        delete_transient('hunnt_ai_capture_assets');

        $css_file = $optimize_dir . $manifest['css']['file'];
        $js_files = [
            'header' => $manifest['js']['header']['file'] ? $optimize_dir . $manifest['js']['header']['file'] : null,
            'footer' => $manifest['js']['footer']['file'] ? $optimize_dir . $manifest['js']['footer']['file'] : null,
        ];
        $manifest_file = $optimize_dir . 'frontend-manifest.json';

        if (!empty($combined_css)) {
            $css_payload = "/* HUNNT AI combined CSS | Generated: " . current_time('mysql') . " */\n" . implode("\n", $combined_css);
            $css_written = file_put_contents($css_file, $css_payload);
            if ($css_written === false) {
                error_log('HUNNT AI: Failed to write combined CSS file');
                return false;
            }
            error_log("HUNNT AI: Combined {$processed_css} CSS files ({$css_written} bytes)");
        } else {
            $placeholder_css = "/* HUNNT AI combined CSS - No styles captured on " . date('Y-m-d H:i:s') . " */\n";
            file_put_contents($css_file, $placeholder_css);
            error_log('HUNNT AI: No CSS files found to combine');
        }

        foreach (['header', 'footer'] as $group) {
            $segments = $combined_js_groups[$group];
            $target = $js_files[$group];

            if (empty($segments) || !$target) {
                $manifest['js'][$group]['file'] = null;
                continue;
            }

            $js_payload = "/* HUNNT AI combined JS ({$group}) | Generated: " . current_time('mysql') . " */\n" . implode("\n", $segments);
            $js_written = file_put_contents($target, $js_payload);
            if ($js_written === false) {
                error_log("HUNNT AI: Failed to write combined JS file for group {$group}");
                return false;
            }

            error_log("HUNNT AI: Combined {$processed_js[$group]} JS files for {$group} ({$js_written} bytes)");
        }

        if (!empty($js_external_dependencies)) {
            $manifest['js']['external_dependencies'] = array_values(array_unique(array_filter($js_external_dependencies)));
        }

        file_put_contents($manifest_file, wp_json_encode($manifest, JSON_PRETTY_PRINT));

        error_log('HUNNT AI: Asset combination manifest stored at ' . $manifest_file);
        $total_js_processed = array_sum($processed_js);
        error_log("HUNNT AI: Asset combination completed - {$processed_css} CSS, {$total_js_processed} JS files processed");

        return true;
    } catch (\Exception $exception) {
        error_log('HUNNT AI: Exception during asset generation: ' . $exception->getMessage());
        return false;
    }
}

// Minify functions
function myplugin_minify_css($css) {
    $css = preg_replace('!/\*.*?\*/!s', '', $css);
    $css = preg_replace('/\n\s*\n/', "\n", $css);
    $css = preg_replace('/[\r\n\t]+/', '', $css);
    $css = preg_replace('/\s+/', ' ', $css);
    return $css;
}

function myplugin_minify_js($js) {
    // Safety: return original if empty
    if (empty(trim($js))) {
        return $js;
    }
    
    try {
        // CRITICAL: JavaScript minification is extremely complex and risky
        // Proper minification requires parsing the AST (Abstract Syntax Tree)
        // Any regex-based approach WILL break valid JavaScript code
        
        // For safety and reliability, we do MINIMAL optimization:
        // 1. Remove excessive consecutive empty lines (3+ in a row)
        // 2. Trim trailing whitespace at end of file
        // 3. Ensure proper statement termination
        // 4. That's it - preserve everything else!
        
        // This approach is safe because:
        // - It doesn't touch comments (they might be inside strings/regex)
        // - It doesn't touch whitespace (JS depends on it for parsing)
        // - It doesn't modify any code structure
        
        // Remove excessive empty lines (3+ consecutive blank lines become 2)
        $js = preg_replace('/\n\s*\n\s*\n+/', "\n\n", $js);
        
        // Remove trailing whitespace at end of file only
        $js = rtrim($js);
        
        // IMPORTANT: Ensure statement termination
        // Add semicolon if code doesn't end with: ; } ) or >
        if (!preg_match('/[;\}\)>]\s*$/', $js)) {
            $js .= ';';
        }
        
        // That's it - safe and won't break anything!
        return $js;
        
    } catch (\Exception $e) {
        // If even this minimal processing fails, return original
        error_log('HUNNT AI: JavaScript minification failed: ' . $e->getMessage());
        return $js;
    }
}