<?php
/**
 * Google OAuth Integration for Agency Accounts
 * Handles Google authentication via Hunnt AI API
 * 
 * @package HunntAI
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Verify Google OAuth token status
 * 
 * Legacy function - Returns detailed array response
 * Uses hunnt_ai_verify_google_token_with_api() internally
 * 
 * @return array Detailed verification result
 */
function hunnt_ai_verify_google_token() {
    $api_key = get_option('hunnt_ai_api_key');
    
    if (empty($api_key)) {
        return [
            'success' => false,
            'verified' => false,
            'message' => esc_html__('License key not found', 'hunnt-ai')
        ];
    }
    
    // Use the reusable verification function with API call and caching
    $is_verified = hunnt_ai_verify_google_token_with_api();
    
    return [
        'success' => true,
        'verified' => $is_verified,
        'message' => $is_verified 
            ? esc_html__('Google token is valid', 'hunnt-ai')
            : esc_html__('Google token is invalid or expired', 'hunnt-ai')
    ];
}

/**
 * Revoke Google OAuth token
 */
function hunnt_ai_revoke_google_token($agency_id) {
    $api_key = get_option('hunnt_ai_api_key');
    
    if (empty($api_key)) {
        return [
            'success' => false,
            'message' => esc_html__('License key not found', 'hunnt-ai')
        ];
    }
    
    $api_url = hunnt_ai_get_api_base() . 'agency/google/token/revoke';
    
    $response = wp_remote_request($api_url, [
        'method' => 'DELETE',
        'headers' => [
            'Accept' => 'application/json',
            'Content-Type' => 'application/json',
            'x-api-key' => $api_key
        ],
        'body' => wp_json_encode(['agency_id' => $agency_id]),
        'timeout' => 15
    ]);
    if (is_wp_error($response)) {
        return [
            'success' => false,
            'message' => $response->get_error_message()
        ];
    }
    
    $status_code = wp_remote_retrieve_response_code($response);
    $body = wp_remote_retrieve_body($response);

    $data = json_decode($body, true);
    
    if ($status_code === 200) {
        // Clear cached verification status
        delete_transient('hunnt_ai_google_token_verified');
        delete_transient('hunnt_ai_google_token_verified_at');
        
        return [
            'success' => true,
            'message' => isset($data['message']) ? $data['message'] : esc_html__('Google token revoked successfully', 'hunnt-ai'),
            'data' => $data
        ];
    }
    
    return [
        'success' => false,
        'message' => isset($data['message']) ? $data['message'] : esc_html__('Failed to revoke Google token', 'hunnt-ai')
    ];
}

/**
 * Check if user is agency type
 */
function hunnt_ai_is_agency_account() {
    return get_option('hunnt_ai_user_type', '') === 'agency';
}

/**
 * Get user ID
 */
function hunnt_ai_get_user_id() {
    return get_option('hunnt_ai_user_id', null);
}

/**
 * Get agency ID (alias for user ID)
 */
function hunnt_ai_get_agency_id() {
    return hunnt_ai_get_user_id();
}

/**
 * Verify Google token via API (reusable function)
 * 
 * @return bool True if token is valid, false otherwise
 */
function hunnt_ai_verify_google_token_with_api() {
    $api_key = get_option('hunnt_ai_api_key');
    
    if (empty($api_key)) {
        return false;
    }
    
    $api_base = hunnt_ai_get_api_base();
    $verify_url = $api_base . 'agency/google/token/verify';
    
    $response = wp_remote_get($verify_url, [
        'timeout' => 15,
        'headers' => [
            'Accept' => 'application/json',
            'X-API-Key' => $api_key
        ]
    ]);
    
    if (is_wp_error($response)) {
        error_log('Hunnt AI: Google verify API error - ' . $response->get_error_message());
        return false;
    }
    
    $status_code = wp_remote_retrieve_response_code($response);
    
    if ($status_code !== 200) {
        return false;
    }
    
    $body = wp_remote_retrieve_body($response);
    $data = json_decode($body, true);

    $normalize_flag = static function($value) {
        if (is_bool($value)) {
            return $value;
        }

        if (is_numeric($value)) {
            return (int) $value === 1;
        }

        if (is_string($value)) {
            $normalized = strtolower(trim($value));
            return in_array($normalized, ['1', 'true', 'yes', 'y', 'connected', 'valid', 'success', 'authorized', 'active'], true);
        }

        return false;
    };

    $is_verified = false;

    if (isset($data['success']) && $normalize_flag($data['success'])) {
        if (isset($data['data'])) {
            $payload = $data['data'];

            if (is_array($payload)) {
                $possibleKeys = ['is_valid', 'isValid', 'verified', 'is_verified', 'connected', 'status'];

                foreach ($possibleKeys as $key) {
                    if (array_key_exists($key, $payload)) {
                        $is_verified = $normalize_flag($payload[$key]);
                        if ($is_verified) {
                            break;
                        }
                    }
                }

                if (!$is_verified && array_key_exists('details', $payload) && is_array($payload['details'])) {
                    $details = $payload['details'];
                    foreach ($possibleKeys as $key) {
                        if (array_key_exists($key, $details)) {
                            $is_verified = $normalize_flag($details[$key]);
                            if ($is_verified) {
                                break;
                            }
                        }
                    }
                }
            } else {
                $is_verified = $normalize_flag($payload);
            }
        }

        if (!$is_verified && isset($data['verified'])) {
            $is_verified = $normalize_flag($data['verified']);
        }

        if (!$is_verified && isset($data['status'])) {
            $is_verified = $normalize_flag($data['status']);
        }

        if (!$is_verified) {
            $messages = [];

            if (isset($data['message']) && is_string($data['message'])) {
                $messages[] = $data['message'];
            }

            if (isset($payload) && is_array($payload)) {
                if (isset($payload['message']) && is_string($payload['message'])) {
                    $messages[] = $payload['message'];
                }

                if (isset($payload['status_message']) && is_string($payload['status_message'])) {
                    $messages[] = $payload['status_message'];
                }
            }

            foreach ($messages as $messageText) {
                $lower = strtolower($messageText);
                $positive = (strpos($lower, 'valid') !== false)
                    || (strpos($lower, 'verified') !== false)
                    || (strpos($lower, 'connected') !== false)
                    || (strpos($lower, 'authorized') !== false);

                if ($positive && strpos($lower, 'not ') === false && strpos($lower, 'fail') === false) {
                    $is_verified = true;
                    break;
                }
            }
        }
    }
    
    // Cache result in transient for 1 hour
    if ($is_verified) {
        set_transient('hunnt_ai_google_token_verified', true, HOUR_IN_SECONDS);
        set_transient('hunnt_ai_google_token_verified_at', current_time('timestamp'), HOUR_IN_SECONDS);
    } else {
        delete_transient('hunnt_ai_google_token_verified');
        delete_transient('hunnt_ai_google_token_verified_at');
    }
    
    return $is_verified;
}

/**
 * Check if Google token is verified
 * Uses cached transient first, falls back to API if not cached
 * 
 * @param bool $force_refresh Force API check instead of using cache
 * @return bool True if token is valid, false otherwise
 */
function hunnt_ai_is_google_token_verified($force_refresh = false) {
    // Check cache first unless forced refresh
    if (!$force_refresh) {
        $cached = get_transient('hunnt_ai_google_token_verified');
        if ($cached !== false) {
            return (bool) $cached;
        }
    }
    
    // Cache miss or forced refresh - verify with API
    return hunnt_ai_verify_google_token_with_api();
}
