<?php
/**
 * AJAX Handlers for Google OAuth Integration
 * 
 * @package HunntAI
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

if (!function_exists('hunnt_ai_save_google_token')) {
    function hunnt_ai_save_google_token($agency_id, $token_data) {
        return [
            'success' => false,
            'message' => esc_html__('Token handler not implemented in this plugin build.', 'hunnt-ai')
        ];
    }
}

/**
 * AJAX: Save Google OAuth token
 */
add_action('wp_ajax_hunnt_ai_save_google_token', function() {
    // Security check
    check_ajax_referer('hunnt_ai_google_oauth_nonce', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => esc_html__('Unauthorized', 'hunnt-ai')]);
    }
    
    // Check if user is agency
    if (!hunnt_ai_is_agency_account()) {
        wp_send_json_error(['message' => esc_html__('This feature is only available for agency accounts', 'hunnt-ai')]);
    }
    
    // Get agency ID
    $agency_id = hunnt_ai_get_agency_id();
    if (empty($agency_id)) {
        wp_send_json_error(['message' => esc_html__('Agency ID not found', 'hunnt-ai')]);
    }
    
    // Get token data from request
    $token_data = [
        'access_token' => isset($_POST['access_token']) ? sanitize_text_field($_POST['access_token']) : '',
        'refresh_token' => isset($_POST['refresh_token']) ? sanitize_text_field($_POST['refresh_token']) : '',
        'expires_in' => isset($_POST['expires_in']) ? intval($_POST['expires_in']) : 3600,
        'refresh_token_expires_in' => isset($_POST['refresh_token_expires_in']) ? intval($_POST['refresh_token_expires_in']) : 2592000,
        'scope' => isset($_POST['scope']) ? sanitize_text_field($_POST['scope']) : 'https://www.googleapis.com/auth/webmasters https://www.googleapis.com/auth/webmasters.readonly https://www.googleapis.com/auth/indexing openid email profile',
        'token_type' => isset($_POST['token_type']) ? sanitize_text_field($_POST['token_type']) : 'Bearer'
    ];
    if (empty($token_data['access_token'])) {
        wp_send_json_error(['message' => esc_html__('Access token is required', 'hunnt-ai')]);
    }

    // Save token via API
    $result = hunnt_ai_save_google_token($agency_id, $token_data);
    
    if ($result['success']) {
        wp_send_json_success([
            'message' => $result['message'],
            'data' => isset($result['data']) ? $result['data'] : null
        ]);
    } else {
        wp_send_json_error([
            'message' => $result['message'],
            'errors' => isset($result['errors']) ? $result['errors'] : []
        ]);
    }
});

/**
 * AJAX: Verify Google OAuth token status
 */
add_action('wp_ajax_hunnt_ai_verify_google_token', function() {
    // Security check
    check_ajax_referer('hunnt_ai_google_oauth_nonce', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => esc_html__('Unauthorized', 'hunnt-ai')]);
    }
    
    // Check if user is agency
    if (!hunnt_ai_is_agency_account()) {
        wp_send_json_error(['message' => esc_html__('This feature is only available for agency accounts', 'hunnt-ai')]);
    }
    
    // Verify token via API (no agency_id needed - License key is tied to agency)
    $result = hunnt_ai_verify_google_token();
    
    if ($result['success']) {
        wp_send_json_success([
            'verified' => $result['verified'],
            'message' => $result['message'],
            'data' => isset($result['data']) ? $result['data'] : null
        ]);
    } else {
        wp_send_json_error([
            'message' => $result['message'],
            'verified' => false
        ]);
    }
});

/**
 * AJAX: Revoke Google OAuth token
 */
add_action('wp_ajax_hunnt_ai_revoke_google_token', function() {
    // Security check
    check_ajax_referer('hunnt_ai_google_oauth_nonce', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => esc_html__('Unauthorized', 'hunnt-ai')]);
    }
    
    // Check if user is agency
    if (!hunnt_ai_is_agency_account()) {
        wp_send_json_error(['message' => esc_html__('This feature is only available for agency accounts', 'hunnt-ai')]);
    }
    
    // Get agency ID
    $agency_id = hunnt_ai_get_agency_id();
    if (empty($agency_id)) {
        wp_send_json_error(['message' => esc_html__('Agency ID not found', 'hunnt-ai')]);
    }
    
    // Revoke token via API
    $result = hunnt_ai_revoke_google_token($agency_id);
    
    if ($result['success']) {
        wp_send_json_success([
            'message' => $result['message'],
            'data' => isset($result['data']) ? $result['data'] : null
        ]);
    } else {
        wp_send_json_error([
            'message' => $result['message']
        ]);
    }
});

/**
 * AJAX: Check user type and Google authentication status
 */
add_action('wp_ajax_hunnt_ai_check_google_auth_status', function() {
    // Security check
    check_ajax_referer('hunnt_ai_google_oauth_nonce', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => esc_html__('Unauthorized', 'hunnt-ai')]);
    }
    
    $is_agency = hunnt_ai_is_agency_account();
    $is_verified = hunnt_ai_is_google_token_verified();
    $user_id = hunnt_ai_get_user_id();
    
    wp_send_json_success([
        'is_agency' => $is_agency,
        'is_verified' => $is_verified,
        'user_id' => $user_id,
        'requires_auth' => ($is_agency && !$is_verified)
    ]);
});

/**
 * AJAX: Mark Google token as verified (called after successful OAuth from API)
 */
add_action('wp_ajax_hunnt_ai_mark_google_verified', function() {
    // Security check
    check_ajax_referer('hunnt_ai_google_oauth_nonce', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => esc_html__('Unauthorized', 'hunnt-ai')]);
    }
    
    // Store in transient (cache) for 1 hour instead of database
    set_transient('hunnt_ai_google_token_verified', true, HOUR_IN_SECONDS);
    set_transient('hunnt_ai_google_token_verified_at', current_time('timestamp'), HOUR_IN_SECONDS);
    
    wp_send_json_success(['message' => esc_html__('Google token verified', 'hunnt-ai')]);
});

/**
 * AJAX: Initiate Google OAuth flow via Hunnt AI API
 */
add_action('wp_ajax_hunnt_ai_initiate_google_oauth', function() {
    check_ajax_referer('hunnt_ai_google_oauth_nonce', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => esc_html__('Unauthorized', 'hunnt-ai')]);
    }
    
    if (!hunnt_ai_is_agency_account()) {
        wp_send_json_error(['message' => esc_html__('This feature is only available for agency accounts', 'hunnt-ai')]);
    }
    
    $api_key = get_option('hunnt_ai_api_key');
    if (empty($api_key)) {
        wp_send_json_error(['message' => esc_html__('License key not found', 'hunnt-ai')]);
    }
    
    // Build callback URL based on where user initiated OAuth
    $current_page = isset($_POST['current_page']) ? sanitize_text_field($_POST['current_page']) : 'hunnt-ai-google-oauth';
    $current_step = isset($_POST['current_step']) ? sanitize_text_field($_POST['current_step']) : '';
    
    $callback_params = ['page' => $current_page, 'google_callback' => '1'];
    if (!empty($current_step)) {
        $callback_params['step'] = $current_step;
    }
    
    $redirect_uri = admin_url('admin.php?' . http_build_query($callback_params));
    $oauth_url = hunnt_ai_get_api_base() . 'client/agency/google/oauth/authorize?' . http_build_query([
        'api_key' => $api_key,
        'redirect_uri' => $redirect_uri
    ]);
    
    wp_send_json_success(['oauth_url' => $oauth_url]);
});

/**
 * AJAX: Handle Google OAuth callback
 */
add_action('wp_ajax_hunnt_ai_handle_google_callback', function() {
    check_ajax_referer('hunnt_ai_google_oauth_nonce', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => esc_html__('Unauthorized', 'hunnt-ai')]);
    }
    
    $success = isset($_POST['success']) ? sanitize_text_field($_POST['success']) : '0';
    $message = isset($_POST['message']) ? sanitize_text_field($_POST['message']) : '';
    
    if ($success === '1') {
        // Verify token with API using reusable function (force refresh to get latest status)
        $is_verified = hunnt_ai_verify_google_token_with_api();
        
        error_log('Hunnt AI: Google OAuth callback - Token verified: ' . ($is_verified ? 'YES' : 'NO'));
        
        // Get user type
        $user_type = get_option('hunnt_ai_user_type', '');
        $is_agency = ($user_type === 'agency');
        
        // Don't auto-redirect - let user click Next button manually
        wp_send_json_success([
            'message' => $message ?: esc_html__('Successfully connected to Google', 'hunnt-ai'),
            'verified' => $is_verified,
            'is_agency' => $is_agency
        ]);
    } else {
        wp_send_json_error(['message' => $message ?: esc_html__('Failed to connect to Google', 'hunnt-ai')]);
    }
});


