<?php
// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

function hunnt_ai_get_api_base() {
    $base = defined('HUNNT_AI_ENDPOINT') ? HUNNT_AI_ENDPOINT : get_option('hunnt_ai_api_base', 'https://demo.hunnt.ai/api/v1/');
    $base = rtrim($base, '/') . '/';
    return apply_filters('hunnt_ai_api_base', $base);
}

function hunnt_ai_api_request($method, $path, $data = null, $extra_headers = []) {
    $api_key = get_option('hunnt_ai_api_key');
    $url = hunnt_ai_get_api_base() . ltrim($path, '/');
    $headers = array_merge([
        'Accept'       => 'application/json',
        'Content-Type' => 'application/json',
        'x-api-key'    => $api_key ? $api_key : '',
    ], $extra_headers);

    $timeout = (int) apply_filters('hunnt_ai_http_timeout', 5);
    if ($timeout <= 0) { $timeout = 5; }
    $args = [
        'method'  => strtoupper($method),
        'headers' => $headers,
        'timeout' => $timeout,
        'redirection' => 2,
        'httpversion' => '1.1',
    ];
    if (!is_null($data)) {
        $args['body'] = wp_json_encode($data);
    }

    $response = wp_remote_request($url, $args);
    if (is_wp_error($response)) {
        return [
            'code' => 0,
            'body' => null,
            'error' => $response->get_error_message(),
        ];
    }
    $code = (int) wp_remote_retrieve_response_code($response);
    $body_raw = wp_remote_retrieve_body($response);
    $body = null;
    if ($body_raw !== '') {
        $decoded = json_decode($body_raw, true);
        $body = is_array($decoded) ? $decoded : $body_raw;
    }
    return [
        'code' => $code,
        'body' => $body,
        'error' => null,
    ];
}

function hunnt_ai_get_pricing_url() {
    $default = 'https://hunnt.ai/en/pricing';
    $url = get_option('hunnt_ai_pricing_url', $default);
    return apply_filters('hunnt_ai_pricing_url', $url);
}

/**
 * Fetch and normalize subscription status from API
 * Returns: [
 *   'active' => bool,
 *   'status' => string, // e.g., 'active','inactive','unknown'
 *   'has_subscription' => bool,
 *   'features' => array,
 *   'raw' => mixed,
 *   'error' => string|null
 * ]
 */
function hunnt_ai_get_subscription_status() {
    $api_key = get_option('hunnt_ai_api_key');
    if (!$api_key) {
        return [ 'active' => false, 'status' => 'inactive', 'has_subscription' => false, 'features' => [], 'raw' => null, 'error' => 'api_key_missing' ];
    }
    $cache_key = 'hunnt_ai_sub_status_' . md5($api_key);
    $cached = get_transient($cache_key);
    if (is_array($cached)) {
        return $cached;
    }
    $resp = hunnt_ai_api_request('POST', 'subscription/check-subscription');
    if (empty($resp['code']) || $resp['code'] < 200 || $resp['code'] >= 300) {
        return [ 'active' => false, 'status' => 'unknown', 'has_subscription' => false, 'features' => [], 'raw' => $resp['body'] ?? null, 'error' => $resp['error'] ?? 'request_failed' ];
    }
    $body = is_array($resp['body']) ? $resp['body'] : [];
    $data = isset($body['data']) && is_array($body['data']) ? $body['data'] : [];
    $hasSub = (bool) ($data['has_subscription'] ?? false);
    $status = strtolower((string) ($data['subscription_status'] ?? ''));
    $planActive = isset($data['plan']['is_active']) ? (bool)$data['plan']['is_active'] : null;
    $expiresAt = $data['expires_at'] ?? null;
    $notExpired = empty($expiresAt) || (strtotime((string)$expiresAt) ?: 0) > time();
    $active = $hasSub && ($status === 'active') && ($planActive === null || $planActive === true) && $notExpired;
    $features = isset($data['features']) && is_array($data['features']) ? $data['features'] : [];
    $result = [
        'active' => (bool)$active,
        'status' => $status ?: ($active ? 'active' : 'inactive'),
        'has_subscription' => $hasSub,
        'features' => $features,
        'raw' => $body,
        'error' => null,
    ];
    //set_transient($cache_key, $result, (int) apply_filters('hunnt_ai_sub_status_ttl', 120));
    return $result;
}

/**
 * Fetch Technical SEO feature access
 * Returns: [ 'has' => bool, 'raw' => mixed, 'error' => string|null ]
 */
function hunnt_ai_get_technical_seo_access() {
    $api_key = get_option('hunnt_ai_api_key');
    if (!$api_key) return [ 'has' => false, 'raw' => null, 'error' => 'api_key_missing' ];
    $cache_key = 'hunnt_ai_tech_access_' . md5($api_key);
    $cached = get_transient($cache_key);
    if (is_array($cached)) {
        return $cached;
    }
    $resp = hunnt_ai_api_request('POST', 'subscription/technical-seo');
    if (empty($resp['code']) || $resp['code'] < 200 || $resp['code'] >= 300) {
        return [ 'has' => false, 'raw' => $resp['body'] ?? null, 'error' => $resp['error'] ?? 'request_failed' ];
    }
    $body = is_array($resp['body']) ? $resp['body'] : [];
    $data = isset($body['data']) && is_array($body['data']) ? $body['data'] : [];
    $has = isset($data['has_technical_seo']) ? (bool)$data['has_technical_seo'] : (bool)($data['feature_details']['is_active'] ?? false);
    $result = [ 'has' => $has, 'raw' => $body, 'error' => null ];
    set_transient($cache_key, $result, (int) apply_filters('hunnt_ai_tech_access_ttl', 120));
    return $result;
}
