<?php
/**
 * Elementor Compatibility Fix
 * 
 * Prevents duplicate control registration notices from Elementor Pro
 * by ensuring controls are only registered once per page load.
 * 
 * @package HunntAI
 * @since 1.0.0
 */

namespace HunntAI\Compatibility;

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

class Elementor_Fix {
    
    /**
     * Track registered controls to prevent duplicates
     * 
     * @var array
     */
    private static $registered_controls = [];
    
    /**
     * Initialize the fix
     */
    public static function init() {
        if ( ! did_action( 'elementor/loaded' ) ) {
            return;
        }
        
        // Suppress duplicate control notices
        add_action( 'elementor/element/before_add_attributes', [ __CLASS__, 'prevent_duplicate_controls' ], 5 );
        
        // Clear control cache on new page load
        add_action( 'wp', [ __CLASS__, 'reset_control_cache' ] );
    }
    
    /**
     * Prevent duplicate control registration
     * 
     * @param \Elementor\Element_Base $element
     */
    public static function prevent_duplicate_controls( $element ) {
        // Track this element to avoid re-registration
        $element_id = $element->get_unique_name();
        
        if ( isset( self::$registered_controls[ $element_id ] ) ) {
            // Element already processed, skip re-registration
            return;
        }
        
        self::$registered_controls[ $element_id ] = true;
    }
    
    /**
     * Reset control cache on new page load
     */
    public static function reset_control_cache() {
        self::$registered_controls = [];
    }
}

// Initialize on plugins_loaded
add_action( 'plugins_loaded', [ 'HunntAI\Compatibility\Elementor_Fix', 'init' ], 20 );
