<?php

class Hunnt_AI
{
    private static $cached_role = null;
    private static $total_articles = 0;
    private static $site_languages_list = [];

    static function init()
    {
        add_action('wp_footer', array(__CLASS__, 'frontend_output'));
        add_action('admin_menu', array(__CLASS__, 'add_admin_menu'));
        // Hook into admin_enqueue_scripts for class static method
        add_action('admin_enqueue_scripts', array(__CLASS__, 'hunnt_ai_enqueue_admin_assets'));
        // Hook into admin_init to handle wizard form submissions BEFORE any output
        add_action('admin_init', array(__CLASS__, 'handle_wizard_form_submissions'));
        if (isset($_GET['page'], $_GET['post_id']) && $_GET['page'] === 'hunnt-ai-restore-article') {
            // Ensure current user has permission
            if (!current_user_can('manage_options')) {
                wp_die(__('You are not allowed to access this page.', 'hunnt-ai'));
            }
            $post_id = intval($_GET['post_id']);
            // Use wp_redirect only after WordPress is fully loaded
            if ($post_id > 0) {
                hunnt_ai_restore_article($post_id); // This function will handle the redirect
            }
        }
        if (is_admin()) {
            self::get_user_role_from_api();
        }
    }

    //to check load agency or client
    public static function get_user_role_from_api()
    {
        // Return cached role if already fetched
        if (self::$cached_role !== null) {
            return self::$cached_role;
        }

        // Get API key
        $api_key = get_option('hunnt_ai_api_key');
        if (!$api_key) {
            self::$cached_role = 'error';
            return 'error';
        }

        // Language header
        $current_lang = hunnt_ai_get_current_language();
        $accept_language = substr($current_lang, 0, 2) ?: 'en';

        // Call API
        $response = wp_remote_get(HUNNT_AI_ENDPOINT . 'api-key/check?with_articles=yes', [
            'headers' => [
                'x-api-key' => $api_key,
                'Content-Type' => 'application/json',
                'Accept-Language' => $accept_language,
            ],
            'timeout' => 10
        ]);
        // If request fails or HTTP code not 200, return 'error'
        if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
            self::$cached_role = 'error';
            return 'error';
        }

        // Decode JSON
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            self::$cached_role = 'error';
            return 'error';
        }

        // Extract role (client or agency)
        $role = $data['data']['user_type'] ?? 'error';
        $total_articles = 0;
        $site_languages_list = [];
        if ($role == "agency") {
            $total_articles = (isset($data['data']['total_articles']) && !empty($data['data']['total_articles'])) ? $data['data']['total_articles'] : 0;
            $site_languages_list = (isset($data['data']['languages']) && !empty($data['data']['languages'])) ? $data['data']['languages'] : [];
        }

        // Cache & return
        self::$cached_role = $role;
        self::$total_articles = $total_articles;
        self::$site_languages_list = $site_languages_list;
        return [
            'role' => $role,
            'total_articles' => $total_articles,
            'site_languages_list' => $site_languages_list
        ];
    }


    /**
     * Handle wizard form submissions early before any output
     * This prevents "headers already sent" errors
     */
    public static function handle_wizard_form_submissions()
    {
        // Check if we're on the wizard page
        if (!isset($_GET['page']) || $_GET['page'] !== 'hunnt-ai') {
            return;
        }

        // Check if this is a POST request
        if ('POST' !== ($_SERVER['REQUEST_METHOD'] ?? '')) {
            return;
        }

        // Get current step
        $current_step = isset($_GET['step']) ? intval($_GET['step']) : 1;

        // Handle Step 3 form submission
        if (
            $current_step === 3 &&
            isset($_POST['hunnt_ai_nonce']) &&
            wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['hunnt_ai_nonce'])), 'hunnt_ai_step3')
        ) {
            // Verify user permissions
            if (!current_user_can('manage_options')) {
                wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'hunnt-ai'));
            }

            // Process the form data
            $saved_plan = [];
            $exclude = ['attachment', 'revision', 'nav_menu_item', 'wp_template', 'wp_template_part'];
            $allowed_types = get_post_types(['public' => true], 'names');
            foreach ($exclude as $ex) {
                if (isset($allowed_types[$ex])) {
                    unset($allowed_types[$ex]);
                }
            }

            $enabled = isset($_POST['plan_enabled']) ? (array)$_POST['plan_enabled'] : [];
            $counts = isset($_POST['plan_count']) ? (array)$_POST['plan_count'] : [];

            $plan = [];
            foreach ($allowed_types as $type_key) {
                $is_enabled = isset($enabled[$type_key]) ? 1 : 0;
                $count = isset($counts[$type_key]) ? max(0, intval($counts[$type_key])) : 0;
                if ($is_enabled && $count > 0) {
                    $plan[$type_key] = ['enabled' => 1, 'count' => $count];
                }
            }

            // Save the plan
            update_option('hunnt_ai_new_posts_plan', $plan);

            // Redirect to step 4
            wp_safe_redirect(admin_url('admin.php?page=hunnt-ai&step=4'));
            exit;
        }
    }

    public static function activate()
    {
        update_option('hunnt_ai_api_key', '');

        if (class_exists('Hunnt_AI_Static_Cache_Manager')) {
            Hunnt_AI_Static_Cache_Manager::on_activation();
        }
    }

    public static function deactivate()
    {
        delete_option('hunnt_ai_api_key');

        if (class_exists('Hunnt_AI_Static_Cache_Manager')) {
            Hunnt_AI_Static_Cache_Manager::on_deactivation();
        }
    }

    public static function frontend_output()
    {
        echo '';
    }

    public static function add_admin_menu()
    {
        add_menu_page(
            esc_html__('Hunnt AI Settings', 'hunnt-ai'),
            esc_html__('Hunnt AI', 'hunnt-ai'),
            'manage_options',
            'hunnt-ai',
            array(__CLASS__, 'admin_page'),
            'dashicons-admin-generic',
            100
        );
        add_submenu_page(
            null, // 👈 no parent => hidden from menu
            esc_html__('Hunnt AI Settings', 'hunnt-ai'),
            esc_html__('Hunnt AI', 'hunnt-ai'),
            'manage_options',
            'hunnt-ai-settings', // 👈 slug you'll use in redirect
            array(__CLASS__, 'admin_page')
        );
        //to generate a new article
        add_submenu_page(
            'hunnt-ai', // parent slug
            esc_html__('Generate Articles', 'hunnt-ai'),
            esc_html__('Generate Articles', 'hunnt-ai'),
            'manage_options',
            'hunnt-ai-generate-articles',
            array(__CLASS__, 'admin_page_generate_articles')
        );
        // Amjad Improve
        add_submenu_page(
            'hunnt-ai', // parent slug
            esc_html__('Improve Articles', 'hunnt-ai'),
            esc_html__('Improve Articles', 'hunnt-ai'),
            'manage_options',
            'hunnt-ai-improve-articles',
            array(__CLASS__, 'admin_page_improve_articles')
        );
    }

    public static function improve_articles_page()
    {
        // Check if wizard is completed
        $wizard_completed = get_option('hunnt_ai_wizard_completed', false);

        if (!$wizard_completed) {
            // Redirect to wizard if not completed
            echo '<div class="wrap">';
            echo '<h1>' . esc_html__('Improve Articles', 'hunnt-ai') . '</h1>';
            echo '<div class="notice notice-warning">';
            echo '<p>' . esc_html__('You must complete the setup wizard before accessing this feature.', 'hunnt-ai') . '</p>';
            echo '<p><a href="' . admin_url('admin.php?page=hunnt-ai') . '" class="button button-primary">' . esc_html__('Complete Setup Wizard', 'hunnt-ai') . '</a></p>';
            echo '</div>';
            echo '</div>';
            return;
        }

        // Load improve articles template
        $template = HUNNT_AI_PATH . 'templates/improve-articles.php';
        if (file_exists($template)) {
            include $template;
        }
    }

    public static function admin_page()
    {
        $template = HUNNT_AI_PATH . 'templates/admin-page.php';
        if (file_exists($template)) {
            include $template;
        }
    }

    public static function admin_page_generate_articles()
    {
        // Check if wizard is completed
        $wizard_completed = get_option('hunnt_ai_wizard_completed', false);

        if (!$wizard_completed) {
            // Redirect to wizard if not completed
            echo '<div class="wrap">';
            echo '<h1>' . esc_html__('Improve Articles', 'hunnt-ai') . '</h1>';
            echo '<div class="notice notice-warning">';
            echo '<p>' . esc_html__('You must complete the setup wizard before accessing this feature.', 'hunnt-ai') . '</p>';
            echo '<p><a href="' . admin_url('admin.php?page=hunnt-ai') . '" class="button button-primary">' . esc_html__('Complete Setup Wizard', 'hunnt-ai') . '</a></p>';
            echo '</div>';
            echo '</div>';
            return;
        }
        if (self::$cached_role == "error") {
            $template = HUNNT_AI_PATH . 'templates/error/api_key_error.php';
        }
        if (self::$cached_role == "client") {
            $template = HUNNT_AI_PATH . 'templates/generate-article.php';
        }
        if (self::$cached_role == "agency") {
            $articles_count = self::$total_articles;
            $site_languages_list = self::$site_languages_list;
            $template = HUNNT_AI_PATH . 'templates/agency/generate-article.php';
        }
        if (file_exists($template)) {
            include $template;
        }
    }

    public static function admin_page_improve_articles()
    {
        $template = HUNNT_AI_PATH . 'templates/improve-article.php';
        if (file_exists($template)) {
            include $template;
        }
    }

    public static function hunnt_ai_enqueue_admin_assets($hook)
    {
        // Optional: Load only on your plugin page
        if (
            !in_array($hook, [
                'toplevel_page_hunnt-ai',
                'admin_page_hunnt-ai-settings',
                'hunnt-ai_page_hunnt-ai-improve-articles',
                'hunnt-ai_page_hunnt-ai-generate-articles'
            ])
        ) return;

        // Check if wizard is completed - if not, don't load script.js to avoid conflicts
        $wizard_completed = get_option('hunnt_ai_wizard_completed', false);

        if ($wizard_completed) {
            // Enqueue CSS
            wp_enqueue_style(
                'hunnt-ai-css',
                HUNNT_AI_URL . 'assets/css/style.css?v=' . strtotime('now'),
                array(),
                null
            );
            wp_enqueue_style(
                'hunnt-ai-wizard-css',
                HUNNT_AI_URL . 'assets/css/wizard.css?v=' . strtotime('now'),
                array(),
                null
            );
            // Enqueue JS
            wp_enqueue_script(
                'hunnt-ai-js',
                HUNNT_AI_URL . 'assets/js/script.js?v=' . strtotime('now'),
                array('jquery'),
                null,
                true
            );
            wp_localize_script('hunnt-ai-js', 'hunntAiStrings', [
                'show' => esc_html__('Show Posts List', 'hunnt-ai'),
                'hide' => esc_html__('Hide Posts List', 'hunnt-ai'),
            ]);
            $hunnt_ajax_nonce = wp_create_nonce('hunnt_wizard_nonce');
            wp_localize_script('hunnt-ai-js', 'hunntAiAdmin', [
                'ajaxUrl' => admin_url('admin-ajax.php'),
                'nonce' => $hunnt_ajax_nonce,
                'locale' => get_locale(),
                'dateTimeOptions' => [
                    'year' => 'numeric',
                    'month' => 'short',
                    'day' => '2-digit',
                    'hour' => '2-digit',
                    'minute' => '2-digit',
                ],
                'labels' => [
                    'showLog' => esc_html__('Show Optimization Log', 'hunnt-ai'),
                    'hideLog' => esc_html__('Hide Optimization Log', 'hunnt-ai'),
                    'enabled' => esc_html__('Enabled', 'hunnt-ai'),
                    'disabled' => esc_html__('Disabled', 'hunnt-ai'),
                    'enableStaticCache' => esc_html__('Enable Static Cache', 'hunnt-ai'),
                    'disableStaticCache' => esc_html__('Disable Static Cache', 'hunnt-ai'),
                    'warmCacheSuccess' => esc_html__('Cache warmed successfully.', 'hunnt-ai'),
                    'warmCacheError' => esc_html__('Failed to warm cache.', 'hunnt-ai'),
                    'purgeCacheSuccess' => esc_html__('Cache purged successfully.', 'hunnt-ai'),
                    'purgeCacheError' => esc_html__('Failed to purge cache.', 'hunnt-ai'),
                    'toggleEnableSuccess' => esc_html__('Static cache enabled.', 'hunnt-ai'),
                    'toggleDisableSuccess' => esc_html__('Static cache disabled.', 'hunnt-ai'),
                    'toggleError' => esc_html__('Unable to update static cache setting.', 'hunnt-ai'),
                    'rebuildSuccess' => esc_html__('Optimization assets rebuilt.', 'hunnt-ai'),
                    'rebuildError' => esc_html__('Failed to rebuild optimization assets.', 'hunnt-ai'),
                    'warmupSuccess' => esc_html__('Warmup triggered.', 'hunnt-ai'),
                    'warmupError' => esc_html__('Failed to start warmup.', 'hunnt-ai'),
                    'genericError' => esc_html__('Something went wrong.', 'hunnt-ai'),
                    'notAvailable' => esc_html__('n/a', 'hunnt-ai'),
                    'success' => esc_html__('Success', 'hunnt-ai'),
                    'failed' => esc_html__('Failed', 'hunnt-ai'),
                    'unknown' => esc_html__('Unknown', 'hunnt-ai'),
                    'loading' => esc_html__('Working…', 'hunnt-ai'),
                    'logEmpty' => esc_html__('No optimization logs recorded yet.', 'hunnt-ai'),
                    'on' => esc_html__('On', 'hunnt-ai'),
                    'off' => esc_html__('Off', 'hunnt-ai'),
                    'yes' => esc_html__('Yes', 'hunnt-ai'),
                    'no' => esc_html__('No', 'hunnt-ai'),
                    'days' => esc_html__('days', 'hunnt-ai'),
                    'hours' => esc_html__('hours', 'hunnt-ai'),
                    'minutes' => esc_html__('minutes', 'hunnt-ai'),
                    'seconds' => esc_html__('seconds', 'hunnt-ai'),
                    'requiresTechSeo' => esc_html__('Enable Technical SEO to use static cache.', 'hunnt-ai'),
                    'cachePermissionError' => esc_html__('Filesystem permissions prevent static cache from running.', 'hunnt-ai'),
                    'cachePermissionFix' => esc_html__('Hunnt AI cannot create its static cache folders. Please update the filesystem permissions and try again.', 'hunnt-ai'),
                    'writableLabel' => esc_html__('Writable', 'hunnt-ai'),
                    'notWritableLabel' => esc_html__('Not writable', 'hunnt-ai'),
                ],
            ]);
            //for client
            if ($hook == "hunnt-ai_page_hunnt-ai-generate-articles" && self::$cached_role == "client") {
                //load new articles js
                wp_enqueue_script(
                    'hunnt-ai-generate-article-js',
                    HUNNT_AI_URL . 'assets/js/article/generate-article.js?v=' . strtotime('now'),
                    array('jquery'),
                    null,
                    true
                );
                wp_enqueue_style(
                    'hunnt-ai-generate-article-css',
                    HUNNT_AI_URL . 'assets/css/article/generate-article.css?v=' . strtotime('now'),
                    [],
                    filemtime(HUNNT_AI_PATH . 'assets/css/article/generate-article.css')
                );
                wp_localize_script('hunnt-ai-generate-article-js', 'ApiArticlesAdmin', [
                    'ajax_url' => admin_url('admin-ajax.php'),
                    'nonce' => wp_create_nonce('api_articles_admin_nonce'),
                    'select_text' => esc_html__('--Select Option--', 'hunnt-ai'), // <-- add this
                ]);
            }
            //for agency
            if ($hook == "hunnt-ai_page_hunnt-ai-generate-articles" && self::$cached_role == "agency") {
                //load new articles js
                wp_enqueue_script(
                    'hunnt-ai-agency-generate-article-js',
                    HUNNT_AI_URL . 'assets/js/article/agency-generate-article.js?v=' . strtotime('now'),
                    array('jquery'),
                    null,
                    true
                );
                wp_enqueue_style(
                    'hunnt-ai-generate-article-css',
                    HUNNT_AI_URL . 'assets/css/article/agency/agency-generate-article.css?v=' . strtotime('now'),
                    [],
                    filemtime(HUNNT_AI_PATH . 'assets/css/article/agency/agency-generate-article.css')
                );
                wp_localize_script('hunnt-ai-agency-generate-article-js', 'AgencyGenerateArticle', [
                    'ajax_url' => admin_url('admin-ajax.php'),
                    'nonce' => wp_create_nonce('agency_articles_admin_nonce'),
                ]);
            }

        }
    }
}
