<?php
if (!defined('ABSPATH')) exit;

class API_insert_schedule_article {

    public static function init() {
        // Register REST API endpoint
        add_action('rest_api_init', [__CLASS__, 'register_routes']);
    }

    /**
     * Register REST API routes
     */
    public static function register_routes() {
        register_rest_route('hunnt-ai/v1', '/insert-article-with-schedule', [
            'methods'  => 'POST',
            'callback' => [__CLASS__, 'insert_article_with_schedule'],
            'permission_callback' => '__return_true', // Use License key check inside callback
        ]);
    }

    /**
     * Insert article from API request
     */
    public static function insert_article_with_schedule($request) {
        // Get raw request body
        $body = json_decode(file_get_contents('php://input'), true);
        // 1️⃣ Validate License key
        $api_key = $request->get_header('x-api-key');
        if ($api_key !== get_option('hunnt_ai_api_key')) {
            return  hunnt_ai_response(
                ['error' => 'Invalid License key'],
                401
            );
        }

        $title   = sanitize_text_field($request->get_param('title'));
        $content = wp_kses_post($request->get_param('content'));
        $post_date = $request->get_param('post_date');
        $timezone = $request->get_param('timezone');
        $categories = $request->get_param('categories');
        $language = $request->get_param('language');
        $post_status = $request->get_param('status');
        $default_lang = hunnt_ai_get_article_language();
        if(empty($language)){
            $language = $default_lang;
        }

        if (empty($title) || empty($content)) {
            return hunnt_ai_response(
                ['error' => 'Title and content are required'],
                400
            );
        }
        //set categories
        $category_name = (!empty($categories) && isset($categories[0])) ? $categories[0]:'';
        $cat = hunnt_ai_get_category($category_name, $language);
        // 3️⃣ Prepare post data
        $post_data = array(
            'post_title'    => $title,
            'post_content'  => $content,
            'post_status'   => $post_status, // important for scheduling
            'post_author'   => 1,        // change author ID if needed
            'post_type'     => 'post',
            'post_date'     => $post_date,
            'post_category' => array($cat['term_id'])
        );

        // 4️⃣ Insert
        $new_post_id = wp_insert_post($post_data, true);
        $action = 'inserted';

        if (is_wp_error($new_post_id)) {
            return hunnt_ai_response(
                [
                    'error'=>$new_post_id->get_error_message()
                ],
                500
            );
        }

        //download images and change content
        if (isset($body['images']) && !empty($body['images'])) {
            //check if has space before download
            $check = smr_check_disk_space_before_download($body['images']);
            if($check['success']) {
                $new_images = Hunnt_AI_API::download_images($body['images'], $new_post_id);
                $content = insert_inline_images($content, $new_images);
                // Update post content with new image URLs
                wp_update_post([
                    'ID' => $new_post_id,
                    'post_content' => $content,
                ]);
            }else{
                send_notification_to_hunnt_ai([
                    'title'=>NotificationEnum::$no_enough_space,
                    'trigger'=>NotificationEnum::$insert_article_trigger,
                    'post_id'=>$new_post_id
                ]);
            }
        }


        // 6️⃣ Return response
        $data = hunnt_ai_post_response([
            'post_id'=>$new_post_id,
            'action'=>'Inserted',
            'success'=>true
        ]);
        return hunnt_ai_response($data);
    }
}

// Initialize
API_insert_schedule_article::init();