<?php

if (!class_exists('Hunnt_AI_API')) {

    class Hunnt_AI_API
    {

        public static function init()
        {
            // Register REST API endpoint
            add_action('rest_api_init', [__CLASS__, 'register_routes']);
        }

        /**
         * Register REST API routes
         */
        public static function register_routes()
        {
            register_rest_route('hunnt-ai/v1', '/insert-article', [
                'methods' => 'POST',
                'callback' => [__CLASS__, 'insert_article'],
                'permission_callback' => '__return_true', // Use License key check inside callback
            ]);
        }

        /**
         * Insert article from API request
         */
        public static function insert_article($request)
        {
            // Get raw request body
            $body = json_decode(file_get_contents('php://input'), true);
            // 1️⃣ Validate License key
            $api_key = $request->get_header('x-api-key');
            if ($api_key !== get_option('hunnt_ai_api_key')) {
                return  hunnt_ai_response(
                    ['error' => 'Invalid License key'],
                    401
                );
            }

            // 2️⃣ Get and sanitize parameters
            $title = sanitize_text_field($request->get_param('title'));
            $content = wp_kses_post($request->get_param('content'));
            $status = sanitize_text_field($request->get_param('status')) ?: 'publish';
            $lang = sanitize_text_field($request->get_param('lang')) ?: get_locale();

            if (empty($title) || empty($content)) {
                return hunnt_ai_response(
                    ['error' => 'Title and content are required'],
                    400
                );
            }
            // 3️⃣ Insert post
            $post_id = wp_insert_post([
                'post_title' => $title,
                'post_content' => $content,
                'post_status' => $status,
                'post_type' => 'post',
            ]);

            if (is_wp_error($post_id)) {
                return hunnt_ai_response(
                    [
                        'error'=>$post_id->get_error_message()
                    ],
                    500
                );
            }
            //download images and change content
            if (isset($body['images']) && !empty($body['images'])) {
                //check if has space before download
                $check = smr_check_disk_space_before_download($body['images']);
                if($check['success']) {
                    $new_images = self::download_images($body['images'], $post_id);
                    $content = insert_inline_images($content, $new_images);
                    // Update post content with new image URLs
                    wp_update_post([
                        'ID' => $post_id,
                        'post_content' => $content,
                    ]);
                }else{
                    send_notification_to_hunnt_ai([
                        'title'=>NotificationEnum::$no_enough_space,
                        'trigger'=>NotificationEnum::$insert_article_trigger,
                        'post_id'=>$post_id
                    ]);
                }
            }
            // 6️⃣ Return response
            $data = hunnt_ai_post_response([
                'post_id'=>$post_id,
                'action'=>'Inserted',
                'success'=>true
            ]);
            return hunnt_ai_response($data);
        }

        public static function download_images($images,$post_id)
        {
            require_once(ABSPATH . 'wp-admin/includes/file.php');
            require_once(ABSPATH . 'wp-admin/includes/media.php');
            require_once(ABSPATH . 'wp-admin/includes/image.php');
            foreach ($images as $index=>$image) {
                $image_url = $image['url'];
                $alt_text = isset($image['alt']) ? $image['alt'] : '';
                $tmp_file = wp_tempnam($image_url);

                $ch = curl_init($image_url);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
                curl_setopt($ch, CURLOPT_TIMEOUT, 20);
                $data = curl_exec($ch);
                $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                curl_close($ch);

                if ($http_code !== 200 || !$data) {
                    return new WP_Error('download_failed', 'Unable to download image.');
                }

                file_put_contents($tmp_file, $data);

                $file_array = [
                    'name' => basename($image_url),
                    'tmp_name' => $tmp_file,
                ];
                // Insert attachment into WordPress media library
                $attachment_id = media_handle_sideload($file_array, 0); // 0 = unattached
                @unlink($tmp_file);
                if (is_wp_error($attachment_id)) {
                    continue; // Skip if media insertion fails
                }
                // Update alt text
                if (!empty($alt_text)) {
                    update_post_meta($attachment_id, '_wp_attachment_image_alt', sanitize_text_field($alt_text));
                    // Update attachment title
                    wp_update_post([
                        'ID'         => $attachment_id,
                        'post_title' => sanitize_text_field($alt_text),
                    ]);
                }
                // Get the new URL
                $new_url = wp_get_attachment_url($attachment_id);
                $images[$index]['url'] = $new_url;
                $images[$index]['attachment_id'] = $attachment_id;
                // **Set as featured image if type is 'feature'**
                if (isset($image['type']) && $image['type'] === 'feature' && isset($post_id)) {
                    set_post_thumbnail($post_id, $attachment_id);
                }
            }

            return  $images;
        }
    }

    // Initialize
    Hunnt_AI_API::init();
}
