<?php
if (!defined('ABSPATH')) exit;

class API_Improve_Article {

    public static function init() {
        // Register REST API endpoint
        add_action('rest_api_init', [__CLASS__, 'register_routes']);
    }

    /**
     * Register REST API routes
     */
    public static function register_routes() {
        register_rest_route('hunnt-ai/v1', '/improve-article', [
            'methods'  => 'POST',
            'callback' => [__CLASS__, 'improve_article'],
            'permission_callback' => '__return_true', // Use License key check inside callback
        ]);
    }

    /**
     * Insert article from API request
     */
    public static function improve_article($request) {

        // Get raw request body
        $body = json_decode(file_get_contents('php://input'), true);
        // 1️⃣ Validate License key
        $api_key = $request->get_header('x-api-key');
        if ($api_key !== get_option('hunnt_ai_api_key')) {
            return  hunnt_ai_response(
                ['error' => 'Invalid License key'],
                401
            );
        }

        // 2️⃣ Get and sanitize parameters
        $post_id = absint($request->get_param('client_article_id'));
        $title   = sanitize_text_field($request->get_param('title'));
        $content = wp_kses_post($request->get_param('content'));
        $categories = $request->get_param('categories');
        $language = $request->get_param('language');
        $default_lang = hunnt_ai_get_article_language();
        if(empty($language)){
            $language = $default_lang;
        }

        if (empty($title) || empty($content)) {
            return  hunnt_ai_response(
                ['error' => 'Title and content are required'],
                400
            );
        }
        //set categories
        $category_name = (!empty($categories) && isset($categories[0])) ? $categories[0]:'';
        $cat = hunnt_ai_get_category($category_name, $language);
        // 3️⃣ Prepare post data
        $post_data = [
            'post_title'   => $title,
            'post_content' => $content,
            'post_type'    => 'post',
            'post_category' => array($cat['term_id']),
            'post_name'     => sanitize_title_with_dashes($title),
        ];

        // 4️⃣ Insert or update post
        if ($post_id && get_post($post_id)) {
            $post_data['ID'] = $post_id;
            $new_post_id = wp_update_post($post_data, true);
            $action = 'updated';
        } else {
            $new_post_id = wp_insert_post($post_data, true);
            $action = 'inserted';
        }

        if (is_wp_error($new_post_id)) {
            return  hunnt_ai_response(
                ['error' => $new_post_id->get_error_message()],
                500
            );
        }
        // ✅ 5️⃣ Save the slug (the one generated from your AI side)
        if (!is_wp_error($new_post_id)) {
            $generated_slug = $post_data['post_name'];

            // Save it as post meta so you can use it later (e.g. for redirect)
            update_post_meta($new_post_id, '_hunnt_ai_original_slug', $generated_slug);

            // You can also store the full URL, just in case
            $generated_url = get_permalink($new_post_id);
            update_post_meta($new_post_id, '_hunnt_ai_original_url', $generated_url);
        }



        // 5️⃣ Add custom post meta to mark improvement
        update_post_meta($new_post_id, '_improved_by_hunnt_ai_api', current_time('mysql'));

        // (Optional: external_id support if API sends it)
        if ($request->get_param('hunnt_ai_id')) {
            update_post_meta($new_post_id, '_hunnt_ai_id', sanitize_text_field($request->get_param('hunnt_ai_id')));
        }
        //remove send to ai meta
        delete_post_meta($post_id, '_send_to_hunnt_ai_api');

        //download images and change content
        if (isset($body['images']) && !empty($body['images'])) {
            //check if has space before download
            $check = smr_check_disk_space_before_download($body['images']);
            if($check['success']) {
                $new_images = Hunnt_AI_API::download_images($body['images'], $new_post_id);
                $content = insert_inline_images($content, $new_images);
                // Update post content with new image URLs
                wp_update_post([
                    'ID' => $new_post_id,
                    'post_content' => $content,
                ]);
            }else{
                send_notification_to_hunnt_ai([
                    'title'=>NotificationEnum::$no_enough_space,
                    'trigger'=>NotificationEnum::$insert_article_trigger,
                    'post_id'=>$new_post_id
                ]);
            }
        }


        // 6️⃣ Return response
        $data = hunnt_ai_post_response([
            'post_id'=>$new_post_id,
            'action'=>(isset($action)) ? $action:'',
            'success'=>true
        ]);
        return hunnt_ai_response($data);
    }
}

// Initialize
API_Improve_Article::init();
