<?php
/**
 * Admin Settings Page for Google OAuth Configuration
 * 
 * @package HunntAI
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Enqueue Google OAuth JavaScript
 */
add_action('admin_enqueue_scripts', function($hook) {
    // Only load on Hunnt AI pages
    if (strpos($hook, 'hunnt-ai') === false) {
        return;
    }
    
    // Enqueue Google OAuth script
    wp_enqueue_script(
        'hunnt-ai-google-oauth',
        HUNNT_AI_URL . 'assets/js/google-oauth.js',
        ['jquery'],
        '1.1.0',
        true
    );
    
    // Localize script with nonce and AJAX URL
    wp_localize_script('hunnt-ai-google-oauth', 'hunntAIGoogleOAuth', [
        'nonce' => wp_create_nonce('hunnt_ai_google_oauth_nonce'),
        'ajaxurl' => admin_url('admin-ajax.php'),
        'labels' => [
            'connect' => esc_html__('Connect Google Account', 'hunnt-ai'),
            'connecting' => esc_html__('Connecting...', 'hunnt-ai'),
            'failed' => esc_html__('Failed to initiate OAuth. Please try again.', 'hunnt-ai')
        ],
        'noticeTarget' => '.wrap',
        'noticeRenderer' => 'admin',
        'useAlerts' => true,
        'refreshOnSuccess' => true,
        'refreshDelay' => 200
    ]);
});

/**
 * Add Google OAuth settings submenu - DISABLED (moved to main Hunnt AI settings Overview tab)
 */
/*
add_action('admin_menu', function() {
    add_submenu_page(
        'hunnt-ai', // Parent slug
        __('Hunnt AI Settings', 'hunnt-ai'),
        __('Hunnt AI Settings', 'hunnt-ai'),
        'manage_options',
        'hunnt-ai-google-oauth',
        'hunnt_ai_google_oauth_settings_page'
    );
}, 100);
*/

/**
 * Render Google OAuth settings page - Simplified version
 */
function hunnt_ai_google_oauth_settings_page( $args = [] ) {
    if ( ! is_array( $args ) ) {
        $args = [ 'layout' => $args ];
    }

    $args = function_exists( 'wp_parse_args' ) ? wp_parse_args( $args, [ 'layout' => 'page' ] ) : [ 'layout' => 'page' ];
    $layout = isset( $args['layout'] ) ? $args['layout'] : 'page';

    // Check user capabilities
    if (!current_user_can('manage_options')) {
        wp_die(__('You do not have sufficient permissions to access this page.', 'hunnt-ai'));
    }

    // Get current values
    $user_type = get_option('hunnt_ai_user_type', '');
    $user_id = get_option('hunnt_ai_user_id', '');
    $token_verified = hunnt_ai_is_google_token_verified();
    
    $is_tab_layout = ( 'tab' === $layout );

    if ( ! $is_tab_layout ) :
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
        <?php
    endif;

    ?>
    <div class="hunnt-ai-google-settings" style="max-width: 100%;">
        <?php if ($user_type !== 'agency'): ?>
            <div class="hunnt-ai-status-card hunnt-ai-warning" style="border:1px solid #dcbc88; border-left:4px solid #dba617; background:#fff8e5; padding:15px; margin-bottom:20px;">
                <p style="margin:0 0 6px; font-weight:600; color:#7a5200;">
                    <?php esc_html_e('Google OAuth is only available for agency accounts.', 'hunnt-ai'); ?>
                </p>
                <p style="margin:0; color:#665200;">
                    <?php esc_html_e('Current account type:', 'hunnt-ai'); ?>
                    <strong><?php echo esc_html($user_type ?: __('Not detected', 'hunnt-ai')); ?></strong>
                </p>
                <p style="margin:6px 0 0; color:#665200;">
                    <?php esc_html_e('Please validate your License key first if you have an agency account.', 'hunnt-ai'); ?>
                </p>
            </div>
        <?php else: ?>
            <!-- Agency Account Detected Status -->
            <div class="hunnt-ai-status-card hunnt-ai-success" style="border:1px solid #b3d4fc; border-left:4px solid #1e8cbe; background:#f5fbff; padding:15px; margin-bottom:20px;margin-top:10px;">
                <p style="margin:0 0 6px; font-weight:600; color:#0b5585;">
                    <?php esc_html_e('Agency Account Detected', 'hunnt-ai'); ?>
                </p>
                <p style="margin:0; color:#0b5585;">
                    <?php esc_html_e('Google Token Status:', 'hunnt-ai'); ?>
                    <?php if ($token_verified): ?>
                        <span style="color: #008000; font-weight: 600;">✓ <?php esc_html_e('Connected', 'hunnt-ai'); ?></span>
                    <?php else: ?>
                        <span style="color: #d63638; font-weight: 600;">✗ <?php esc_html_e('Not Connected', 'hunnt-ai'); ?></span>
                    <?php endif; ?>
                </p>
            </div>

            <?php if (!$token_verified): ?>
                <!-- Google Connect Prompt -->
                <div class="card" style="max-width: 100%; margin-top: 20px;">
                    <h2 class="title"><?php esc_html_e('Connect Google Account', 'hunnt-ai'); ?></h2>
                    <p><?php esc_html_e('Sign in with your Google account to enable Search Console features for your clients.', 'hunnt-ai'); ?></p>
                    <button type="button" class="button button-primary hunnt-ai-connect-google">
                        <?php esc_html_e('Connect Google Account', 'hunnt-ai'); ?>
                    </button>
                </div>
            <?php elseif (!empty($user_id)): ?>
                <!-- Google Account Management -->
                <div class="card" style="max-width: 100%; margin-top: 20px;">
                    <h2 class="title"><?php esc_html_e('Google Account Management', 'hunnt-ai'); ?></h2>
                    
                    <p><?php esc_html_e('Your Google account is currently connected.', 'hunnt-ai'); ?></p>
                    <button type="button" id="hunnt-ai-revoke-google-token" class="button button-secondary">
                        <?php esc_html_e('Disconnect Google Account', 'hunnt-ai'); ?>
                    </button>
                    <div id="revoke-message" style="margin-top: 10px;"></div>
                    
                    <script>
                    jQuery(document).ready(function($) {
                        $('#hunnt-ai-revoke-google-token').on('click', function() {
                            if (!confirm('<?php esc_html_e('Are you sure you want to disconnect your Google account?', 'hunnt-ai'); ?>')) {
                                return;
                            }
                            
                            var $btn = $(this);
                            $btn.prop('disabled', true).text('<?php esc_html_e('Disconnecting...', 'hunnt-ai'); ?>');
                            
                            $.ajax({
                                url: ajaxurl,
                                type: 'POST',
                                data: {
                                    action: 'hunnt_ai_revoke_google_token',
                                    nonce: '<?php echo wp_create_nonce('hunnt_ai_google_oauth_nonce'); ?>'
                                },
                                success: function(res) {
                                    if (res.success) {
                                        $('#revoke-message').html('<div class="notice notice-success"><p>' + res.data.message + '</p></div>');
                                        setTimeout(function() { location.reload(); }, 1500);
                                    } else {
                                        $('#revoke-message').html('<div class="notice notice-error"><p>' + res.data.message + '</p></div>');
                                        $btn.prop('disabled', false).text('<?php esc_html_e('Disconnect Google Account', 'hunnt-ai'); ?>');
                                    }
                                },
                                error: function() {
                                    $('#revoke-message').html('<div class="notice notice-error"><p><?php esc_html_e('Request failed. Please try again.', 'hunnt-ai'); ?></p></div>');
                                    $btn.prop('disabled', false).text('<?php esc_html_e('Disconnect Google Account', 'hunnt-ai'); ?>');
                                }
                            });
                        });
                    });
                    </script>
                </div>
            <?php endif; ?>
        <?php endif; ?>
    </div>

    <?php if ( ! $is_tab_layout ) : ?>
        </div>
    <?php endif; ?>
    <?php
}
