document.addEventListener('DOMContentLoaded', function() {
    const toggle = document.getElementById('hunnt_ai_toggle');
    const input = document.getElementById('hunnt_ai_api_key');

    // Check if elements exist before adding event listeners
    if (toggle && input) {
        toggle.addEventListener('click', function() {
            if (input.type === 'password') {
                input.type = 'text';
                toggle.innerHTML = `
                    <svg id="hunnt-eye-closed" xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="#ffffff" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M17.94 17.94A10.94 10.94 0 0 1 12 20c-7 0-11-8-11-8a19.66 19.66 0 0 1 5-5"/>
                        <path d="M1 1l22 22"/>
                    </svg>
                `;
            } else {
                input.type = 'password';
                toggle.innerHTML = `
                    <svg id="hunnt-eye-open" xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="#ffffff" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"/>
                        <circle cx="12" cy="12" r="3"/>
                    </svg>
                `;
            }
        });
    }
});

document.addEventListener("DOMContentLoaded", function () {
    const toggleBtn = document.getElementById("hunnt_ai_toggle_posts");
    const postsList = document.getElementById("hunnt_ai_posts_list");

    // Check if elements exist before adding event listeners
    if (toggleBtn && postsList) {
        toggleBtn.addEventListener("click", function () {
            if (postsList.style.display === "none") {
                postsList.style.display = "block";
                this.textContent = hunntAiStrings.hide;
            } else {
                postsList.style.display = "none";
                this.textContent = hunntAiStrings.show;
            }
        });
    }
});

document.addEventListener('DOMContentLoaded', function () {
    const adminConfig = window.hunntAiAdmin || {};

    const feedbackBox = document.getElementById('tech-seo-diagnostics-feedback');
    const feedbackText = feedbackBox ? feedbackBox.querySelector('p') : null;
    const logToggle = document.getElementById('tech-seo-log-toggle');
    const logViewer = document.getElementById('tech-seo-log-viewer');
    const staticToggle = document.getElementById('hunnt-static-cache-toggle');
    const staticStatus = document.getElementById('hunnt-static-cache-status');
    const staticWp = document.getElementById('hunnt-static-cache-wp');
    const staticDropin = document.getElementById('hunnt-static-cache-dropin');
    const staticCount = document.getElementById('hunnt-static-cache-count');
    const staticBytes = document.getElementById('hunnt-static-cache-bytes');
    const staticTtl = document.getElementById('hunnt-static-cache-ttl');
    const staticGenerated = document.getElementById('hunnt-static-cache-generated');
    const staticWarmInput = document.getElementById('hunnt-static-cache-warm-urls');
    const staticWarmBtn = document.getElementById('hunnt-static-cache-warm');
    const staticPurgeBtn = document.getElementById('hunnt-static-cache-purge');
    const staticPermissionBox = document.getElementById('hunnt-static-cache-permission');
    const staticPermissionMessage = document.getElementById('hunnt-static-cache-permission-message');
    const staticPermissionList = document.getElementById('hunnt-static-cache-permission-paths');
    const rebuildBtn = document.getElementById('tech-seo-rebuild-assets');
    const warmupBtn = document.getElementById('tech-seo-warmup-now');
    const lastResult = document.getElementById('tech-seo-last-result');
    const lastRun = document.getElementById('tech-seo-last-run');
    const totalSaved = document.getElementById('tech-seo-total-saved');
    const totalRatio = document.getElementById('tech-seo-total-ratio');
    const warmupStatus = document.getElementById('tech-seo-last-warmup');
    const warmupAt = document.getElementById('tech-seo-last-warmup-at');
    const nextWarmup = document.getElementById('tech-seo-next-warmup');
    const lastFailure = document.getElementById('tech-seo-last-failure');

    const resolvedLocale = adminConfig.locale ? adminConfig.locale.replace(/_/g, '-') : undefined;
    const numberFormatter = new Intl.NumberFormat(resolvedLocale, { maximumFractionDigits: 0 });
    const decimalFormatter = new Intl.NumberFormat(resolvedLocale, { maximumFractionDigits: 2 });
    const percentFormatter = new Intl.NumberFormat(resolvedLocale, { maximumFractionDigits: 2 });
    const dateFormatter = (() => {
        try {
            return new Intl.DateTimeFormat(resolvedLocale, adminConfig.dateTimeOptions || {
                year: 'numeric',
                month: 'short',
                day: '2-digit',
                hour: '2-digit',
                minute: '2-digit'
            });
        } catch (error) {
            return null;
        }
    })();

    const labels = Object.assign({
        showLog: 'Show Optimization Log',
        hideLog: 'Hide Optimization Log',
        enabled: 'Enabled',
        disabled: 'Disabled',
        enableStaticCache: 'Enable Static Cache',
        disableStaticCache: 'Disable Static Cache',
        warmCacheSuccess: 'Cache warmed successfully.',
        warmCacheError: 'Failed to warm cache.',
        purgeCacheSuccess: 'Cache purged successfully.',
        purgeCacheError: 'Failed to purge cache.',
        toggleEnableSuccess: 'Static cache enabled.',
        toggleDisableSuccess: 'Static cache disabled.',
        toggleError: 'Unable to update static cache setting.',
        rebuildSuccess: 'Optimization assets rebuilt.',
        rebuildError: 'Failed to rebuild optimization assets.',
        warmupSuccess: 'Warmup triggered.',
        warmupError: 'Failed to start warmup.',
        genericError: 'Something went wrong.',
        notAvailable: 'n/a',
        success: 'Success',
        failed: 'Failed',
        unknown: 'Unknown',
        loading: 'Working…',
        logEmpty: 'No optimization logs recorded yet.',
        requiresTechSeo: 'Enable Technical SEO to use static cache.',
        cachePermissionError: 'Filesystem permissions prevent static cache from running.',
        cachePermissionFix: 'Hunnt AI cannot create its static cache folders. Please update the filesystem permissions and try again.',
        writableLabel: 'Writable',
        notWritableLabel: 'Not writable'
    }, adminConfig.labels || {});

    function formatBytes(bytes) {
        const size = Number(bytes) || 0;
        if (!size) {
            return '0 B';
        }
        const units = ['B', 'KB', 'MB', 'GB', 'TB'];
        const exponent = Math.min(Math.floor(Math.log(size) / Math.log(1024)), units.length - 1);
        const value = size / Math.pow(1024, exponent);
        return `${decimalFormatter.format(value)} ${units[exponent]}`;
    }

    function humanizeSeconds(seconds) {
        const total = Number(seconds) || 0;
        if (!total) {
            return labels.notAvailable;
        }
        const parts = [];
        const units = [
            { label: labels.days || 'days', value: 86400 },
            { label: labels.hours || 'hours', value: 3600 },
            { label: labels.minutes || 'minutes', value: 60 }
        ];
        let remaining = total;
        for (const unit of units) {
            if (remaining >= unit.value) {
                const quantity = Math.floor(remaining / unit.value);
                remaining -= quantity * unit.value;
                parts.push(`${quantity} ${quantity === 1 ? (unit.labelSingular || unit.label.replace(/s$/, '')) : unit.label}`);
            }
        }
        if (!parts.length) {
            parts.push(`${total} ${labels.seconds || 'seconds'}`);
        }
        return parts.slice(0, 2).join(', ');
    }

    function formatDate(value) {
        if (!value) {
            return labels.notAvailable;
        }
        let dateValue = value;
        if (typeof dateValue === 'string') {
            if (/^\d{4}-\d{2}-\d{2} /.test(dateValue)) {
                dateValue = dateValue.replace(' ', 'T');
            }
            const parsed = Date.parse(dateValue);
            if (!Number.isNaN(parsed)) {
                dateValue = parsed;
            }
        }
        if (typeof dateValue === 'number') {
            if (dateValue < 1e12) {
                dateValue *= 1000;
            }
            dateValue = new Date(dateValue);
        } else {
            dateValue = new Date(dateValue);
        }
        if (Number.isNaN(dateValue.getTime())) {
            return labels.notAvailable;
        }
        if (dateFormatter) {
            return dateFormatter.format(dateValue);
        }
    return dateValue.toLocaleString(resolvedLocale);
    }

    function setFeedback(type, message) {
        if (!feedbackBox || !feedbackText) {
            return;
        }
        feedbackBox.classList.remove('notice-success', 'notice-error', 'notice-warning', 'notice-info');
        switch (type) {
            case 'success':
                feedbackBox.classList.add('notice-success');
                break;
            case 'error':
                feedbackBox.classList.add('notice-error');
                break;
            case 'warning':
                feedbackBox.classList.add('notice-warning');
                break;
            default:
                feedbackBox.classList.add('notice-info');
        }
        feedbackText.textContent = message;
        feedbackBox.style.display = 'block';
    }

    function clearFeedback() {
        if (!feedbackBox) {
            return;
        }
        feedbackBox.style.display = 'none';
        feedbackBox.classList.remove('notice-success', 'notice-error', 'notice-warning', 'notice-info');
        if (feedbackText) {
            feedbackText.textContent = '';
        }
    }

    function setLoading(button, isLoading) {
        if (!button) {
            return;
        }
        if (isLoading) {
            button.dataset.originalText = button.dataset.originalText || button.textContent;
            button.textContent = labels.loading;
            button.disabled = true;
        } else {
            button.disabled = false;
            if (button.dataset.originalText) {
                if (button.textContent === labels.loading) {
                    button.textContent = button.dataset.originalText;
                }
                delete button.dataset.originalText;
            }
        }
    }

    function postAjax(action, payload) {
        if (!adminConfig.ajaxUrl) {
            return Promise.reject(new Error('Missing AJAX URL.'));
        }
        const params = new URLSearchParams();
        params.append('action', action);
        if (adminConfig.nonce) {
            params.append('nonce', adminConfig.nonce);
        }
        if (payload) {
            Object.entries(payload).forEach(([key, value]) => {
                if (Array.isArray(value)) {
                    value.forEach((item) => params.append(key, item));
                } else if (value !== undefined && value !== null) {
                    params.append(key, value);
                }
            });
        }

        return fetch(adminConfig.ajaxUrl, {
            method: 'POST',
            credentials: 'same-origin',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8'
            },
            body: params.toString()
        }).then((response) => {
            if (!response.ok) {
                throw new Error(labels.genericError);
            }
            return response.json();
        });
    }

    function updateStaticCache(summary) {
        if (!summary) {
            return;
        }
        const enabled = !!summary.enabled;
        const techEnabled = summary.technical_seo_enabled !== false;
        const hasWriteAccess = summary.has_write_access !== false;
        const statusEl = staticStatus;
        if (staticStatus) {
            staticStatus.dataset.enabled = enabled && techEnabled ? '1' : '0';
            staticStatus.dataset.techEnabled = techEnabled ? '1' : '0';
            staticStatus.dataset.writeAccess = hasWriteAccess ? '1' : '0';
            staticStatus.classList.remove('hunnt-status-success', 'hunnt-status-neutral', 'hunnt-status-warning');
            if (!techEnabled) {
                staticStatus.textContent = labels.requiresTechSeo;
                staticStatus.classList.add('hunnt-status-warning');
            } else if (!hasWriteAccess) {
                staticStatus.textContent = labels.cachePermissionError;
                staticStatus.classList.add('hunnt-status-warning');
            } else if (enabled) {
                staticStatus.textContent = labels.enabled;
                staticStatus.classList.add('hunnt-status-success');
            } else {
                staticStatus.textContent = labels.disabled;
                staticStatus.classList.add('hunnt-status-neutral');
            }
        }
        if (staticToggle) {
            staticToggle.dataset.enabled = enabled && techEnabled ? '1' : '0';
            staticToggle.dataset.techEnabled = techEnabled ? '1' : '0';
            staticToggle.dataset.writeAccess = hasWriteAccess ? '1' : '0';
            staticToggle.disabled = !techEnabled || !hasWriteAccess;
            if (!techEnabled) {
                staticToggle.textContent = labels.requiresTechSeo;
            } else if (!hasWriteAccess) {
                staticToggle.textContent = labels.cachePermissionFix;
            } else {
                staticToggle.textContent = enabled ? labels.disableStaticCache : labels.enableStaticCache;
            }
        }
        if (staticWp && typeof summary.wp_cache !== 'undefined') {
            staticWp.textContent = summary.wp_cache ? (labels.on || 'On') : (labels.off || 'Off');
        }
        if (staticDropin && typeof summary.dropin_active !== 'undefined') {
            staticDropin.textContent = summary.dropin_active ? (labels.yes || 'Yes') : (labels.no || 'No');
        }
        if (staticWarmBtn) {
            staticWarmBtn.disabled = !techEnabled || !hasWriteAccess;
        }
        if (staticPurgeBtn) {
            staticPurgeBtn.disabled = !techEnabled || !hasWriteAccess;
        }
        if (staticCount) {
            const count = summary.inventory ? Number(summary.inventory.count) || 0 : 0;
            staticCount.textContent = numberFormatter.format(count);
        }
        if (staticBytes) {
            const bytes = summary.inventory ? summary.inventory.bytes : 0;
            staticBytes.textContent = formatBytes(bytes);
        }
        if (staticTtl) {
            staticTtl.textContent = typeof summary.ttl !== 'undefined' ? humanizeSeconds(summary.ttl) : labels.notAvailable;
        }
        if (staticGenerated) {
            staticGenerated.textContent = summary.last_generated ? formatDate(summary.last_generated) : labels.notAvailable;
        }
        if (staticPermissionBox) {
            if (hasWriteAccess) {
                staticPermissionBox.style.display = 'none';
                if (staticPermissionList) {
                    staticPermissionList.innerHTML = '';
                }
            } else {
                staticPermissionBox.style.display = 'block';
                if (staticPermissionMessage) {
                    if (summary.last_error && summary.last_error.message) {
                        staticPermissionMessage.textContent = summary.last_error.message;
                    } else {
                        staticPermissionMessage.textContent = labels.cachePermissionFix;
                    }
                }
                if (staticPermissionList) {
                    staticPermissionList.innerHTML = '';
                    const directories = summary.directories || {};
                    Object.keys(directories).forEach((key) => {
                        const meta = directories[key] || {};
                        const li = document.createElement('li');
                        li.dataset.key = key;
                        const code = document.createElement('code');
                        code.textContent = meta.path || key;
                        const status = document.createElement('span');
                        const writable = !!meta.writable;
                        status.className = `hunnt-static-cache-permission-state ${writable ? 'state-ok' : 'state-bad'}`;
                        status.textContent = ` — ${writable ? labels.writableLabel : labels.notWritableLabel}`;
                        li.appendChild(code);
                        li.appendChild(status);
                        staticPermissionList.appendChild(li);
                    });
                }
            }
        }
    }

    function updateDiagnostics(diagnostics) {
        if (!diagnostics) {
            return;
        }
        const debug = diagnostics.last_debug || {};
        const warm = diagnostics.warmup || {};

        if (lastResult) {
            let resultLabel = labels.unknown;
            if (typeof debug.success !== 'undefined') {
                resultLabel = debug.success ? labels.success : labels.failed;
            }
            lastResult.textContent = resultLabel;
        }
        if (lastRun) {
            lastRun.textContent = debug.timestamp ? formatDate(debug.timestamp) : labels.notAvailable;
        }
        if (totalSaved) {
            totalSaved.textContent = formatBytes(debug.total_bytes_saved);
        }
        if (totalRatio) {
            const ratio = Number(debug.total_reduction_ratio) || 0;
            totalRatio.textContent = `${percentFormatter.format(ratio * 100)}%`;
        }
        if (warmupStatus) {
            if (warm.last_result === 'success') {
                warmupStatus.textContent = labels.success;
            } else if (warm.last_result === 'failed') {
                warmupStatus.textContent = labels.failed;
            } else {
                warmupStatus.textContent = labels.unknown;
            }
        }
        if (warmupAt) {
            warmupAt.textContent = warm.last_timestamp ? formatDate(warm.last_timestamp) : labels.notAvailable;
        }
        if (nextWarmup) {
            nextWarmup.textContent = warm.next_timestamp ? formatDate(warm.next_timestamp) : labels.notAvailable;
        }
        if (lastFailure) {
            const failureSpan = lastFailure.querySelector('span');
            if (debug.failure_reason) {
                lastFailure.style.display = '';
                if (failureSpan) {
                    failureSpan.textContent = debug.failure_reason;
                }
            } else {
                lastFailure.style.display = 'none';
                if (failureSpan) {
                    failureSpan.textContent = '';
                }
            }
        }
        if (logViewer) {
            if (diagnostics.logs && diagnostics.logs.minify) {
                logViewer.textContent = JSON.stringify(diagnostics.logs.minify, null, 2);
            } else {
                logViewer.textContent = labels.logEmpty;
            }
        }
        if (staticBytes && diagnostics.summary && diagnostics.summary.inventory) {
            staticBytes.textContent = formatBytes(diagnostics.summary.inventory.bytes);
        }
    }

    if (logToggle && logViewer) {
        logToggle.addEventListener('click', function () {
            const isOpen = this.getAttribute('data-open') === '1';
            if (isOpen) {
                logViewer.style.display = 'none';
                this.setAttribute('data-open', '0');
                this.textContent = labels.showLog;
            } else {
                logViewer.style.display = 'block';
                this.setAttribute('data-open', '1');
                this.textContent = labels.hideLog;
            }
        });
    }

    if (staticToggle && staticStatus) {
        staticToggle.addEventListener('click', function () {
            if (this.getAttribute('data-tech-enabled') === '0') {
                setFeedback('warning', labels.requiresTechSeo);
                return;
            }
            if (this.getAttribute('data-write-access') === '0') {
                setFeedback('warning', labels.cachePermissionError);
                return;
            }
            const enable = this.getAttribute('data-enabled') !== '1';
            clearFeedback();
            setLoading(staticToggle, true);
            postAjax('hunnt_ai_toggle_static_cache', { enable: enable ? 1 : 0 })
                .then((response) => {
                    if (!response || !response.success) {
                        throw new Error(response && response.data ? response.data : labels.toggleError);
                    }
                    if (response.data && response.data.summary) {
                        updateStaticCache(response.data.summary);
                    }
                    setFeedback('success', enable ? labels.toggleEnableSuccess : labels.toggleDisableSuccess);
                })
                .catch((error) => {
                    console.error(error);
                    setFeedback('error', error.message || labels.toggleError);
                })
                .finally(() => {
                    setLoading(staticToggle, false);
                    staticToggle.disabled = staticToggle.getAttribute('data-tech-enabled') === '0'
                        || staticToggle.getAttribute('data-write-access') === '0';
                });
        });
    }

    if (staticWarmBtn) {
        staticWarmBtn.addEventListener('click', function () {
            if (staticToggle && staticToggle.getAttribute('data-tech-enabled') === '0') {
                setFeedback('warning', labels.requiresTechSeo);
                return;
            }
            if (staticToggle && staticToggle.getAttribute('data-write-access') === '0') {
                setFeedback('warning', labels.cachePermissionError);
                return;
            }
            clearFeedback();
            const urls = [];
            if (staticWarmInput && staticWarmInput.value.trim()) {
                staticWarmInput.value.split(/\r?\n/).forEach((line) => {
                    const trimmed = line.trim();
                    if (trimmed) {
                        urls.push(trimmed);
                    }
                });
            }
            setLoading(staticWarmBtn, true);
            postAjax('hunnt_ai_static_cache_warm', urls.length ? { 'urls[]': urls } : {})
                .then((response) => {
                    if (!response || !response.success) {
                        throw new Error(response && response.data ? response.data : labels.warmCacheError);
                    }
                    if (response.data && response.data.summary) {
                        updateStaticCache(response.data.summary);
                    }
                    setFeedback('success', labels.warmCacheSuccess);
                })
                .catch((error) => {
                    console.error(error);
                    setFeedback('error', error.message || labels.warmCacheError);
                })
                .finally(() => {
                    setLoading(staticWarmBtn, false);
                    if (staticToggle) {
                        staticWarmBtn.disabled = staticToggle.getAttribute('data-tech-enabled') === '0'
                            || staticToggle.getAttribute('data-write-access') === '0';
                    }
                });
        });
    }

    if (staticPurgeBtn) {
        staticPurgeBtn.addEventListener('click', function () {
            if (staticToggle && staticToggle.getAttribute('data-tech-enabled') === '0') {
                setFeedback('warning', labels.requiresTechSeo);
                return;
            }
            if (staticToggle && staticToggle.getAttribute('data-write-access') === '0') {
                setFeedback('warning', labels.cachePermissionError);
                return;
            }
            clearFeedback();
            setLoading(staticPurgeBtn, true);
            postAjax('hunnt_ai_static_cache_purge')
                .then((response) => {
                    if (!response || !response.success) {
                        throw new Error(response && response.data ? response.data : labels.purgeCacheError);
                    }
                    if (response.data && response.data.summary) {
                        updateStaticCache(response.data.summary);
                    }
                    setFeedback('success', labels.purgeCacheSuccess);
                })
                .catch((error) => {
                    console.error(error);
                    setFeedback('error', error.message || labels.purgeCacheError);
                })
                .finally(() => {
                    setLoading(staticPurgeBtn, false);
                    if (staticToggle) {
                        staticPurgeBtn.disabled = staticToggle.getAttribute('data-tech-enabled') === '0'
                            || staticToggle.getAttribute('data-write-access') === '0';
                    }
                });
        });
    }

    if (rebuildBtn) {
        rebuildBtn.addEventListener('click', function () {
            clearFeedback();
            setLoading(rebuildBtn, true);
            postAjax('hunnt_ai_rebuild_technical_seo_assets')
                .then((response) => {
                    if (!response || !response.success) {
                        throw new Error(response && response.data ? response.data : labels.rebuildError);
                    }
                    if (response.data && response.data.diagnostics) {
                        updateDiagnostics(response.data.diagnostics);
                    }
                    if (response.data && response.data.summary) {
                        updateStaticCache(response.data.summary);
                    }
                    setFeedback('success', labels.rebuildSuccess);
                })
                .catch((error) => {
                    console.error(error);
                    setFeedback('error', error.message || labels.rebuildError);
                })
                .finally(() => {
                    setLoading(rebuildBtn, false);
                });
        });
    }

    if (warmupBtn) {
        warmupBtn.addEventListener('click', function () {
            clearFeedback();
            setLoading(warmupBtn, true);
            postAjax('hunnt_ai_trigger_technical_seo_warmup')
                .then((response) => {
                    if (!response || !response.success) {
                        throw new Error(response && response.data ? response.data : labels.warmupError);
                    }
                    if (response.data && response.data.diagnostics) {
                        updateDiagnostics(response.data.diagnostics);
                    }
                    setFeedback('success', labels.warmupSuccess);
                })
                .catch((error) => {
                    console.error(error);
                    setFeedback('error', error.message || labels.warmupError);
                })
                .finally(() => {
                    setLoading(warmupBtn, false);
                });
        });
    }

    window.hunntAiUpdateStaticCache = updateStaticCache;
});
