/**
 * Google OAuth Integration JavaScript
 * Provides a centralized handler for initiating and processing Google OAuth across the plugin.
 * @package HunntAI
 */

(function($) {
    'use strict';

    var config = window.hunntAIGoogleOAuth || {};
    var ajaxEndpoint = config.ajaxurl || (typeof ajaxurl !== 'undefined' ? ajaxurl : '');
    var labels = config.labels || {};
    var defaultConnectLabel = labels.connect || 'Connect Google Account';
    var defaultLoadingLabel = labels.connecting || 'Connecting...';
    var defaultFailedLabel = labels.failed || 'Failed to initiate OAuth. Please try again.';

    function renderNotice(type, message) {
        if (!message) {
            return;
        }

        var targetSelector = config.noticeTarget;
        if (targetSelector === 'none') {
            return;
        }

        var $target = targetSelector ? $(targetSelector) : $();
        if (!$target.length) {
            $target = $('.wrap').first();
        }
        if (!$target.length) {
            return;
        }

        var renderer = config.noticeRenderer || 'admin';
        if (renderer === 'inline' || $target.hasClass('google-auth-status')) {
            var inlineClass = type === 'success' ? 'hunnt-inline-success' : 'hunnt-inline-error';
            $target.html('<div class="' + inlineClass + '">' + message + '</div>');
        } else {
            var noticeClass = type === 'success' ? 'notice-success' : 'notice-error';
            var noticeHtml = '<div class="notice ' + noticeClass + ' is-dismissible"><p>' + message + '</p></div>';
            $target.prepend(noticeHtml);
        }
    }

    function getUrlParam(rawUrl, name) {
        var regex = new RegExp('[?&]' + name + '=([^&#]*)');
        var results = regex.exec(rawUrl);
        return results === null ? null : decodeURIComponent(results[1].replace(/\+/g, ' '));
    }

    function getCurrentContext() {
        var params = new URLSearchParams(window.location.search);
        return {
            page: params.get('page') || 'hunnt-ai-google-oauth',
            step: params.get('step') || ''
        };
    }

    function getAjaxUrl() {
        if (ajaxEndpoint) {
            return ajaxEndpoint;
        }
        if (typeof window.ajaxurl !== 'undefined') {
            return window.ajaxurl;
        }
        return '';
    }

    function removeQueryParams(rawUrl, params) {
        try {
            var urlObj = new URL(rawUrl, window.location.origin);
            params.forEach(function(param) {
                urlObj.searchParams.delete(param);
            });
            var search = urlObj.searchParams.toString();
            return urlObj.origin + urlObj.pathname + (search ? '?' + search : '') + urlObj.hash;
        } catch (err) {
            return rawUrl;
        }
    }

    $(document).on('click', '.hunnt-ai-connect-google', function(e) {
        e.preventDefault();

        if (!config || !config.nonce) {
            console.error('Hunnt AI Google OAuth: missing configuration.');
            return;
        }

        var $btn = $(this);
        var originalLabel = $btn.data('original-label');
        if (!originalLabel) {
            originalLabel = $.trim($btn.text()) || defaultConnectLabel;
            $btn.data('original-label', originalLabel);
        }

        var loadingLabel = $btn.data('loading-label') || defaultLoadingLabel;
        $btn.prop('disabled', true).text(loadingLabel);

        var context = getCurrentContext();

        $(document).trigger('hunnt-ai-google-connect-start', [{ button: $btn, context: context }]);

        var ajaxUrl = getAjaxUrl();
        if (!ajaxUrl) {
            console.error('Hunnt AI Google OAuth: AJAX endpoint not available.');
            $btn.prop('disabled', false).text(originalLabel);
            return;
        }

        $.post(ajaxUrl, {
            action: 'hunnt_ai_initiate_google_oauth',
            nonce: config.nonce,
            current_page: context.page,
            current_step: context.step
        }).done(function(response) {
            if (response && response.success && response.data && response.data.oauth_url) {
                $(document).trigger('hunnt-ai-google-connect-redirect', [{ button: $btn, context: context, response: response }]);
                window.location.href = response.data.oauth_url;
            } else {
                var message = response && response.data && response.data.message ? response.data.message : defaultFailedLabel;
                if (config.useAlerts !== false) {
                    alert(message);
                }
                $btn.prop('disabled', false).text(originalLabel);
                $(document).trigger('hunnt-ai-google-connect-error', [{ button: $btn, context: context, response: response, message: message }]);
            }
        }).fail(function(xhr) {
            if (config.useAlerts !== false) {
                alert(defaultFailedLabel);
            }
            $btn.prop('disabled', false).text(originalLabel);
            $(document).trigger('hunnt-ai-google-connect-error', [{ button: $btn, context: context, xhr: xhr, message: defaultFailedLabel }]);
        });
    });

    var currentUrl = window.location.href;
    var hasGoogleCallback = currentUrl.indexOf('google_callback=1') !== -1;

    if (hasGoogleCallback && config && config.nonce) {
        var successParam = getUrlParam(currentUrl, 'success');
        var messageParam = getUrlParam(currentUrl, 'message') || '';

        var callbackAjaxUrl = getAjaxUrl();
        if (!callbackAjaxUrl) {
            console.error('Hunnt AI Google OAuth: AJAX endpoint not available for callback processing.');
            return;
        }

        $.post(callbackAjaxUrl, {
            action: 'hunnt_ai_handle_google_callback',
            nonce: config.nonce,
            success: successParam,
            message: messageParam
        }).done(function(response) {
            var success = !!(response && response.success);
            var data = response && response.data ? response.data : {};
            var message = data.message || messageParam;

            renderNotice(success ? 'success' : 'error', message);

            $(document).trigger('hunnt-ai-google-callback', [{
                success: success,
                response: response,
                data: data,
                message: message,
                originalMessage: messageParam,
                successParam: successParam,
                url: currentUrl
            }]);

            if (success) {
                var followUpDelay = typeof config.refreshDelay === 'number' ? config.refreshDelay : 200;
                var redirectUrl = data.redirect_url || config.successRedirect || null;

                if (redirectUrl) {
                    setTimeout(function() {
                        window.location.href = redirectUrl;
                    }, followUpDelay);
                } else if (config.refreshOnSuccess !== false) {
                    var cleaned = removeQueryParams(currentUrl, ['google_callback', 'success', 'message']);
                    setTimeout(function() {
                        window.location.href = cleaned;
                    }, followUpDelay);
                }
            }
        }).fail(function(xhr) {
            var errorMessage = defaultFailedLabel;
            renderNotice('error', errorMessage);
            $(document).trigger('hunnt-ai-google-callback', [{
                success: false,
                response: null,
                data: {},
                message: errorMessage,
                originalMessage: messageParam,
                successParam: successParam,
                url: currentUrl,
                xhr: xhr
            }]);
        });
    }

})(jQuery);
