jQuery(document).ready(function ($) {
    // ✅ Load dropdown values + store languages
    $.post(ApiArticlesAdmin.ajax_url, {
        action: 'get_article_counts',
        nonce: ApiArticlesAdmin.nonce
    }, function (response) {
        let dropdown = $('#articles_count');
        dropdown.empty();

        if (response.success) {
            // Add placeholder
            dropdown.append('<option value="">' + ApiArticlesAdmin.select_text + '</option>');

            // Dropdown options
            response.available_options.forEach(num => {
                dropdown.append('<option value="' + num + '">' + num + '</option>');
            });
            // Save available languages & remaining globally
            window.availableLanguages = response.available_languages || [];
            window.remainingArticles = response.remaining || 0;
        } else {
            console.log('Error:', response.message);
        }
    }).fail(function(xhr) {
        let msg = "Unexpected error occurred.";

        // Try to extract a message from JSON if possible
        try {
            const json = JSON.parse(xhr.responseText);
            msg = json.data?.message || json.message || msg;
        } catch (e) {
            // fall back to the raw response text if not JSON
            if (xhr.responseText) {
                msg = xhr.responseText;
            }
        }

        // Show message to the user
        alert(msg);

        // Debug (optional)
        console.error("HTTP " + xhr.status + ":", xhr.responseText);
    });

// ✅ Generate inputs dynamically depending on available_languages
    $('#articles_count').on('change', function () {
        let count = parseInt($(this).val()); // selected max articles
        let container = $('#articles_inputs');
        container.empty();

        if (isNaN(count) || count <= 0) return;

        if (!window.availableLanguages || window.availableLanguages.length === 0) {
            container.append('<p class="note">⚠ No available languages.</p>');
            return;
        }

        let html = '<div class="article-row">';
        window.availableLanguages.forEach(lang => {
            let label = (lang === 'ar') ? 'Arabic' : (lang === 'en') ? 'English' : lang.toUpperCase();

            html += `
            <label>${label}:</label>
            <input type="number" 
                   name="articles[${lang}]" 
                   placeholder="Number of ${label} articles" 
                   min="0" 
                   max="${count}" />
        `;
        });
        html += '</div>';
        html += `<p class="note">Max number of articles allowed: ${count} (Remaining: ${window.remainingArticles})</p>`;

        container.append(html);
    });




    // ✅ Save articles
    // ✅ Validate before saving
    $('#save_articles').on('click', function () {
        let count = parseInt($('#articles_count').val()); // selected from dropdown
        if (isNaN(count) || count <= 0) {
            alert('❌ Please select number of articles first.');
            return;
        }

        let total = 0;
        let articles = {};
        let max_articles = $(".content-generate-article select[name=max_articles]").val();

        $('#articles_inputs input[type="number"]').each(function () {
            let lang = $(this).attr('name').match(/\[(.*?)\]/)[1]; // extract language key
            let val = parseInt($(this).val()) || 0;
            articles[lang] = val;
            total += val;
        });

        // ❌ Validation: must equal dropdown count
        if (total !== count) {
            alert(`❌ The total of all language fields must equal ${count}. Currently: ${total}`);
            return;
        }

        // ✅ Send to server
        $.post(ApiArticlesAdmin.ajax_url, {
            action: 'save_generate_articles',
            nonce: ApiArticlesAdmin.nonce,
            articles: articles,
            total: count,
            max_articles: max_articles
        }, function (response) {
            if (response.success) {
                alert('✅ Saved!');
            } else {
                // Handle structured error from PHP/API
                if (typeof response.data === 'object') {
                    let msg = response.data.message || 'Unknown error';
                    let details = '';

                    if (response.data.errors) {
                        // Join all error messages
                        for (let key in response.data.errors) {
                            if (response.data.errors.hasOwnProperty(key)) {
                                details += `\n- ${response.data.errors[key]}`;
                            }
                        }
                    }

                    alert(`❌ ${msg}${details}`);
                } else {
                    alert('❌ Error: ' + response.data);
                }
            }
        }).fail(function (xhr) {
            // Catch real HTTP errors (e.g., 401/422 returned by PHP or API)
            let res = {};
            try {
                res = JSON.parse(xhr.responseText);
            } catch (e) {}

            if (res && res.data) {
                let msg = res.data.message || 'Request failed';
                let details = '';

                if (res.data.errors) {
                    for (let key in res.data.errors) {
                        if (res.data.errors.hasOwnProperty(key)) {
                            details += `\n- ${res.data.errors[key]}`;
                        }
                    }
                }

                alert(`❌ ${msg}${details}`);
            } else {
                alert('❌ Request failed: ' + xhr.status + ' ' + xhr.statusText);
            }
        });
    });


});
